package com.franciaflex.faxtomail.ui.swing.content.reply;

/*
 * #%L
 * FaxToMail :: UI
 * $Id: ReplyFormUIHandler.java 185 2014-06-11 05:32:51Z kmorin $
 * $HeadURL: http://svn.codelutin.com/faxtomail/tags/faxtomail-0.2/faxtomail-ui-swing/src/main/java/com/franciaflex/faxtomail/ui/swing/content/reply/ReplyFormUIHandler.java $
 * %%
 * Copyright (C) 2014 Franciaflex, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import com.franciaflex.faxtomail.persistence.entities.AttachmentFileImpl;
import org.apache.commons.io.FileUtils;
import org.nuiton.util.FileUtil;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import java.awt.Component;
import java.awt.Desktop;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.ComboBoxEditor;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JPanel;
import javax.swing.JTextField;

import jaxx.runtime.JAXXUtil;
import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailUIHandler;
import com.franciaflex.faxtomail.ui.swing.util.Cancelable;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.google.common.collect.Lists;

/**
 * @author Kevin Morin (Code Lutin)
 * @since x.x
 */
public class ReplyFormUIHandler extends AbstractFaxToMailUIHandler<ReplyFormUIModel, ReplyFormUI> implements Cancelable {

    private static final Log log = LogFactory.getLog(ReplyFormUIHandler.class);

    @Override
    public void beforeInit(ReplyFormUI ui) {

        super.beforeInit(ui);

        ReplyFormUIModel model = new ReplyFormUIModel();
        long maxSize = getContext().getFaxToMailConfiguration().getEmailMaxSize();
        model.setMaxAttachmentLength(maxSize);

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(ReplyFormUI replyFormUI) {
        initUI(ui);

        JEditorPane editor = ui.getMessage();
        editor.setCaretPosition(0);

        ReplyFormUIModel model = getModel();
        model.addPropertyChangeListener(ReplyFormUIModel.PROPERTY_ORIGINAL_DEMAND, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                DemandeUIModel demand = (DemandeUIModel) evt.getNewValue();
                ReplyFormUIModel model = (ReplyFormUIModel) evt.getSource();

                List<DemandeUIModel> models = new ArrayList<DemandeUIModel>();
                models.add(demand);
                models.addAll(demand.getGroupedDemandes());

                for (DemandeUIModel demandeUIModel : models) {
                    for (Attachment attachment : demandeUIModel.getAttachment()) {
                        AttachmentFile originalFile = attachment.getOriginalFile();
                        model.addAvailableAttachment(originalFile);
                        AttachmentFile file = attachment.getEditedFile();
                        if (file != null) {
                            model.addAvailableAttachment(file);
                        }
                    }
                }

                JComboBox<AttachmentFile> addAttachmentFile = ui.getAddAttachmentFile();
                addAttachmentFile.setModel(newComboModel(model.getAvailableAttachments().toArray()));
                addAttachmentFile.setSelectedItem(null);

                model.setFrom(demand.getRecipient());

                model.setSubject(t("faxtomail.reply.subject", demand.getSubject()));

                String plainContent = JAXXUtil.getStringValue(demand.getPlainContent());
                String quotedReply = t("faxtomail.reply.message",
                                       decorate(demand.getReceptionDate()),
                                       demand.getSender(),
                                       plainContent.replaceAll("\n", "\n> "));
                model.setMessage(quotedReply);

                MailFolder folder = demand.getMailFolder();
                List<String> folderReplyAdresses = new ArrayList<String>();
                List<String> folderReplyDomains = new ArrayList<String>();
                while (folder != null) {
                    folderReplyAdresses.addAll(folder.getReplyAddresses());
                    folderReplyDomains.addAll(folder.getReplyDomains());
                    folder = folder.getParent();
                }
                model.setSenderAllowedDomains(folderReplyDomains);

                List<String> replyToAddresses = Lists.newArrayList(demand.getRecipient());
                if (folderReplyAdresses != null) {
                    replyToAddresses.addAll(folderReplyAdresses);
                }

                ui.getFromComboBox().setModel(newComboModel(replyToAddresses.toArray()));
            }
        });

        model.addPropertyChangeListener(ReplyFormUIModel.PROPERTY_ATTACHMENT, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                updateAttachmentPanel((Collection<AttachmentFile>) evt.getNewValue());
            }
        });

        model.addPropertyChangeListener(ReplyFormUIModel.PROPERTY_READONLY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                updateAttachmentPanel(getModel().getAttachments());
            }
        });

        JComboBox addAttachmentFile = ui.getAddAttachmentFile();
        addAttachmentFile.setRenderer(new DecoratorListCellRenderer(getDecorator(AttachmentFile.class, null)));
        addAttachmentFile.setEditor(new FileComboBoxEditor());

        getValidator().addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                ReplyFormUIModel model = getModel();
                if (log.isDebugEnabled()) {
                    log.debug("Model [" + model +
                                      "] pass to valid state [" +
                                      evt.getNewValue() + "]");
                }
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }

    @Override
    public void cancel() {
        closeFrame();
    }

    @Override
    public void onCloseUI() {
    }

    @Override
    public SwingValidator<ReplyFormUIModel> getValidator() {
        return getUI().getValidator();
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getMessage();
    }

    public void removeAttachment(AttachmentFile attachment) {
        ReplyFormUIModel model = getModel();

        model.removeAttachment(attachment);

        model.addAvailableAttachment(attachment);
        ui.getAddAttachmentFile().addItem(attachment);
    }

    public void addAttachment() {
        JComboBox addAttachmentFile = ui.getAddAttachmentFile();
        AttachmentFile attachmentFile = null;
        Object selectedItem = addAttachmentFile.getSelectedItem();
        if (AttachmentFile.class.isAssignableFrom(selectedItem.getClass())) {
            attachmentFile = (AttachmentFile) selectedItem;

        } else if (File.class.isAssignableFrom(selectedItem.getClass())) {
            File file = (File) selectedItem;
            attachmentFile = new AttachmentFileImpl();
            attachmentFile.setFilename(file.getName());
            try {
                attachmentFile.setContent(FileUtils.readFileToByteArray(file));

            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Error while converting the file " + file.getName() + " into a byte[]", e);
                }
            }
        }
        if (attachmentFile != null) {
            ReplyFormUIModel model = getModel();
            model.addAttachment(attachmentFile);

            model.removeAvailableAttachment(attachmentFile);
            ui.getAddAttachmentFile().removeItem(attachmentFile);

            addAttachmentFile.setSelectedItem(null);
        }
    }

    protected void updateAttachmentPanel(Collection<AttachmentFile> attachments) {
        JPanel attachmentsPanel = ui.getAttachmentsPanel();
        attachmentsPanel.removeAll();
        for (AttachmentFile attachment : attachments) {
            AttachmentItem item = new AttachmentItem();
            item.setHandler(this);
            item.setAttachmentFile(attachment);
            attachmentsPanel.add(item);
        }
        attachmentsPanel.updateUI();
    }

    public void openLocation() {
        // use last selected file
        ReplyFormUIModel model = getModel();

        File startFile = model.getLastVisitedDirectory();
        JFileChooser fc = new JFileChooser(startFile);

//        fc.setDialogTitle(view.getTitle());

        // used to enable file selection
        fc.setFileSelectionMode(JFileChooser.FILES_ONLY);

        int returnVal = fc.showOpenDialog(ui);
        model.setLastVisitedDirectory(fc.getCurrentDirectory());

        if (returnVal == JFileChooser.APPROVE_OPTION) {

            // get selected to display in ui
            File file = fc.getSelectedFile();

            ui.getAddAttachmentFile().setSelectedItem(file);
        }
    }

    public void openAttachment(AttachmentFile attachment) {

        File file = attachment.getFile();

        Desktop desktop = FaxToMailUIUtil.getDesktopForOpen();
        try {
            desktop.open(file);

        } catch (IOException e) {
            getContext().getErrorHelper().showErrorDialog(t("swing.error.cannot.open.file"));
        }
    }

    protected class FileComboBoxEditor implements ComboBoxEditor {

        protected AttachmentFile oldValue;

        protected JTextField editorComponent = new JTextField();

        @Override
        public Component getEditorComponent() {
            return editorComponent;
        }

        @Override
        public void setItem(Object anObject) {
            String text;
            AttachmentFile attachmentFile = null;
            if (anObject != null) {
                if (AttachmentFile.class.isAssignableFrom(anObject.getClass())) {
                    attachmentFile = (AttachmentFile) anObject;

                } else if (File.class.isAssignableFrom(anObject.getClass())) {
                    File file = (File) anObject;
                    attachmentFile = new AttachmentFileImpl();
                    attachmentFile.setFilename(file.getName());
                }
            }

            if (attachmentFile != null)  {
                text = decorate(attachmentFile);
                oldValue = attachmentFile;

            } else {
                text = "";
            }
            if (! text.equals(editorComponent.getText())) {
                editorComponent.setText(text);
            }
        }

        @Override
        public Object getItem() {
            Object newValue = editorComponent.getText();

            if (oldValue != null)  {
                // The original value is not a string. Should return the value in it's
                // original type.
                if (newValue.equals(decorate(oldValue))) {
                    return oldValue;

                } else {
                    // Must take the value from the editor and get the value and cast it to the new type.
                    Class<?> cls = oldValue.getClass();
                    try {
                        Method method = cls.getMethod("valueOf", new Class[]{String.class});
                        newValue = method.invoke(oldValue, new Object[] { editorComponent.getText()});
                    } catch (Exception ex) {
                        // Fail silently and return the newValue (a String object)
                    }
                }
            }
            return newValue;
        }

        @Override
        public void selectAll() {
            editorComponent.selectAll();
            editorComponent.requestFocus();
        }

        @Override
        public void addActionListener(ActionListener l) {
            editorComponent.addActionListener(l);
        }

        @Override
        public void removeActionListener(ActionListener l) {
            editorComponent.removeActionListener(l);
        }
    }
}
