package com.franciaflex.faxtomail.ui.swing.content;

/*
 * #%L
 * FaxToMail :: UI
 * %%
 * Copyright (C) 2014 Franciaflex
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.ui.swing.FaxToMailScreen;
import com.franciaflex.faxtomail.ui.swing.FaxToMailUIContext;
import com.franciaflex.faxtomail.ui.swing.RunFaxToMail;
import com.franciaflex.faxtomail.ui.swing.content.config.FaxToMailConfigUI;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeListUI;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUI;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandeUIModel;
import com.franciaflex.faxtomail.ui.swing.content.demande.DemandesUI;
import com.franciaflex.faxtomail.ui.swing.content.search.SearchUI;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailUIHandler;
import com.franciaflex.faxtomail.ui.swing.util.CloseableUI;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUI;
import com.franciaflex.faxtomail.ui.swing.util.RemoveablePropertyChangeListener;
import com.google.common.base.Preconditions;
import jaxx.runtime.JAXXBinding;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.JAXXWidgetUtil;
import jaxx.runtime.swing.TabInfo;
import jaxx.runtime.swing.TabInfoPropertyChangeListener;
import jaxx.runtime.swing.session.SwingSession;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.jaxx.application.swing.action.ApplicationActionUI;

import javax.swing.*;
import java.awt.Cursor;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeListenerProxy;
import java.io.IOException;
import java.util.Locale;

import static org.nuiton.i18n.I18n.t;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class MainUIHandler extends AbstractFaxToMailUIHandler<FaxToMailUIContext, MainUI> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(MainUIHandler.class);

    protected JComponent currentBody;

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInit(MainUI ui) {
        super.beforeInit(ui);
        FaxToMailUIContext context = getContext();
        ui.setContextValue(context);
        context.setMainUI(ui);
        context.setActionUI(new ApplicationActionUI(ui, context));
        context.addPropertyChangeListener(new RemoveablePropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String propertyName = evt.getPropertyName();
                if (propertyName.equals(FaxToMailUIContext.PROPERTY_SCREEN)) {
                    setScreen((FaxToMailScreen) evt.getNewValue());
                }
            }
        });
        ui.setContextValue(ui, MainUI.class.getName());

        // ecoute des changements de l'état busy
        context.addPropertyChangeListener(FaxToMailUIContext.PROPERTY_BUSY, new RemoveablePropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Boolean newvalue = (Boolean) evt.getNewValue();
                updateBusyState(newvalue != null && newvalue);
            }
        });

        // ecoute des changements de l'état busy
        context.addPropertyChangeListener(FaxToMailUIContext.PROPERTY_HIDE_BODY, new RemoveablePropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Boolean newvalue = (Boolean) evt.getNewValue();
                if (getUI() != null && getUI().getBody() != null) {
                    getUI().getBody().setVisible(newvalue != null && newvalue);
                }
            }
        });
    }

    protected void updateBusyState(boolean busy) {
        if (busy) {
            // ui bloquee
            if (log.isDebugEnabled()) {
                log.debug("block ui in busy mode");
            }
            ui.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        } else {
            // ui debloquee
            if (log.isDebugEnabled()) {
                log.debug("unblock ui in none busy mode");
            }
            ui.setCursor(Cursor.getDefaultCursor());
        }
    }

    @Override
    public void afterInit(MainUI ui) {

        initUI(ui);

        // installation layer de blocage en mode busy
//        SwingUtil.setLayerUI(ui.getBody(), ui.getBusyBlockLayerUI());
//
        //FIXME-TC See why this binding is not setted ?
//        ui.applyDataBinding(MainUI.BINDING_MENU_CHANGE_LOCALE_FR_ENABLED);
//        ui.applyDataBinding(MainUI.BINDING_MENU_CHANGE_LOCALE_UK_ENABLED);

        // Init SwingSession
        SwingSession swingSession = getContext().getSwingSession();
        swingSession.add(ui);
        try {
            swingSession.save();

        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("error while saving the swing session", e);
            }
        }

//        SwingUtil.getLayer(ui.getBody()).setUI(new BetaLayerUI());

        changeTitle();

        JToolBar bar = ui.getBottomBar();
        ui.getStatus().addWidget(bar, 0);

        getContext().setScreen(FaxToMailScreen.LIST);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return currentBody;
    }

    @Override
    public void onCloseUI() {

        FaxToMailUIContext context = getContext();

        // remove any screen
        context.setScreen(null);
        context.removeMessageNotifier(this);

        // clean context

        PropertyChangeListener[] propertyChangeListeners =
                context.getPropertyChangeListeners();
        for (PropertyChangeListener listener : propertyChangeListeners) {
            if (listener instanceof PropertyChangeListenerProxy) {
                PropertyChangeListenerProxy proxy = (PropertyChangeListenerProxy) listener;
                listener = proxy.getListener();
            }
            if (listener instanceof RemoveablePropertyChangeListener) {
                if (log.isDebugEnabled()) {
                    log.debug("Remove listener: " + listener);
                }
                context.removePropertyChangeListener(listener);
            }
        }

        if (ui != null) {

            // clean ui

            JAXXBinding[] bindings = ui.getDataBindings();
            for (JAXXBinding binding : bindings) {
                SwingUtil.removeDataBinding(ui, binding.getId());
            }
            ui.setVisible(false);
            ui.dispose();
        }
    }

    @Override
    public SwingValidator<FaxToMailUIContext> getValidator() {
        return null;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void reloadUI() {

        //close ui
        onCloseUI();

        // restart ui
        RunFaxToMail.startFaxToMail(getContext(), false);
    }

    public boolean acceptLocale(Locale l, String expected) {
        return l != null && l.toString().equals(expected);
    }

    public void changeLocale(Locale locale) {

//        // change locale (and save configuration)
        getModel().setLocale(locale);
//
//        // change i18n locale
        I18n.setDefaultLocale(getConfig().getI18nLocale());
//
//        // reload decorator service (TODO Check if this is necessary)
//        getContext().reloadDecoratorService();
//
//        // close reload
        reloadUI();
    }

//    public void gotoSite() {
//        FaxToMailApplicationConfig config = context.getConfig();
//
//        URL siteURL = config.getSiteUrl();
//
//        if (log.isDebugEnabled()) {
//            log.debug("goto " + siteURL);
//        }
//        TuttiUIUtil.openLink(siteURL);
//    }

//    public void showHelp() {
//        getModel().showHelp(ui, ui.getBroker(), null);
//    }

    @Override
    public final void showInformationMessage(String message) {
        ui.getStatus().setStatus("<html><body>" + message + "</body></html>");
    }

    public void registerValidator(SwingValidator validator) {
        ui.getValidatorMessageWidget().registerValidator(validator);
    }

    @Override
    public void clearValidators() {
        ui.getValidatorMessageWidget().clearValidators();
    }

    public boolean quitCurrentScreen() {

        boolean canClose;
        if (getContext().getScreen() == null || currentBody == null) {

            // no screen, surely can quit
            canClose = true;
            if (log.isWarnEnabled()) {
                log.warn("==================================================");
                log.warn("No screen, Should then skipCheckCurrent in action.");
                log.warn("==================================================");
            }
        } else {
            FaxToMailUI<?, ?> body = (FaxToMailUI<?, ?>) currentBody;
            Preconditions.checkNotNull(currentBody);
            AbstractFaxToMailUIHandler<?, ?> handler = body.getHandler();
            if (handler instanceof CloseableUI) {

                // try to quit UI
                canClose = ((CloseableUI) handler).quitUI();
            } else {

                // can always close ui
                canClose = true;
            }
        }
        return canClose;
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void setScreen(FaxToMailScreen screen) {

        // close current body (if any)
        if (currentBody != null) {
            FaxToMailUI<?, ?> body = (FaxToMailUI<?, ?>) currentBody;
            body.getHandler().onCloseUI();

            try {
                getContext().getSwingSession().save();

            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("error while saving the swing session", e);
                }
            }

            ui.getBody().remove(currentBody);

            currentBody = null;
        }

        if (screen != null) {

            // load new body

            JComponent screenUI;
            JToolBar rightDecoration = null;
            String screenTitle;

            Icon icon;
            switch (screen) {
                case CONFIG:
                    screenUI = new FaxToMailConfigUI(ui);
                    screenTitle = t("faxtomail.config.title");
                    icon = ui.getMenuFileConfiguration().getIcon();
                    break;

                case SEARCH:
                    screenUI = new SearchUI(ui);
                    screenTitle = t("faxtomail.search.title");
                    icon = ui.getMenuFileRecherche().getIcon();
                    break;

                case DEMANDE:
                    screenUI = new DemandesUI(ui);
                    screenTitle = t("faxtomail.demand.title");
                    icon = JAXXWidgetUtil.createActionIcon("email-group");
                    rightDecoration = ((DemandesUI) screenUI).getTopToolBar();
                    break;

                default:
                case LIST:
                    screenUI = new DemandeListUI(ui);
                    screenTitle = t("faxtomail.demandeList.title");
                    icon = ui.getMenuFileDemandeList().getIcon();
                    break;
            }

//            JButton showHelp = ui.getShowHelp();
//            if (rightDecoration == null) {
//                rightDecoration = new JToolBar();
//                rightDecoration.setFloatable(false);
//                rightDecoration.setOpaque(false);
//                rightDecoration.setBorderPainted(false);
//            } else {
//                rightDecoration.remove(showHelp);
//            }
//            rightDecoration.add(showHelp, 0);
            this.currentBody = screenUI;
            getContext().getSwingSession().add(currentBody, true);
            ui.getBody().setTitle(screenTitle);
            ui.getBody().setLeftDecoration(new JLabel(icon));
            ui.getBody().setRightDecoration(rightDecoration);
            ui.getBody().add(currentBody);

        }
    }

    public void changeTitle() {

        String title = getSelectedCruiseTitle();

        ui.setTitle(t("faxtomail.main.title.application",
                      getConfig().getVersion(),
//                      "0.1",
                      title));
    }

    protected String getSelectedCruiseTitle() {


        String title;

//        if (context.isDbLoaded()) {
//
//            if (context.isProgramFilled()) {
//
//                // selected program
//
//                Program program = getDataContext().getProgram();
//
//                title = _("tutti.main.title.selectedProgram", program.getName()) + " / ";
//
//                if (context.isCruiseFilled()) {
//
//                    // selected cruise
//                    Cruise cruise = getDataContext().getCruise();
//
//                    if (cruise != null) {
//                        title += _("tutti.main.title.selectedCruise", cruise.getName());
//                    }
//                } else {
//
//                    // no selected cruise
//
//                    title += _("tutti.main.title.noSelectedCruise");
//                }
//
//            } else {
//
//                // no program selected (so neither cruise)
//
//                title = _("tutti.main.title.noSelectedProgram");
//
//            }
//
//            title += " / ";
//
//            if (context.isProtocolFilled()) {
//
//                // selected protocol
//                TuttiProtocol protocol = getDataContext().getProtocol();
//
//                title += _("tutti.main.title.selectedProtocol", protocol.getName());
//            } else {
//
//                // no selected protocol
//
//                title += _("tutti.main.title.noSelectedProtocol");
//            }
//        } else {

            // no db loaded

            title = t("faxtomail.main.title.nodb");
//        }
        return title;
    }

    public void setBodyTitle(String title) {
        ui.getBody().setTitle(title);
    }

    public JComponent getCurrentBody() {
        return currentBody;
    }
}
