/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyQueryParser.java 1462 2012-04-05 15:31:15Z maven-release $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.5/wikitty-api/src/main/java/org/nuiton/wikitty/query/WikittyQueryParser.java $
 * %%
 * Copyright (C) 2012 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.query;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.query.conditions.Aggregate;
import org.nuiton.wikitty.query.conditions.And;
import org.nuiton.wikitty.query.conditions.Between;
import org.nuiton.wikitty.query.conditions.Condition;
import org.nuiton.wikitty.query.conditions.ConditionValue;
import org.nuiton.wikitty.query.conditions.ConditionValueString;
import org.nuiton.wikitty.query.conditions.ContainsAll;
import org.nuiton.wikitty.query.conditions.ContainsOne;
import org.nuiton.wikitty.entities.Element;
import org.nuiton.wikitty.query.conditions.Equals;
import org.nuiton.wikitty.query.conditions.False;
import org.nuiton.wikitty.query.conditions.Greater;
import org.nuiton.wikitty.query.conditions.GreaterOrEquals;
import org.nuiton.wikitty.query.conditions.Keyword;
import org.nuiton.wikitty.query.conditions.Less;
import org.nuiton.wikitty.query.conditions.LessOrEquals;
import org.nuiton.wikitty.query.conditions.Like;
import org.nuiton.wikitty.query.conditions.Not;
import org.nuiton.wikitty.query.conditions.NotEquals;
import org.nuiton.wikitty.query.conditions.Or;
import org.nuiton.wikitty.query.conditions.NotNull;
import org.nuiton.wikitty.query.conditions.Null;
import org.nuiton.wikitty.query.conditions.Select;
import org.nuiton.wikitty.query.conditions.True;
import org.nuiton.wikitty.query.conditions.Unlike;
import org.parboiled.BaseParser;
import org.parboiled.Context;
import org.parboiled.Parboiled;
import org.parboiled.Rule;
import org.parboiled.annotations.BuildParseTree;
import org.parboiled.errors.ErrorUtils;
import org.parboiled.parserunners.RecoveringParseRunner;
import org.parboiled.parserunners.ReportingParseRunner;
import org.parboiled.parserunners.TracingParseRunner;
import org.parboiled.support.ParseTreeUtils;
import org.parboiled.support.ParsingResult;
import org.parboiled.support.Var;

/**
 * Cette classe permet d'interpreter une requete faite textuellement en la
 * convertisant en sa representation objet. Si l'objet est instancier pour
 * utiliser les fonctionnalites d'alias, il est possible de l'utiliser dans
 * plusieurs thread concurent. La map d'alias est protegee.
 *
 * Pour plus d'information reportez-vous à la
 * <a href="http://maven-site.nuiton.org/wikitty/user/query.html">documentation</a>
 *
 * @author poussin
 * @version $Revision: 1462 $
 * @since 3.3
 *
 * Last update: $Date: 2012-04-05 17:31:15 +0200 (jeu, 05 avr 2012) $
 * by : $Author: maven-release $
 */
@BuildParseTree
public class WikittyQueryParser extends BaseParser<Object> {

    public static final String AVG = "AVG";
    public static final String COUNT = "COUNT";
    public static final String MAX = "MAX";
    public static final String MIN = "MIN";
    public static final String SUM = "SUM";
    public static final String SELECT = "SELECT";
    public static final String WHERE = "WHERE";
    public static final String IN = "IN";
    public static final String LITERAL_OPEN_SIMPLE = "'";
    public static final String LITERAL_CLOSE_SIMPLE = "'";
    public static final String LITERAL_OPEN_DOUBLE = "\"";
    public static final String LITERAL_CLOSE_DOUBLE = "\"";
    public static final String NULL = "NULL";
    public static final String TO = "TO";
    public static final String FALSE = "FALSE";
    public static final String TRUE = "TRUE";
    public static final String UNLIKE = "UNLIKE";
    public static final String AND = "AND";
    public static final String COMMA = ",";
    public static final String CURLY_BRACKET_CLOSE = "}";
    public static final String CURLY_BRACKET_OPEN = "{";
    public static final String EQUALS = "=";
    public static final String EQUALS_IGNORE_CASE_AND_ACCENT = "~";
    public static final String GREATER = ">";
    public static final String GREATER_OR_EQUALS = ">=";
    public static final String LESS = "<";
    public static final String LESS_OR_EQUALS = "<=";
    public static final String LIKE = "LIKE";
    public static final String NOT = "NOT";
    public static final String NOT_EQUALS = "!=";
    public static final String NOT_EQUALS_IGNORE_CASE_AND_ACCENT = "!~";
    public static final String OR = "OR";
    public static final String BRACKET_CLOSE = ")";
    public static final String BRACKET_OPEN = "(";
    public static final String SQUARE_BRACKET_CLOSE = "]";
    public static final String SQUARE_BRACKET_OPEN = "[";

    public static final String OFFSET = "#OFFSET";
    public static final String LIMIT = "#LIMIT";

    public Rule icOFFSET = IgnoreCase(OFFSET);
    public Rule icLIMIT = IgnoreCase(LIMIT);
    
    public Rule icEXTENSION = IgnoreCase(Element.EXTENSION.getValue());
    public Rule icID = IgnoreCase(Element.ID.getValue());
    public Rule icNOT = IgnoreCase(NOT);
    public Rule icAND = IgnoreCase(AND);
    public Rule icOR = IgnoreCase(OR);
    public Rule icAVG = IgnoreCase(AVG);
    public Rule icCOUNT = IgnoreCase(COUNT);
    public Rule icMAX = IgnoreCase(MAX);
    public Rule icMIN = IgnoreCase(MIN);
    public Rule icSUM = IgnoreCase(SUM);
    public Rule icSELECT = IgnoreCase(SELECT);
    public Rule icWHERE = IgnoreCase(WHERE);
    public Rule icIN = IgnoreCase(IN);
    public Rule icTO = IgnoreCase(TO);
    public Rule icFALSE = IgnoreCase(FALSE);
    public Rule icTRUE = IgnoreCase(TRUE);
    public Rule icLIKE = IgnoreCase(LIKE);
    public Rule icUNLIKE = IgnoreCase(UNLIKE);
    public Rule icNULL = IgnoreCase(NULL);

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyQueryParser.class);

    protected Map<String, String> alias = new LinkedHashMap<String, String>();

    public WikittyQueryParser() {
    }

    boolean debug(String text, Context context) {
        System.out.println("DEBUG("+context.getCurrentIndex()+"):" + text + "  nodes:" +context.getSubNodes());
        return true;
    }

    /**
     * Retourne une vue non modifiable des alias disponibles.
     * @return
     */
    public Map<String, String> getAlias() {
        return Collections.unmodifiableMap(alias);
    }

    /**
     * Modifie l'ensemble des alias, la map passee en parametre est copiee en
     * interne.
     * @param alias la map des alias qui sera copiee
     * @return le WikittyQueryParser lui meme (this)
     */
    public WikittyQueryParser setAlias(Map<String, String> alias) {
        // on passe par la creation d'une nouvelle map, pour eviter que l'utilisateur
        // ne puisse modifier les alias depuis l'exterieur et eviter le plus
        // possible les synchronize
        this.alias = new LinkedHashMap<String, String>();
        if (alias != null) {
            this.alias.putAll(alias);
        }
        return this;
    }

    public WikittyQueryParser addAlias(String aliasName, String aliasValue) {
        synchronized(alias) {
            alias.put(aliasName, aliasValue);
        }
        return this;
    }

    public WikittyQueryParser clearAlias() {
        // on passe par la creation d'une nouvelle map, pour eviter le plus
        // possible les synchronize
        this.alias = new LinkedHashMap<String, String>();
        return this;
    }

    /**
     * Parse query and use alias added with {@link #addAlias} or {@link #setAlias}
     *
     * @param queryString query to parse
     * @return
     */
    public WikittyQuery parseQuery(String queryString) {
        WikittyQuery result = parse(queryString, alias);
        return result;
    }

    /**
     * Parse query without alias
     *
     * @param queryString query to parse
     * @return
     */
    static public WikittyQuery parse(String queryString) {
        WikittyQuery result = parse(queryString, null);
        return result;
    }

    /**
     * Parse query and use alias in argument
     *
     * @param queryString query to parse
     * @param alias alias to used to change query
     * @return
     */
    static public WikittyQuery parse(String queryString, Map<String, String> alias) {
        if (alias != null) {
            String queryStringInit = queryString;
            // on synchronise l'utilisation de la map, pour etre sur qu'il n'y
            // ait pas d'ajout/modif durant son utilisation pour le remplacement
            // on fait direcement les remplacements de la requete car, c'est
            // le seul endroit ou l'alias est utilise, et c'est aussi rapide
            // que de faire une copie de la map
            synchronized(alias) {
                // first replace alias in queryString
                for (Map.Entry<String, String> a : alias.entrySet()) {
                    queryString = queryString.replaceAll(a.getKey(), a.getValue());
                }
            }
            if (log.isDebugEnabled()) {
                log.debug(String.format("QueryString \n'%s' become after alias \n'%s'\naliases are %s",
                        queryStringInit, queryString, alias));
            }
        }

        WikittyQueryParser parser = Parboiled.createParser(WikittyQueryParser.class);

       ParsingResult<?> result = new ReportingParseRunner(parser.start()).run(queryString);
//        ParsingResult<?> result = new TracingParseRunner(parser.start()).run(queryString);
//        ParsingResult<?> result = new RecoveringParseRunner(parser.start()).run(queryString);

        if (result.hasErrors() || !result.matched) {
            System.out.println("\nParse Errors:\n" + ErrorUtils.printParseErrors(result));
        }

        if (log.isDebugEnabled()) {
            log.debug("\nParse Tree:\n" + ParseTreeUtils.printNodeTree(result) + '\n');
        }
//        System.out.println("\nParse Tree:\n" + ParseTreeUtils.printNodeTree(result) + '\n');
        
        WikittyQuery query = (WikittyQuery)result.resultValue;

        return query;
    }

    /**
     * can be field, extension name or id element
     * @param v
     * @return
     */
    protected Element toElement(String v) {
        Element result = Element.get(v);
        return result;
    }

    protected int toInt(String v) {
        int result = Integer.parseInt(v);
        return result;
    }

    /**
     * Remove quote at beginning and ending of String in parameter if necessary
     *
     * <li>"toto" return toto
     * <li>"toto return "toto
     * <li> toto return toto"
     * <li> to"to return to"to
     * 
     * @param s
     * @return
     */
    protected String removeQuote(String s) {
        String result = s;
        if (StringUtils.startsWithAny(s, LITERAL_OPEN_SIMPLE, LITERAL_OPEN_DOUBLE)
                && StringUtils.endsWithAny(s, LITERAL_CLOSE_SIMPLE, LITERAL_CLOSE_DOUBLE)) {
            result = StringUtils.substring(s, 1, -1);
        }
        return result;
    }

   Rule start() {
       return Sequence(or(), push(new WikittyQuery((Condition)pop())),
               offset(), limit(), space(), EOI);
   }

   Rule offset() {
       return Optional(space(), icOFFSET, FirstOf(EQUALS, space()), OneOrMore(AnyOf("1234567890")),
               push(((WikittyQuery)pop()).setOffset(toInt(match())))
               );
   }

   Rule limit() {
       return Optional(space(), icLIMIT, FirstOf(EQUALS, space()), OneOrMore(AnyOf("1234567890")),
               push(((WikittyQuery)pop()).setLimit(toInt(match())))
               );
   }

   Rule or() {
        return Sequence(and(), ZeroOrMore(space(), icOR, space(), and(),
                push(new Or((Condition)pop(1), (Condition)pop()))));
    }

    Rule and() {
        return Sequence(term(), ZeroOrMore(
                // when no AND or OR is used, AND is default
                // don't change order of FirstOf, this order is important
                FirstOf(Sequence(space(), icAND, space()), Sequence(space(), TestNot(icOR))),
                term(),
                push(new And((Condition)pop(1), (Condition)pop()))));
    }

    Rule term() {
        return FirstOf(condition(), Parens());
    }

    Rule Parens() {
        return Sequence(space(), BRACKET_OPEN, space(), or(), space(), BRACKET_CLOSE, space());
    }

    Rule condition() {
        // ATTENTION l'ordre est important par exemple le '>' doit etre apres le '>='
        return FirstOf(
                not(), isNull(), isNotNull(), select(),
                greatereq(), lesseq(),
                between(), containsAll(), containsOne(),
                eq(), neq(), eqIgnoreCaseAndAccent(), neqIgnoreCaseAndAccent(),
                less(), greater(), like(), notlike(),
                rTrue(), rFalse(), keyword()
                );
    }

    Rule not() {
        return Sequence(space(), icNOT, space(), term(),
                push(new Not((Condition)pop())));
    }

    /**
     * gere eq, startsWith, endsWith, isNull
     * @return
     */
    Rule isNull() {
        return Sequence(field(), push(match()), space(), EQUALS, space(), icNULL,
                push(new Null(toElement(pop().toString()))));
    }

    /**
     * gere eq, isNull
     * @return
     */
    Rule isNotNull() {
        return Sequence(field(), push(match()), space(), NOT_EQUALS, space(), icNULL,
                push(new NotNull(toElement(pop().toString()))));
    }

    /**
     * gere eq, startsWith, endsWith, isNull
     * @return
     */
    Rule eq() {
        return Sequence(field(), push(match()), space(), EQUALS, space(), value(),
                push(new Equals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }

    /**
     * gere eq, isNull
     * @return
     */
    Rule neq() {
        return Sequence(field(), push(match()), space(), NOT_EQUALS, space(), value(),
                push(new NotEquals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }

    /**
     * gere eq, startsWith, endsWith, isNull
     * @return
     */
    Rule eqIgnoreCaseAndAccent() {
        return Sequence(field(), push(match()), space(), EQUALS_IGNORE_CASE_AND_ACCENT, space(), value(),
                push(new Equals(toElement(pop(1).toString()), (ConditionValue)pop(), true)));
    }

    /**
     * gere eq, isNull
     * @return
     */
    Rule neqIgnoreCaseAndAccent() {
        return Sequence(field(), push(match()), space(), NOT_EQUALS_IGNORE_CASE_AND_ACCENT, space(), value(),
                push(new NotEquals(toElement(pop(1).toString()), (ConditionValue)pop(), true)));
    }

    Rule less() {
        return Sequence(field(), push(match()), space(), LESS, space(), value(),
                push(new Less(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule lesseq() {
        return Sequence(field(), push(match()), space(), LESS_OR_EQUALS, space(), value(),
                push(new LessOrEquals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule greater() {
        return Sequence(field(), push(match()), space(), GREATER, space(), value(),
                push(new Greater(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule greatereq() {
        return Sequence(field(), push(match()), space(), GREATER_OR_EQUALS, space(), value(),
                push(new GreaterOrEquals(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule like() {
        return Sequence(field(), push(match()), space(), icLIKE, space(), value(),
                push(new Like(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule notlike() {
        return Sequence(field(), push(match()), space(), icUNLIKE, space(), value(),
                push(new Unlike(toElement(pop(1).toString()), (ConditionValue)pop())));
    }
    Rule between() {
        return Sequence(field(), push(match()), space(), EQUALS, space(), SQUARE_BRACKET_OPEN, space(),
                value(), space(), icTO, space(),
                value(), space(), SQUARE_BRACKET_CLOSE,
                push(new Between(toElement(pop(2).toString()), (ConditionValue)pop(1), (ConditionValue)pop())));
    }
    Rule containsAll() {
        Var<List<ConditionValue>> elems = new Var<List<ConditionValue>>(new LinkedList<ConditionValue>());
        return Sequence(field(), push(match()), space(), EQUALS, space(), SQUARE_BRACKET_OPEN, space(),
                value(), elems.get().add((ConditionValue)pop()), space(), ZeroOrMore(COMMA, space(),
                value(), elems.get().add((ConditionValue)pop()), space()), SQUARE_BRACKET_CLOSE,
                push(new ContainsAll(toElement(pop().toString()), elems.get())));
    }
    Rule containsOne() {
        Var<List<ConditionValue>> elems = new Var<List<ConditionValue>>(new LinkedList<ConditionValue>());
        return Sequence(field(), push(match()), space(), EQUALS, space(), CURLY_BRACKET_OPEN, space(),
                value(), elems.get().add((ConditionValue)pop()), space(), ZeroOrMore(COMMA, space(),
                value(), elems.get().add((ConditionValue)pop()), space()), CURLY_BRACKET_CLOSE,
                push(new ContainsOne(toElement(pop().toString()), elems.get())));
    }
    Rule select() {
        Var<Aggregate> aggregate = new Var<Aggregate>();
        return Sequence(icSELECT, space(), Optional(aggregate(aggregate)), space(), field(), push(match()), space(), icWHERE, space(), term(),
                push(new Select(toElement(pop(1).toString()), aggregate.get(), (Condition)pop())));
    }
    Rule aggregate(Var<Aggregate> aggregate) {
        return FirstOf(avg(aggregate), count(aggregate), max(aggregate), min(aggregate), sum(aggregate));
    }

    Rule avg(Var<Aggregate> aggregate) {
        return Sequence(icAVG, aggregate.set(Aggregate.AVG));
    }

    Rule count(Var<Aggregate> aggregate) {
        return Sequence(icCOUNT, aggregate.set(Aggregate.COUNT));
    }

    Rule max(Var<Aggregate> aggregate) {
        return Sequence(icMAX, aggregate.set(Aggregate.MAX));
    }

    Rule min(Var<Aggregate> aggregate) {
        return Sequence(icMIN, aggregate.set(Aggregate.MIN));
    }

    Rule sum(Var<Aggregate> aggregate) {
        return Sequence(icSUM, aggregate.set(Aggregate.SUM));
    }

    Rule keyword() {
        return Sequence(value(), push(new Keyword().addCondition((ConditionValue)pop())));
    }

    Rule field() {
        return OneOrMore(FirstOf(CharRange('0', '9'), CharRange('a', 'z'), CharRange('A', 'Z'), AnyOf("_.*")));
    }

    Rule rTrue() {
        return Sequence(icTRUE, push(new True()));
    }

    Rule rFalse() {
        return Sequence(icFALSE, push(new False()));
    }

    Rule value() {
        return FirstOf(select(), valueText());
    }

    Rule valueText() {
        return Sequence(FirstOf(StringLiteralDouble(), StringLiteralSimple(), SimpleString()),
                push(new ConditionValueString(removeQuote(match()))));
    }

    /**
     * Une chaine simple sans espace, parenthese, retour chariot, tabulation,
     * accolade, crochet, ", '.
     * @return
     */
    Rule SimpleString() {
        return OneOrMore(FirstOf(
                Escape(),
                Sequence(TestNot(AnyOf(" #\t\r\n\\"
                +COMMA
                +LITERAL_OPEN_SIMPLE+LITERAL_OPEN_DOUBLE
                +BRACKET_OPEN+BRACKET_CLOSE
                +CURLY_BRACKET_OPEN+CURLY_BRACKET_CLOSE
                +SQUARE_BRACKET_OPEN+SQUARE_BRACKET_CLOSE)), ANY)
                )).suppressSubnodes();
    }

    Rule StringLiteralSimple() {
        return Sequence(
                LITERAL_OPEN_SIMPLE,
                ZeroOrMore(
                        FirstOf(
                                EscapeSimple(),
                                Sequence(TestNot(AnyOf("\r\n\\"+LITERAL_CLOSE_SIMPLE)), ANY)
                        )
                ).suppressSubnodes(),
                LITERAL_CLOSE_SIMPLE
                );
    }

    Rule StringLiteralDouble() {
        return Sequence(
                LITERAL_OPEN_DOUBLE,
                ZeroOrMore(
                        FirstOf(
                                EscapeDouble(),
                                Sequence(TestNot(AnyOf("\r\n\\"+LITERAL_CLOSE_DOUBLE)), ANY)
                        )
                ).suppressSubnodes(),
                LITERAL_CLOSE_DOUBLE
                );
    }

    Rule Escape() {
        return Sequence('\\', FirstOf(AnyOf("btnfr\'\\"
                +LITERAL_OPEN_SIMPLE+LITERAL_CLOSE_SIMPLE+LITERAL_OPEN_DOUBLE+LITERAL_CLOSE_DOUBLE),
                OctalEscape(), UnicodeEscape()));
    }

    Rule EscapeSimple() {
        return Sequence('\\', FirstOf(AnyOf("btnfr\'\\"+LITERAL_CLOSE_SIMPLE),
                OctalEscape(), UnicodeEscape()));
    }

    Rule EscapeDouble() {
        return Sequence('\\', FirstOf(AnyOf("btnfr\'\\"+LITERAL_CLOSE_DOUBLE),
                OctalEscape(), UnicodeEscape()));
    }

    Rule OctalEscape() {
        return FirstOf(
                Sequence(CharRange('0', '3'), CharRange('0', '7'), CharRange('0', '7')),
                Sequence(CharRange('0', '7'), CharRange('0', '7')),
                CharRange('0', '7')
        );
    }

    Rule UnicodeEscape() {
        return Sequence(OneOrMore('u'), HexDigit(), HexDigit(), HexDigit(), HexDigit());
    }

    Rule HexDigit() {
        return FirstOf(CharRange('a', 'f'), CharRange('A', 'F'), CharRange('0', '9'));
    }

    Rule space() {
        return ZeroOrMore(AnyOf(" \t\f"));
    }

}
