/*
 * #%L
 * Wikitty :: api
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty;

import static org.nuiton.i18n.I18n._;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.search.Search;

/**
 *
 * FIXME add security policy level two on wikittyAuthorisation to prevent writing
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyServiceSecurity implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceSecurity.class);

    protected WikittyService ws;

    /** cache de l'id du groupe AppAdmin */
    transient protected String appAdminGroupId = null;

    public WikittyServiceSecurity(WikittyService ws) {
        this.ws = ws;
    }

    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public String login(String login, String password) {
        Wikitty user = ws.findByCriteria(null, Search.query().eq(
                WikittyUser.FQ_FIELD_WIKITTYUSER_LOGIN, login).criteria());
        if (user == null) {
            throw new IllegalArgumentException(String.format("no such account '%s'", login));
        } else {
            // check password is valid
            if (WikittyUserHelper.getPassword(user).equals(password)) {
                String tokenId = WikittyUtil.genSecurityTokenId();
                Wikitty wikittyToken = new WikittyImpl(tokenId);
                // force add extension to wikitty
                WikittyTokenHelper.addExtension(wikittyToken);
                WikittyTokenHelper.setUser(wikittyToken, user.getId());
                ws.store(null, wikittyToken);
                log.debug(String.format("token '%s' is for login '%s'",
                                                               tokenId, login));
                return tokenId;
            } else {
                throw new SecurityException("bad password");
            }
        }
    }

    @Override
    public void logout(String securityToken) {
        if (securityToken == null) {
            throw new IllegalArgumentException("security token is null");
        } else {
            getUserId(securityToken); // will throw exception if token is not valid
            ws.delete(securityToken, securityToken);
        }
    }

    @Override
    public void clear(String securityToken) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            // seul les AppAdmin on le droit a cette method
            ws.clear(securityToken);
        } else {
            throw new SecurityException(_("user %s can't clear data", userId));
        }
    }

    /**
     * if app-admin group exists, return true if given userId is app-admin
     * if app-admin group doesn't exists, return true if user is anonymous
     */
    protected boolean userIsAnonymousOrAppAdmin(String securityToken, String userId) {
        boolean userIsAnonymousOrAppAdmin = false;

        if (getAppAdminGroup(securityToken) == null) {
            if (securityToken == null) {
                // user is anonymous
                userIsAnonymousOrAppAdmin = true;
            }
        } else {
            if (isAppAdmin(securityToken, userId)) {
                // user is appAdmin
                userIsAnonymousOrAppAdmin = true;
            }
        }

        return userIsAnonymousOrAppAdmin;
    }

    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        Collection<Wikitty> wikitties = Arrays.asList(wikitty);
        wikitties = checkStore(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, wikitties);
        return result;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        Collection<Wikitty> wikittiesToStore = checkStore(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, wikittiesToStore);
        return result;
    }
    
    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties, boolean force) {
        Collection<Wikitty> wikittiesToStore = checkStore(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, wikittiesToStore, force);
        return result;
    }

    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction, Collection<Wikitty> wikitties, boolean force) {
        Collection<Wikitty> wikittiesToStore = checkStore(securityToken, wikitties);
        UpdateResponse result = ws.store(securityToken, transaction, wikittiesToStore, force);
        return result;
    }

    protected Collection<Wikitty> checkStore(String securityToken, Collection<Wikitty> wikitties) {
        String userId = getUserId(securityToken);
        List<Wikitty> wikittiesToStore = new ArrayList<Wikitty>();
        for (Wikitty wikitty : wikitties) {
            
            // usual case, a user want to store a wikitty
            Wikitty oldVersion = ws.restore(securityToken, wikitty.getId());

            Collection<String> newExtensions = new ArrayList<String>(wikitty.getExtensionNames());
            if (oldVersion != null) {
                // we already checked the rights for those extension
                // re-do the check has too much cost, avoid it
                newExtensions.removeAll(oldVersion.getExtensionNames());
            }

            // check that **reader** right on Security for all extension
            for (String extensionName: newExtensions) {

                Wikitty extensionRights = restoreExtensionAuthorisation(
                        securityToken, extensionName);
                boolean canCreate = extensionRights == null ||
                                canRead(securityToken, userId, null, extensionRights);
                if ( ! canCreate ) {
                   throw new SecurityException(_(
                           "user %s can't create instance of extension %s",
                           userId, extensionRights));
                }
            }

            if (oldVersion != null) { // it's an update

                for (String fqFieldDirtyName : wikitty.getDirty()) {
                    
                    String concernedExtensionName = WikittyUtil.getExtensionNameFromFQFieldName(fqFieldDirtyName);
                    
                    if (log.isTraceEnabled()) {
                        log.trace(String.format("will update field %s from extension %s",
                                               fqFieldDirtyName, concernedExtensionName));
                    }

                    boolean fieldRequireAdminRights = // true if field is a field of WikittyAuthorisation
                                // concerned extension is "WikittyAuthorisation"
                                WikittyAuthorisation.EXT_WIKITTYAUTHORISATION.equals(concernedExtensionName)
                                // or concerned extension is something like "AnyExtension:WikittyAuthorisation"
                             || WikittyAuthorisation.EXT_WIKITTYAUTHORISATION.equals(
                                        WikittyUtil.getMetaExtensionNameFromFQMetaExtensionName(concernedExtensionName));

                    boolean canChange; // will be true if user can modify the value of this field
                                       // according to his level of rights
                    if (fieldRequireAdminRights) {
                        canChange = canAdmin(securityToken, userId, concernedExtensionName, wikitty);
                    } else {
                        canChange = canWrite(securityToken, userId, concernedExtensionName, wikitty);
                    }

                    if (canChange) {
                        Object newValue = wikitty.getFqField(fqFieldDirtyName);
                        oldVersion.setFqField(fqFieldDirtyName, newValue);
                    } else {
                        throw new SecurityException(_("user %s can't write field %s on wikitty %s",
                                userId, fqFieldDirtyName, wikitty));
                    }
                }
            }

            wikittiesToStore.add(wikitty);
        }
        return wikittiesToStore;
    }

    @Override
    public Wikitty restore(String securityToken, String id) {
        String userId = getUserId(securityToken);
        Wikitty wikitty = ws.restore(securityToken, id);
        if (wikitty != null) {
            refuseUnauthorizedRead(securityToken, userId, wikitty);
        }
        return wikitty;
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        String userId = getUserId(securityToken);
        List<Wikitty> wikitties = ws.restore(securityToken, ids);
        for (Wikitty wikitty : wikitties) {
            refuseUnauthorizedRead(securityToken, userId, wikitty);
        }
        return wikitties;
    }

    @Override
    public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> ids) {
        String userId = getUserId(securityToken);
        List<Wikitty> wikitties = ws.restore(securityToken, transaction, ids);
        for (Wikitty wikitty : wikitties) {
            refuseUnauthorizedRead(securityToken, userId, wikitty);
        }
        return wikitties;
    }

    /** throw an exception if read is not allowed */
    protected void refuseUnauthorizedRead( String securityToken,
                                           String userId,
                                           Wikitty wikitty) {
        if (wikitty != null) {
            for (String extensionName : wikitty.getExtensionNames()) {
                if ( ! canRead(securityToken, userId, extensionName, wikitty)) {
                    throw new SecurityException(_("user %s can't read extension %s on wikitty %s, it may be due to a global policy on the wikitty",
                                                userId, extensionName, wikitty));
                }
            }
        }
    }
    
    protected boolean canRead(String securityToken, String userId,
                              String extensionName, Wikitty wikitty) {
        
        boolean canRead = false;

        // first, check per-extension rights
        if (wikitty.hasMetaExtension(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                                     extensionName)) {
            // there is a policy on the extension
            canRead = isReader(securityToken, userId, wikitty, extensionName)
                   || canWrite(securityToken, userId, extensionName, wikitty);
        } else if ( ! canRead && 
                    wikitty.hasExtension(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION) ) {
            // there is no policy for this extension
            // but there is a policy for all extension of wikitty
            canRead = isReader(securityToken, userId, wikitty, null)
                   || canWrite(securityToken, userId, extensionName, wikitty);            
        } else {
            // no security policy, everything is allowed
            canRead = true;
        }

        return canRead;
    }

    protected boolean canWrite(String securityToken, String userId,
                               String extensionName, Wikitty wikitty) {
        boolean canWrite = false;

        // first, check per-extension rights
        if (wikitty.hasMetaExtension(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                                     extensionName)) {
            // there is a policy on the extension of fqFieldDirtyName
            canWrite = isWriter(securityToken, userId, wikitty, extensionName)
                    || canAdmin(securityToken, userId, extensionName, wikitty);
        } else if ( ! canWrite &&
                    wikitty.hasExtension(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION) ) {
            // there is no policy for this extension
            // but there is a policy for all extension of wikitty
            canWrite = isWriter(securityToken, userId, wikitty, null)
                    || canAdmin(securityToken, userId, extensionName, wikitty);            
        } else {
            // no security policy, everything is allowed
            canWrite = true;
        }

        return canWrite;
    }

    protected boolean canAdmin(String securityToken, String userId,
                               String extensionName, Wikitty wikitty) {

        boolean canAdmin = false;

        // first, check per-extension rights
        if (wikitty.hasMetaExtension(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                                     extensionName)) {
            // there is a policy on the extension of fqFieldDirtyName
            canAdmin = isAdmin(securityToken, userId, wikitty, extensionName)
                    || isOwner(securityToken, userId, wikitty, extensionName);
        } else if ( ! canAdmin &&
                    wikitty.hasExtension(WikittyAuthorisation.EXT_WIKITTYAUTHORISATION) ) {
            // there is no policy for this extension
            // but there is a policy for all extension of wikitty
            canAdmin = isAdmin(securityToken, userId, wikitty, null)
                    || isOwner(securityToken, userId, wikitty, null);
        } else if ( ! canAdmin ) {
            // still not admin, check appAdmin
            canAdmin = userIsAnonymousOrAppAdmin(securityToken, userId);
        }

        return canAdmin;
    }

    @Override
    public void delete(String securityToken, String id) {
        Collection<String> ids = Arrays.asList(id);
        delete(securityToken, ids);
    }

    @Override
    public void delete(String securityToken, Collection<String> ids) {
        String userId = getUserId(securityToken);
        List<String> idsAsList = new ArrayList<String>(ids);
        List<Wikitty> wikitties = ws.restore(securityToken, idsAsList);
        for (Wikitty wikitty : wikitties) {
            for (String extensionName : wikitty.getExtensionNames()) {
                if ( ! canWrite(securityToken, userId, extensionName, wikitty)) {
                    throw new SecurityException(_(
                            "user %s doesn't have rights on extension %s on wikitty %s",
                            userId, extensionName, wikitty));
                }
            }
        }
        ws.delete(securityToken, ids);
    }

    @Override
    @Deprecated
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public boolean canDelete(String securityToken, String wikittyId) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Deprecated
    public boolean canRead(String securityToken, String wikittyId) {
        throw new UnsupportedOperationException();
    }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        // All people can read extension
        return ws.getAllExtensionIds(securityToken);
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        // All people can read extension
        return ws.getAllExtensionsRequires(securityToken, extensionName);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken, WikittyExtension ext) {
        Collection<WikittyExtension> exts = Arrays.asList(ext);
        checkStoreExtension(securityToken, exts);
        return storeExtension(securityToken, exts);
    }

    
    
    /* *** storing and restoring extensions ***/

    protected void checkStoreExtension(String securityToken, 
                                       Collection<WikittyExtension> exts) {
        String userId = getUserId(securityToken);
        if ( ! userIsAnonymousOrAppAdmin(securityToken, userId)) {
            for (WikittyExtension extension : exts) {
                Wikitty extensionAuthorisation = restoreExtensionAuthorisation(securityToken, extension.getName());
                if (extensionAuthorisation != null) {
                    // canWrite is true if this user can modify the field for this extension
                    boolean canWrite = canWrite(securityToken, userId, null, extensionAuthorisation);
                    if ( ! canWrite) {
                        throw new SecurityException(_("user %s don't have write right for extension %s", userId, extension));
                    }
                }
            }
        }
    }
    
    @Override
    public UpdateResponse storeExtension(String securityToken,
                                         Collection<WikittyExtension> exts) {
        checkStoreExtension(securityToken, exts);
        return ws.storeExtension(securityToken, exts);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        checkStoreExtension(securityToken, exts);
        return ws.storeExtension(securityToken, transaction, exts);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, String id) {
        // All people can read extension
        return ws.restoreExtension(securityToken, id);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken,
            WikittyTransaction transaction, String id) {
        // All people can read extension
        return ws.restoreExtension(securityToken, transaction, id);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(String securityToken,
                                                        String name) {
        // All people can read extension
        return ws.restoreExtensionLastVersion(securityToken, name);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name) {
        // All people can read extension
        return ws.restoreExtensionLastVersion(securityToken, transaction, name);
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken, Criteria criteria) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByCriteria(securityToken, criteria);
        return result;
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
                    WikittyTransaction transaction, Criteria criteria) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByCriteria(
                securityToken, transaction, criteria);
        return result;
    }

    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        String userId = getUserId(securityToken);
        Wikitty wikitty = ws.findByCriteria(securityToken, criteria);
        refuseUnauthorizedRead(securityToken, userId, wikitty);
        return wikitty;
    }

    @Override
    public WikittyTree restoreTree(String securityToken, String wikittyId) {
        String userId = getUserId(securityToken);
        WikittyTree restoredTree = ws.restoreTree(securityToken, wikittyId);
        checkRestoreTree(securityToken, userId, restoredTree);
        return restoredTree;
    }

    protected void checkRestoreTree(String securityToken, String userId, WikittyTree tree) {
        checkRestoreTreeNode(securityToken, userId, tree.node);
        for (WikittyTree subTree : tree.getChildren()) {
            checkRestoreTree(securityToken, userId, subTree);
        }
    }


    protected void checkRestoreTreeNode(String securityToken, String userId, WikittyTreeNode treeNode) {
        refuseUnauthorizedRead(securityToken, userId, treeNode.getWikitty());
    }

    @Override
    public Entry<WikittyTreeNode, Integer> restoreNode(String securityToken, String wikittyId, Criteria filter) {
        String userId = getUserId(securityToken);
        Entry<WikittyTreeNode, Integer> entry = ws.restoreNode(securityToken, wikittyId, filter);
        checkRestoreTreeNode(securityToken, userId, entry.getKey());
        return entry;
    }

    @Override
    public Map<WikittyTreeNode, Integer> restoreChildren(String securityToken,
                                                  String wikittyId,
                                                  Criteria filter) {
        String userId = getUserId(securityToken);
        Map<WikittyTreeNode, Integer> children = ws.restoreChildren(securityToken, wikittyId, filter);
        for (Map.Entry<WikittyTreeNode, Integer> child : children.entrySet()) {
            checkRestoreTreeNode(securityToken, userId, child.getKey());
        }
        return children;
    }

    @Override
    public List<String> deleteTree(String securityToken, String treeNodeId) {
        WikittyTreeNode treeNode = ws.restoreNode(securityToken, treeNodeId, null).getKey();
        Collection<Wikitty> wikitties = Arrays.asList(treeNode.getWikitty());
        checkStore(securityToken, wikitties);
        return ws.deleteTree(securityToken, treeNodeId);
    }

    @Override
    public Wikitty restoreVersion(String securityToken, String wikittyId, String version) {
        Wikitty wikitty = ws.restoreVersion(securityToken, wikittyId, version);
        String userId = getUserId(securityToken);
        refuseUnauthorizedRead(securityToken, userId, wikitty);
        return wikitty;
    }

    @Override
    public Wikitty findByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        Wikitty wikitty = ws.findByCriteria(securityToken, transaction, criteria);
        String userId = getUserId(securityToken);
        refuseUnauthorizedRead(securityToken, userId, wikitty);
        return wikitty;
    }

    @Override
    public UpdateResponse syncEngin(String securityToken) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            // seul les AppAdmin on le droit a cette method
            return ws.syncEngin(securityToken);
        } else {
            throw new SecurityException(_("user %s can't sync search engine",
                    getUserId(securityToken)));
        }
    }

    //
    // Method helper to check right
    //

    /** tell who own a token (who got this token after login).
     * @param securityToken the token whose owner will be returned
     * @return a wikitty Id (wikitty has extension WikittyUser)
     */
    protected String getUserId(String securityToken) {
        String result = null;
        // recuperation de l'utilisateur associe au securityToken
        // le securityToken est aussi l'id de l'objet
        if (securityToken != null) {
            Wikitty securityTokenWikitty = ws.restore(securityToken, securityToken);
            if (securityTokenWikitty == null) {
                throw new SecurityException("bad (obsolete ?) token");
            } else {
                result = WikittyTokenHelper.getUser(securityTokenWikitty);
            }
        }
        return result;
    }

    /**
     * 
     * @param securityToken
     * @param userId
     * @param wikitty
     * @param extensionName may be null
     * @return
     */
    protected boolean isReader(String securityToken, String userId, Wikitty wikitty, String extensionName) {
        boolean result;
        String metaFieldName = WikittyUtil.getMetaFieldName(
                WikittyAuthorisation.EXT_WIKITTYAUTHORISATION, extensionName,
                WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_READER);
        result = isMember(securityToken, userId, wikitty, metaFieldName, true);
        return result;
    }

    /**
     * 
     * @param securityToken
     * @param userId
     * @param wikitty
     * @param extensionName may be null
     * @return
     */
    protected boolean isWriter(String securityToken, String userId, Wikitty wikitty, String extensionName) {
        boolean result;
        String metaFieldName = WikittyUtil.getMetaFieldName(
                WikittyAuthorisation.EXT_WIKITTYAUTHORISATION, extensionName,
                WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_WRITER);
        log.trace("meta field name " + metaFieldName);
        result = isMember(securityToken, userId, wikitty, metaFieldName);
        return result;
    }

    /**
     * 
     * @param securityToken
     * @param userId
     * @param wikitty
     * @param extensionName may be null
     * @return
     */
    protected boolean isAdmin(String securityToken, String userId, Wikitty wikitty, String extensionName) {
        boolean result;
        String metaFieldName = WikittyUtil.getMetaFieldName(
                WikittyAuthorisation.EXT_WIKITTYAUTHORISATION, extensionName,
                WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_ADMIN);
        result = isMember(securityToken, userId, wikitty, metaFieldName);
        return result;
    }

    /** true if given user is owner
     * 
     * @param securityToken
     * @param userId
     * @param wikitty
     * @param extensionName may be null
     * @return
     */
    protected boolean isOwner(String securityToken, String userId, Wikitty wikitty, String extensionName) {
        
        String metaFieldName = WikittyUtil.getMetaFieldName(
                WikittyAuthorisation.EXT_WIKITTYAUTHORISATION, extensionName,
                WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_OWNER);
        
        String actualExtensionName = WikittyUtil.getExtensionNameFromFQFieldName(metaFieldName);
        String fieldName = WikittyUtil.getFieldNameFromFQFieldName(metaFieldName);
        
        String owner = wikitty.getFieldAsString(actualExtensionName, fieldName);
        
        boolean isOwner;
        if (owner == null) {
            isOwner = false;
        } else {
            isOwner = owner.equals(userId);
        }
        return isOwner;
    }

    /** {@link #isMember(String, String, Wikitty, String, boolean)} with default value */
    protected boolean isMember(String securityToken, String userId, Wikitty extensionRights, String fqFieldName) {
        // by default, user is considered not member if he is not in the group, so passing "false"
        return isMember(securityToken, userId, extensionRights, fqFieldName, false);
    }

    /** check if a user is listed in a level of rights
     *
     * @param securityToken
     * @param userId the userId to look for
     * @param extensionRights a wikitty with WikittyAuthorisation as extension <strong>OR</strong> meta-extension
     * @param fqFieldName the field to look into, it should be one of the field of extension WikittyAuthorisation
     *                    it has to be a FQN and may contain an extension-name if using meta-extension
     * @param considerEmptyGroupAsMembership if true, an empty field value will be considered as
     *        "every-one is in the group". Most of the time, it will be false but true should be
     *        passed for "reader" level because user has right to read if he belongs to "reader" OR
     *        if reader is empty
     * @return true if userId appear in the single/list of group/user of given field
     */
    protected boolean isMember(String securityToken, String userId,
                Wikitty extensionRights, String fqFieldName, boolean considerEmptyGroupAsMembership) {

        String extensionName = WikittyUtil.getExtensionNameFromFQFieldName(fqFieldName);
        String fieldName = WikittyUtil.getFieldNameFromFQFieldName(fqFieldName);

        Set<String> groupOrUser = extensionRights.getFieldAsSet(extensionName,
                                                                fieldName,
                                                                String.class);

        boolean isMember;
        if (groupOrUser == null || groupOrUser.isEmpty()) {
            isMember = considerEmptyGroupAsMembership;
        } else {
            isMember = isMember(securityToken, userId, groupOrUser);
        }

        if ( ! isMember) {
            // user don't have right on current object, check parent right
            String parentId = WikittyAuthorisationHelper.getParent(extensionRights);
            if (parentId != null) {
                Wikitty parent = ws.restore(securityToken, parentId);
                isMember = isMember(securityToken, userId, parent, fqFieldName);
            }
        }
        return isMember;
    }

    /** check if a given user belong to the group of app-admins. */
    protected boolean isAppAdmin(String securityToken, String userId) {
        Wikitty group = getAppAdminGroup(securityToken);
        Set<String> ids = WikittyGroupHelper.getMembers(group);
        boolean result = isMember(securityToken, userId, ids);
        return result;
    }
    
    /** get the wikitty with extension WikittyGroup that contains all app-admin. */
    protected Wikitty getAppAdminGroup(String securityToken) {
        Wikitty group;
        if (appAdminGroupId == null) {
            // 1er fois, on le recherche
            group = ws.findByCriteria(securityToken, Search.query().eq(
                    WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME,
                    WikittySecurityHelper.WIKITTY_APPADMIN_GROUP_NAME).criteria());
            // group peut-etre null s'il n'existe pas
        } else {
            // on a deja fait la recherche precedement, on essaie de reutilise
            // le meme id
            group = ws.restore(securityToken, appAdminGroupId);
            // group peut-etre null, si entre temps un admin a supprime le group
        }

        return group;
    }

    /**
     * Verifie recursivement si un utilisateur est dans un groupe qui peut etre
     * constitue d'autre groupe ou d'utilisateur
     *
     * @param userId l'utilisateur recherche
     * @param groupOrUser la liste des id d'utilisateurs ou d'autres groupes
     * @return vrai si userId est retrouve, false sinon
     */
    protected boolean isMember(
            String securityToken, String userId, Set<String> groupOrUser) {
        if (groupOrUser != null) {
            for (String id : groupOrUser) {
                if (id.equals(userId)) {
                    return true;
                } else {
                    Wikitty groupWikitty = ws.restore(securityToken, id);
                    if (WikittyGroupHelper.hasExtension(groupWikitty)) {
                        Set<String> members = WikittyGroupHelper.getMembers(groupWikitty);
                        return isMember(securityToken, userId, members);
                    }
                }
            }
        }
        return false; // not found in groupOrUser
    }

    /**
     * restore the wikitty authorisation attached to given extension.
     *
     * @return a wikitty with WikittyAuthorisation extension, or null if given
     *         extension has no security policy attached
     */
    protected Wikitty restoreExtensionAuthorisation(String securityToken,
                                                 WikittyExtension extension) {
        return restoreExtensionAuthorisation(securityToken, extension.getName());
    }

    /**
     * restore the wikitty authorisation attached to given extension.
     *
     * @return a wikitty with WikittyAuthorisation extension, or null if given
     *         extension has no security policy attached
     */
    protected Wikitty restoreExtensionAuthorisation(String securityToken,
                                                 String extensionName) {
        String wikittyAuthorisationId = WikittyMetaExtensionUtil.generateId(
                WikittyAuthorisation.EXT_WIKITTYAUTHORISATION, extensionName);
        Wikitty wikittyAuthorisation = ws.restore(securityToken, wikittyAuthorisationId);
        if (wikittyAuthorisation == null) {
            log.debug(extensionName + " has no authorization attached");
        }
        return wikittyAuthorisation;
    }


}
