/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: JGroupsNotifierTransporter.java 426 2010-10-19 14:51:48Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-2.2.2/wikitty-api/src/main/java/org/nuiton/wikitty/JGroupsNotifierTransporter.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty;

import java.util.Properties;
import org.apache.commons.lang.StringUtils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jgroups.Channel;
import org.jgroups.ChannelException;
import org.jgroups.JChannel;
import org.jgroups.Message;
import org.jgroups.ReceiverAdapter;

/**
 * JGroups notifier.
 * 
 * @author chatellier
 * @version $Revision: 426 $
 * 
 * Last update : $Date: 2010-10-19 16:51:48 +0200 (mar., 19 oct. 2010) $
 * By : $Author: bpoussin $
 */
public class JGroupsNotifierTransporter extends ReceiverAdapter
        implements WikittyServiceNotifier.RemoteNotifierTransporter {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(JGroupsNotifierTransporter.class);

    /**
     * Utilisation du canal de communication basé sur jgroups avec
     * comme identifiant d'application le nom de canal en option.
     *
     * Si ce nom est vide, jgroups n'est pas utilisé.
     * Si {@link #WIKITTY_EVENT_PROPAGATE_OPTION} est a true et que cette
     * options est vide, une exception est levée.
     */
    static public final String WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION = "wikitty.service.event.jgroupschannelname";

    /** Notifier service reference reference. */
    protected WikittyServiceNotifier ws;


    /** JGroup channel. */
    protected JChannel channel;

    public JGroupsNotifierTransporter(WikittyServiceNotifier ws, Properties props) {
        this.ws = ws;

        // can be null according to default constructor
        if (props != null) {
            // add notifier as listener
            String jgroupChannel = props.getProperty(WIKITTY_EVENT_JGROUPCHANNELNAME_OPTION);
            if (!StringUtils.isBlank(jgroupChannel)) {
                initChannel(jgroupChannel);
            } else {
                throw new IllegalArgumentException("Can't use propagate cache without a valid jgroups channel name !!!");
            }
        }
    }

    /**
     * Init jgroup channel.
     * 
     * @param channelName channel name
     */
    protected void initChannel(String channelName) {
        if (log.isDebugEnabled()) {
            log.debug("Init jgroup communication channel...");
        }

        try {
            // use default udp.xml in classpath
            channel = new JChannel();
            channel.setReceiver(this);

            // don't receive messages sent by myself
            channel.setOpt(Channel.LOCAL, false);

            channel.connect(channelName);

            if (log.isInfoEnabled()) {
                log.info("JGroup communication channel initialized to "
                        + channel.getAddressAsString() + " ("
                        + channel.getClusterName() + ")");
                log.info("Channel view: " + channel.getView());
            }
        } catch (ChannelException eee) {
            throw new WikittyException("Can't initialize jgroup channel", eee);
        }
    }

    /**
     * Send a jgroup message to all other channel member.
     * 
     * @param event message to send
     */
    @Override
    public void sendMessage(WikittyServiceEvent event) throws Exception {
        Message msg = new Message(null, null, event);
        channel.send(msg);
    }

    /*
     * @see org.jgroups.ReceiverAdapter#receive(org.jgroups.Message)
     */
    @Override
    public void receive(Message msg) {

        Object message = msg.getObject();

        if (log.isInfoEnabled()) {
            log.info("Receive message : " + message);
        }

        if (message instanceof WikittyServiceEvent) {
            WikittyServiceEvent event = (WikittyServiceEvent)message;
            ws.processRemoteEvent(event);
        }
    }

}
