/* *##%
 * Copyright (c) 2010 poussin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.nuiton.wikitty;

import static org.nuiton.i18n.I18n._;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.search.Search;

/**
 *
 * @author poussin
 * @version $Revision$
 *
 * Last update: $Date$
 * by : $Author$
 */
public class WikittyServiceSecurity implements WikittyService {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(WikittyServiceSecurity.class);

    /** nom du groupe des administrateurs de l'application */
    static final public String WIKITTY_APPADMIN_GROUP_NAME = "WikittyAppAdmin";

    protected WikittyService ws;

    /** cache de l'id du groupe AppAdmin */
    transient protected String appAdminGroupId = null;


    public WikittyServiceSecurity(WikittyService ws) {
        this.ws = ws;
    }

    @Override
    public void addWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public void removeWikittyServiceListener(WikittyServiceListener listener, ServiceListenerType type) {
        ws.addWikittyServiceListener(listener, type);
    }

    @Override
    public String login(String login, String password) {
        String token = WikittyUtil.genSecurityToken();
        Wikitty wToken = new WikittyImpl(token);
        // force add extension to wikitty
        SecurityTokenHelper.addExtension(wToken);
        // on passe token comme securityToken, mais il ne me semble pas
        // que ce soit tres utile, mais comme ca c'est uniform
        ws.store(null, wToken);
        return token;
    }

    @Override
    public void logout(String securityToken) {
        // on passe securityToken comme token, mais il ne me semble pas
        // que ce soit tres utile, mais comme ca c'est uniform
        ws.delete(null, securityToken);
    }

    @Override
    public void clear(String securityToken) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            // seul les AppAdmin on le droit a cette method
            ws.clear(securityToken);
        } else {
            throw new SecurityException(_("user %s can't clear data",
                    getUserId(securityToken)));
        }
    }

    /**
     * Prepare l'ecriture en ajoutant s'il le faut l'extension
     * WikittyAuthorisation et en fixant l'owner a l'utilisateur courant
     *
     * @param securityToken le token de securite qui permet de retrouver
     * l'utilisateur
     * @param wikitty le wikitty a sauver
     */
    protected void prepareWrite(String securityToken, Wikitty wikitty) {
        Wikitty oldVersion = ws.restore(securityToken, wikitty.getId());
        if (oldVersion == null) {
            // creation d'une nouvelle entity, on a des choses a faire

            // recuperation de l'utilisateur associe au securityToken
            String userId = getUserId(securityToken);

            // on ajoute et on fixe les droits par defaut
            WikittyAuthorisationHelper.addExtension(wikitty);
            WikittyAuthorisationHelper.setOwner(wikitty, userId);
        }
    }

    @Override
    public boolean canWrite(String securityToken, Wikitty wikitty) {
        boolean result = false;

        String userId = getUserId(securityToken);
        //
        // check security
        //

        // recuperation de l'ancienne version de l'objet pour verifier les droits
        Wikitty oldVersion = ws.restore(securityToken, wikitty.getId());
        if (oldVersion == null) {
            // creation d'une nouvelle entity

            // on verifie que l'on a le droit de creer une entity avec cette extension
            // TODO poussin 20100607 trouver ou mettre l'autorisation qui retient l'information de qui a le droit de cree une extension
            result = true;
        } else {
            // modification d'une entity existante

            // si c'est le owner il a tous les droits, a defaut les admins
            // peuvent aussi le modifier
            result =
                // owner et admin peuvent tout modifier
                    isOwner(securityToken, userId, oldVersion)
                    || isAppAdmin(securityToken, userId)
                    || isAdmin(securityToken, userId, oldVersion)
                // un writer ne peut pas modifier l'extension d'autorisation
                    || (WikittyAuthorisationAbstract.equals(oldVersion, wikitty)
                        && isWriter(securityToken, userId, oldVersion));
        }
        return result;
    }

    @Override
    public boolean canDelete(String securityToken, String wikittyId) {
        boolean result = false;

        //
        // check security
        //

        // recuperation de l'ancienne version de l'objet pour verifier les droits
        Wikitty oldVersion = ws.restore(securityToken, wikittyId);
        if (oldVersion == null) {
           // l'objet n'existe pas donc la suppression retourne true
            result = true;
        } else {
            // suppresion d'une entity existante
            String userId = getUserId(securityToken);

            // si c'est le owner il a tous les droits, a defaut les admins
            // peuvent aussi le supprimer
            result =
                // owner et admin peuvent tout modifier
                    isOwner(securityToken, userId, oldVersion)
                    || isAppAdmin(securityToken, userId)
                    || isAdmin(securityToken, userId, oldVersion);
        }
        return result;
    }
    
    @Override
    public boolean canRead(String securityToken, String wikittyId) {
        // recuperation de l'utilisateur associe au securityToken
        Wikitty securityTokenWikitty = ws.restore(securityToken, securityToken);
        String userId = SecurityTokenHelper.getUser(securityTokenWikitty);

        //
        // check security
        //

        // recuperation de l'objet pour verifier les droits
        Wikitty w = ws.restore(securityToken, wikittyId);
        boolean result = isReader(securityToken, userId, w)
                || isOwner(securityToken, userId, w)
                || isAppAdmin(securityToken, userId)
                || isAdmin(securityToken, userId, w)
                || isWriter(securityToken, userId, w);
        return result;
    }

    @Override
    public UpdateResponse store(String securityToken, Wikitty wikitty) {
        if (canWrite(securityToken, wikitty)) {
            prepareWrite(securityToken, wikitty);
            UpdateResponse result = ws.store(securityToken, wikitty);
            return result;
        } else {
            throw new SecurityException(_("user %s can't modify object %s",
                    getUserId(securityToken), wikitty.getId()));
        }
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties) {
        for (Wikitty w : wikitties) {
            if (!canWrite(securityToken, w)) {
                throw new SecurityException(_("user %s can't modify object %s",
                        getUserId(securityToken), w.getId()));
            }
        }
        for (Wikitty w : wikitties) {
            prepareWrite(securityToken, w);
        }
        UpdateResponse result = ws.store(securityToken, wikitties);
        return result;
    }

    @Override
    public UpdateResponse store(String securityToken, Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        for (Wikitty w : wikitties) {
            if (!canWrite(securityToken, w)) {
                throw new SecurityException(_("user %s can't modify object %s",
                        getUserId(securityToken), w.getId()));
            }
        }
        for (Wikitty w : wikitties) {
            prepareWrite(securityToken, w);
        }
        UpdateResponse result = ws.store(securityToken, wikitties, disableAutoVersionIncrement);
        return result;
    }

    @Override
    public UpdateResponse store(String securityToken, WikittyTransaction transaction,
            Collection<Wikitty> wikitties, boolean disableAutoVersionIncrement) {
        for (Wikitty w : wikitties) {
            if (!canWrite(securityToken, w)) {
                throw new SecurityException(_("user %s can't modify object %s",
                        getUserId(securityToken), w.getId()));
            }
        }
        for (Wikitty w : wikitties) {
            // preparation des wikitty pour la sauvegarde
            // - ajout extension d'autorisation si necessaire
            prepareWrite(securityToken, w);
        }
        UpdateResponse result = ws.store(securityToken, transaction, wikitties,
                disableAutoVersionIncrement);
        return result;
   }

    @Override
    public List<String> getAllExtensionIds(String securityToken) {
        // All people can read extension
        return ws.getAllExtensionIds(securityToken);
    }

    @Override
    public List<String> getAllExtensionsRequires(
            String securityToken, String extensionName) {
        // All people can read extension
        return ws.getAllExtensionsRequires(securityToken, extensionName);
    }

    @Override
    public UpdateResponse storeExtension(
            String securityToken, WikittyExtension ext) {
        // TODO poussin 20100607 check security, mais qui a le droit ?
        return ws.storeExtension(securityToken, ext);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            Collection<WikittyExtension> exts) {
        // TODO poussin 20100607 check security, mais qui a le droit ?
        return ws.storeExtension(securityToken, exts);
    }

    @Override
    public UpdateResponse storeExtension(String securityToken,
            WikittyTransaction transaction, Collection<WikittyExtension> exts) {
        // TODO poussin 20100607 check security, mais qui a le droit ?
        return ws.storeExtension(securityToken, transaction, exts);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken, String id) {
        // All people can read extension
        return ws.restoreExtension(securityToken, id);
    }

    @Override
    public WikittyExtension restoreExtension(String securityToken,
            WikittyTransaction transaction, String id) {
        // All people can read extension
        return ws.restoreExtension(securityToken, transaction, id);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, String name) {
        // All people can read extension
        return ws.restoreExtensionLastVersion(securityToken, name);
    }

    @Override
    public WikittyExtension restoreExtensionLastVersion(
            String securityToken, WikittyTransaction transaction, String name) {
        // All people can read extension
        return ws.restoreExtensionLastVersion(securityToken, transaction, name);
    }

    @Override
    public Wikitty restore(String securityToken, String id) {
        Wikitty result = null;
        if (canRead(securityToken, id)) {
            result = ws.restore(securityToken, id);
        } else {
            if (log.isDebugEnabled()) {
                    log.debug(_("user %s can't read object %s",
                    getUserId(securityToken), id));
            }
        }
        return result;
    }

    @Override
    public List<Wikitty> restore(String securityToken, List<String> ids) {
        List<String> authorizedIds = new LinkedList<String>(ids);
        for (Iterator<String> i=authorizedIds.iterator(); i.hasNext();) {
            String id = i.next();
            if (!canRead(securityToken, id)) {
                if (log.isDebugEnabled()) {
                    log.debug(_(
                            "user %s can't read object %s, remove it in restore list",
                            getUserId(securityToken), id));
                }
                i.remove();
            }
        }

        return ws.restore(securityToken, authorizedIds);
    }

    @Override
    public List<Wikitty> restore(String securityToken, WikittyTransaction transaction, List<String> ids) {
        List<String> authorizedIds = new LinkedList<String>(ids);
        for (Iterator<String> i=authorizedIds.iterator(); i.hasNext();) {
            String id = i.next();
            if (!canRead(securityToken, id)) {
                if (log.isDebugEnabled()) {
                    log.debug(_(
                            "user %s can't read object %s, remove it in restore list",
                            getUserId(securityToken), id));
                }
                i.remove();
            }
        }

        return ws.restore(securityToken, transaction, authorizedIds);
    }

    @Override
    public void delete(String securityToken, String id) {
        if (canDelete(securityToken, id)) {
            ws.delete(securityToken, id);
        }
    }

    @Override
    public void delete(String securityToken, Collection<String> ids) {
        for (String id : ids) {
            if (!canDelete(securityToken, id)) {
                throw new SecurityException(_("user %s can't delete object %s",
                        getUserId(securityToken), id));
            }
        }

        ws.delete(securityToken, ids);
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken, Criteria criteria) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByCriteria(securityToken, criteria);
        return result;
    }

    @Override
    public PagedResult<String> findAllByCriteria(String securityToken,
            WikittyTransaction transaction, Criteria criteria) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByCriteria(
                securityToken, transaction, criteria);
        return result;
    }

    @Override
    public Wikitty findByCriteria(String securityToken, Criteria criteria) {
        Wikitty result = ws.findByCriteria(securityToken, criteria);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public Wikitty findByCriteria(String securityToken, WikittyTransaction transaction, Criteria criteria) {
        Wikitty result = ws.findByCriteria(securityToken, transaction, criteria);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public void addLabel(String securityToken, String wikittyId, String label) {
        // TODO poussin 20100607 check security
        ws.addLabel(securityToken, wikittyId, label);
    }

    @Override
    public PagedResult<String> findAllByLabel(String securityToken,
            String label, int firstIndex, int endIndex) {
        // All people can read PagedResult that contains only id
        PagedResult<String> result = ws.findAllByLabel(
                securityToken, label, firstIndex, endIndex);
        return result;
    }

    @Override
    public Wikitty findByLabel(String securityToken, String label) {
        Wikitty result = ws.findByLabel(securityToken, label);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public Set<String> findAllAppliedLabels(String securityToken, String wikittyId) {
        Set<String> result = ws.findAllAppliedLabels(securityToken, wikittyId);
        return result;
    }

    @Override
    public Tree restoreTree(String securityToken, String wikittyId) {
        // FIXME poussin 20100607 check security
        return ws.restoreTree(securityToken, wikittyId);
    }

    @Override
    public Entry<TreeNode, Integer> restoreNode(
            String securityToken, String wikittyId, Criteria filter) {
        // FIXME poussin 20100607 check security
        return ws.restoreNode(securityToken, wikittyId, filter);
    }

    @Override
    public Map<TreeNode, Integer> restoreChildren(
            String securityToken, String wikittyId, Criteria filter) {
        // FIXME poussin 20100607 check security
        return ws.restoreChildren(securityToken, wikittyId, filter);
    }

    @Override
    public Wikitty restoreVersion(
            String securityToken, String wikittyId, String version) {
        Wikitty result = ws.restoreVersion(securityToken, wikittyId, version);
        if (!canRead(securityToken, result.getId())) {
            // user don't have correct right, return null
            result = null;
        }
        return result;
    }

    @Override
    public UpdateResponse syncEngin(String securityToken) {
        String userId = getUserId(securityToken);
        if (isAppAdmin(securityToken, userId)) {
            // seul les AppAdmin on le droit a cette method
            return ws.syncEngin(securityToken);
        } else {
            throw new SecurityException(_("user %s can't sync sear engine",
                    getUserId(securityToken)));
        }
    }


    //
    // Method helper to check right
    //

    /**
     * 
     * @param pagedResult
     * @return
     */
    protected PagedResult<Wikitty> checkPagedResult(PagedResult<Wikitty> pagedResult) {
        // TODO poussin 20100610 que faire
        // TODO  - parcourir tous les resultats pour retirer ceux auquel on a pas le droit
        // TODO  - lever une exception des qu'on trouve un element interdit

        return pagedResult;
    }

    /**
     * Recupere l'identifiant de l'utilisateur associe au securityToken
     *
     * @param securityToken
     * @return l'identifiant de l'utilisateur, ou null si le token est invalide
     */
    protected String getUserId(String securityToken) {
        String result = null;
        // recuperation de l'utilisateur associe au securityToken
        // le securityToken est aussi l'id de l'objet
        Wikitty securityTokenWikitty = ws.restore(securityToken, securityToken);
        if (securityTokenWikitty != null) {
            result = SecurityTokenHelper.getUser(securityTokenWikitty);
        }
        return result;
    }


    /**
     * Verifie que l'utilisateur est bien le proprietaire de l'objet
     *
     * @param userId
     * @param w
     * @return
     */
    protected boolean isOwner(String securityToken, String userId, Wikitty w) {
        boolean result = false;
        if (WikittyAuthorisationHelper.isExtension(w)) {
            String owner = WikittyAuthorisationHelper.getOwner(w);
            result = userId.equals(owner);
        }
        return result;
    }

    /**
     * verifie que l'utilisateur est dans la liste des admin
     *
     * @param userId
     * @param w
     * @return vrai si et seulement si l'utilisateur est dans la liste des
     * admin
     */
    protected boolean isAdmin(String securityToken, String userId, Wikitty w) {
        boolean result = isMember(
                securityToken, userId, w, WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_ADMIN);
        return result;
    }

    /**
     * verifie que l'utilisateur est dans la liste des writer
     *
     * @param userId
     * @param w
     * @return vrai si et seulement si l'utilisateur est dans la liste des
     * writers
     */
    protected boolean isWriter(String securityToken, String userId, Wikitty w) {
        boolean result = isMember(
                securityToken, userId, w, WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_WRITER);
        return result;
    }

    /**
     * Par defaut un objet est lisible par tous, sauf s'il a l'extension
     * d'autorisation et que la liste des readers existe et n'est pas vide
     *
     * @param userId
     * @param w
     * @return true si l'utilisateur est dans la liste des reader (ou que cette
     * liste n'existe pas ce qui indique que tout le monde est reader)
     */
    protected boolean isReader(String securityToken, String userId, Wikitty w) {
        boolean result = true;
        if (WikittyAuthorisationHelper.isExtension(w)) {
            Set<String> groupOrUser = WikittyAuthorisationHelper.getReader(w);
            if (groupOrUser == null || groupOrUser.size() == 0) {
                // il n'y a pas de reader sur l'objet actuel, il faut regarder
                // sur le parent s'il y en a
                String parentId = WikittyAuthorisationHelper.getParent(w);
                if (parentId != null) {
                    Wikitty parent = ws.restore(securityToken, parentId);
                    result = isReader(securityToken, userId, parent);
                }
            } else {
                // il y a des readers sur l'objet actuel, il faut donc checker
                // comme pour les autres droits en parent aussi les parents
                result = isMember(
                        securityToken, userId, w, WikittyAuthorisation.FIELD_WIKITTYAUTHORISATION_READER);
            }
        }
        return result;
    }

    /**
     * Verifie si l'utilisateur est considere comme un AppAdmin
     *
     * @param userId
     * @return
     */
    protected boolean isAppAdmin(String securityToken, String userId) {
        Wikitty group;
        if (appAdminGroupId == null) {
            // 1er fois, on le recherche
            group = ws.findByCriteria(securityToken, Search.query().eq(
                    WikittyGroup.FQ_FIELD_WIKITTYGROUP_NAME, WIKITTY_APPADMIN_GROUP_NAME).criteria());
            // group peut-etre null s'il n'existe pas
        } else {
            // on a deja fait la recherche precedement, on essaie de reutilise
            // le meme id
            group = ws.restore(securityToken, appAdminGroupId);
            // group peut-etre null, si entre temps un admin a supprime le group
        }
        if (group == null) {
            // il n'existe pas on le cree.
            WikittyGroup appAdminGroup = new WikittyGroupImpl();
            appAdminGroup.setName(WIKITTY_APPADMIN_GROUP_NAME);
        }
        // on garde l'id pour ne plus faire la recherche,
        // vu que le groupe doit etre unique cela ne pose pas de probleme
        appAdminGroupId = group.getId();

        Set<String> ids = WikittyGroupHelper.getMembers(group);
        boolean result = isMember(securityToken, userId, ids);
        return result;
    }

    /**
     * verifie qu'un utilisateur est membre d'un groupe passe en parametre via
     * l'arguement field
     *
     * @param userId
     * @param w
     * @param field must be WikittyAuthorisation field name: admin, writer, reader
     * @return
     */
    protected boolean isMember(
            String securityToken, String userId, Wikitty w, String field) {
        boolean result = false;
        if (WikittyAuthorisationHelper.isExtension(w)) {
            Set<String> groupOrUser = w.getFieldAsSet(
                    WikittyAuthorisation.EXT_WIKITTYAUTHORISATION,
                    field,
                    String.class);
            result = isMember(securityToken, userId, groupOrUser);
            if (!result) {
                // user don't have right on current object, check parent right
                String parentId = WikittyAuthorisationHelper.getParent(w);
                if (parentId != null) {
                    Wikitty parent = ws.restore(securityToken, parentId);
                    result = isMember(securityToken, userId, parent, field);
                }
            }
        }
        return result;
    }

    /**
     * Verifie recursivement si un utilisateur est dans un groupe qui peut etre
     * constitue d'autre groupe ou d'utilisateur
     *
     * @param userId l'utilisateur recherche
     * @param groupOrUser la liste des id d'utilisateurs ou d'autres groupes
     * @return vrai si userId est retrouve, false sinon
     */
    protected boolean isMember(
            String securityToken, String userId, Set<String> groupOrUser) {
        boolean result = false;
        if (groupOrUser != null) {
            for (String id : groupOrUser) {
                if (userId.equals(id)) {
                    result = true;
                    break;
                } else {
                    Wikitty w = ws.restore(securityToken, id);
                    if (WikittyGroupHelper.isExtension(w)) {
                        Set<String> members = WikittyGroupHelper.getMembers(w);
                        if (isMember(securityToken, userId, members)) {
                            result = true;
                            break;
                        }
                    }
                }
            }
        }
        // not found in groupOrUser
        return result;
    }

}
