/*
 * #%L
 * ToPIA :: Persistence
 * 
 * $Id: BeanTransformer.java 1964 2010-05-20 10:41:48Z fdesbois $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.4/topia-persistence/src/main/java/org/nuiton/topia/generator/BeanTransformer.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.generator;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.java.ObjectModelTransformerToJava;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelDependency;
import org.nuiton.eugene.models.object.ObjectModelInterface;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;
import org.nuiton.topia.persistence.TopiaEntity;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;





/**
 * BeanTransformer
 * <p/>
 * Created: 28 oct. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 * @author tchemit <tchemit@codelutin.com>
 * @version $Id: BeanTransformer.java 1964 2010-05-20 10:41:48Z fdesbois $
 * @plexus.component role="org.nuiton.eugene.Template" role-hint="org.nuiton.topia.generator.BeanTransformer"
 */
public class BeanTransformer extends ObjectModelTransformerToJava {

    private static final Log log = LogFactory.getLog(BeanTransformer.class);

    @Override
    public void transformFromClass(ObjectModelClass clazz) {
        if (!clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_BEAN) &&
                !clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DTO)) {
            return;
        }

        ObjectModelClass resultClass;
        boolean isSuperClassSet = false;
        if (clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_BEAN)) {
            resultClass = createAbstractClass(clazz.getName(), clazz.getPackageName());

            // Get name for Impl class
            String implQualifiedName = clazz.getQualifiedName() + "Impl";
            String resourceName = "/" + implQualifiedName.replaceAll("\\.", "/");

            // Do not generate Impl if operations exist : must have an Impl class in Resource created by developper
            boolean hasOperations = !clazz.getAllOtherOperations(true).isEmpty() || !clazz.getOperations().isEmpty();

            // Generate the Impl class if not already exist in classpath and no operation is defined in model
            if (getClass().getResource(resourceName) == null && !hasOperations) {
                String implName = clazz.getName() + "Impl";
                ObjectModelClass resultClassImpl = createClass(implName, clazz.getPackageName());
                // set the abstract resulClass as the resultClassImpl super class
                setSuperClass(resultClassImpl, resultClass.getQualifiedName());
                isSuperClassSet = true;
            }
        } else {
            resultClass = createClass(clazz.getName(), clazz.getPackageName());
        }

        List<ObjectModelAttribute> attributes = (List<ObjectModelAttribute>) clazz.getAttributes();

        createForDTO(resultClass, clazz, attributes);

        // Set superclass
        if (!isSuperClassSet) {
            Iterator<ObjectModelClass> j = clazz.getSuperclasses().iterator();
            if (j.hasNext()) {
                ObjectModelClass p = j.next();
                String qualifiedName = p.getQualifiedName();
                // We want to set the inheritance to the implementation class of the father
                // Ex for model : A -> B (a inherits B) we want : A -> BImpl -> B
                if (clazz.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_BEAN)) {
                    qualifiedName = p.getQualifiedName() + "Impl";
                }
                setSuperClass(resultClass, qualifiedName);
            }
        }

        // Add interfaces from inputModel
        for (ObjectModelInterface parentInterface : clazz.getInterfaces()) {
            addInterface(resultClass, parentInterface.getQualifiedName());
        }

        // Default constructor
        ObjectModelOperation constructor = addConstructor(resultClass, ObjectModelModifier.PUBLIC);

        createListeners(resultClass, clazz);

        boolean hasEntity = false;
        boolean hasMultipleAttribute = false;
        String toStringAppend = ""; // Append pour la méthode toString()
        String initTabs = ""; // initialisation des tableaux dans le constructeur

        // Add attributes with getter/setter
        for (ObjectModelAttribute attr : attributes) {

            if (attr.isNavigable() || attr.hasAssociationClass()) {
                String attrType = attr.getType();
                String simpleType = GeneratorUtil.getSimpleName(attrType);
                String attrName = attr.getName();
                String attrNameCapitalized = StringUtils.capitalize(attrName);

                // multiple attribute
                if (attr.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ARRAY)) {

                    int maxSize = attr.getMaxMultiplicity();
                    int maxSizeMoinsUn = maxSize - 1;

                    initTabs += "\n\tthis." + attrName + " = new " + attrType + "[" + maxSize + "];";

                    // Set Value
                    ObjectModelOperation setValue = addOperation(resultClass, "set" + attrNameCapitalized,
                            "void", ObjectModelModifier.PUBLIC);
                    addParameter(setValue, "int", "index");
                    addParameter(setValue, attrType, "value");
                    addException(setValue, "java.lang.ArrayIndexOutOfBoundsException");
                    setOperationBody(setValue, ""
                            +"\n"
+"                                if (index >= "+maxSize+" || index < 0) {\n"
+"                                    throw new ArrayIndexOutOfBoundsException(\"Wrong index [\" + index + \"] for array "+attrName+",\" +\n"
+"                                            \"index must be between 0 and "+maxSizeMoinsUn+"\");\n"
+"                                }\n"
+"                                "+simpleType+"[] oldValue = get"+attrNameCapitalized+"();\n"
+"                                this."+attrName+"[index] = value;\n"
+"                                firePropertyChange(\""+attrName+"\", oldValue, this."+attrName+");\n"
+"                            "
                    );

                    // Get Value
                    ObjectModelOperation getValue = addOperation(resultClass, "get" + attrNameCapitalized,
                            attrType, ObjectModelModifier.PUBLIC);
                    addParameter(getValue, "int", "index");
                    addException(setValue, "java.lang.ArrayIndexOutOfBoundsException");
                    setOperationBody(getValue, ""
                            +"\n"
+"                                if (index >= "+maxSize+" || index < 0) {\n"
+"                                    throw new ArrayIndexOutOfBoundsException(\"Wrong index [\" + index + \"] for array "+attrName+",\" +\n"
+"                                            \"index must be between 0 and "+maxSizeMoinsUn+"\");\n"
+"                                }\n"
+"                                return this."+attrName+"[index];\n"
+"                            "
                    );

                    attrType += "[]";
                    simpleType = GeneratorUtil.getSimpleName(attrType);
                } else if (GeneratorUtil.isNMultiplicity(attr)) {
                    hasMultipleAttribute = true;

                    // Add getChild
                    ObjectModelOperation getChild = addOperation(resultClass, "get" + attrNameCapitalized,
                            attrType, ObjectModelModifier.PUBLIC);
                    addParameter(getChild, "int", "index");
                    setOperationBody(getChild, ""
                            +"\n"
+"                                "+simpleType+" o = getChild("+attrName+", index);\n"
+"                                return o;\n"
+"                            "
                    );

                    // Add getEntity
                    ObjectModelClass attrEntity = null;
                    if (getModel().hasClass(attr.getType())) {
                        attrEntity = getModel().getClass(attr.getType());
                    }
                    boolean isEntity = (attrEntity != null && attrEntity.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY));

                    if (isEntity) {
                        hasEntity = true;
                        ObjectModelOperation getChildEntity = addOperation(resultClass, "get" + attrNameCapitalized,
                                attrType, ObjectModelModifier.PUBLIC);
                        addParameter(getChildEntity, String.class.getName(), "topiaId");
                        setOperationBody(getChildEntity, ""
                                +"\n"
+"                                    "+simpleType+" o = getEntity("+attrName+", topiaId);\n"
+"                                    return o;\n"
+"                                "
                        );
                    }

                    // Add addChild
                    ObjectModelOperation addChild = addOperation(resultClass, "add" + attrNameCapitalized,
                            attrType, ObjectModelModifier.PUBLIC);
                    addParameter(addChild, attrType, attrName);
                    setOperationBody(addChild, ""

                            +"\n"
+"                                get"+attrNameCapitalized+"().add("+attrName+");\n"
+"                                firePropertyChange(\""+attrName+"\", null, "+attrName+");\n"
+"                                return "+attrName+";\n"
+"                            "
                    );

                    // Add removeChild
                    ObjectModelOperation removeChild = addOperation(resultClass, "remove" + attrNameCapitalized,
                            "boolean", ObjectModelModifier.PUBLIC);
                    addParameter(removeChild, attrType, attrName);
                    setOperationBody(removeChild, ""

                            +"\n"
+"                                boolean  removed = get"+attrNameCapitalized+"().remove("+attrName+");\n"
+"                                if (removed) {\n"
+"                                    firePropertyChange(\""+attrName+"\", "+attrName+", null);\n"
+"                                }\n"
+"                                return removed;\n"
+"                            "
                    );

                    // Change type for Multiple attribute
                    if (attr.isOrdered()) {
                        attrType = List.class.getName() + "<" + attrType + ">";
                    } else {
                        attrType = Collection.class.getName() + "<" + attrType + ">";
                    }
                    simpleType = GeneratorUtil.getSimpleName(attrType);
                } // end multiple attribute

                if (attr.hasAssociationClass()) {
                    String assocAttrName = TopiaGeneratorUtil.getAssocAttrName(attr);
                    attrName = GeneratorUtil.toLowerCaseFirstLetter(assocAttrName);
                    attrType = attr.getAssociationClass().getName();
                }

                // Add attribute to the class
                String visibility = attr.getVisibility();
                addAttribute(resultClass, attrName, attrType, "", ObjectModelModifier.toValue(visibility));

                // Add getter operation
                ObjectModelOperation getter = addOperation(resultClass, "get" + attrNameCapitalized, attrType,
                        ObjectModelModifier.PUBLIC);
                setOperationBody(getter, ""
                        +"\n"
+"                            return this."+attrName+";\n"
+"                        "
                );

                // Add setter operation
                ObjectModelOperation setter = addOperation(resultClass, "set" + attrNameCapitalized, "void",
                        ObjectModelModifier.PUBLIC);
                addParameter(setter, attrType, "newValue");
                setOperationBody(setter, ""
                        +"\n"
+"                            "+simpleType+" oldValue = get"+attrNameCapitalized+"();\n"
+"                            this."+attrName+" = newValue;\n"
+"                            firePropertyChange(\""+attrName+"\", oldValue, newValue);\n"
+"                        "
                );

                // toString append for toString method
                toStringAppend += "\n\t\t.append(\"" + attrName + "\", this." + attrName + ")";

            }
        }

        // Add helper operations
        if (hasMultipleAttribute) {
            ObjectModelOperation getChild = addOperation(resultClass, "getChild", "<T> T",
                    ObjectModelModifier.PROTECTED);
            addParameter(getChild, "java.util.Collection<T>", "childs");
            addParameter(getChild, "int", "index");
            setOperationBody(getChild, ""
                    +"\n"
+"                        if (childs != null) {\n"
+"                            int i = 0;\n"
+"                            for (T o : childs) {\n"
+"                                if (index == i) {\n"
+"                                    return o;\n"
+"                                }\n"
+"                                i++;\n"
+"                            }\n"
+"                        }\n"
+"                        return null;\n"
+"                    "
            );
        }

        if (hasEntity) {
            //FIXME-TC20091219 : there is some bugs in import managers, ...
            // due to I think cache extension, sometimes, there is no more imports
            addImport(resultClass, TopiaEntity.class);
            ObjectModelOperation getEntity = addOperation(resultClass, "getEntity",
                    "<T extends org.nuiton.topia.persistence.TopiaEntity> T", ObjectModelModifier.PROTECTED);
            addParameter(getEntity, "java.util.Collection<T>", "childs");
            addParameter(getEntity, "java.lang.String", "topiaId");
            setOperationBody(getEntity, ""
                    +"\n"
+"                        if (childs != null) {\n"
+"                            for (T o : childs) {\n"
+"                                if (topiaId.equals(o.getTopiaId())) {\n"
+"                                    return o;\n"
+"                                }\n"
+"                            }\n"
+"                        }\n"
+"                        return null;\n"
+"                    "
            );
        }

        // Set body for default constructor
        setOperationBody(constructor, ""
                +"\n"
+"                    pcs = new PropertyChangeSupport(this);"+initTabs+"\n"
+"                "
        );

        // Add operations
        for (ObjectModelOperation op : clazz.getOperations()) {
            String visibility = op.getVisibility();
            ObjectModelOperation resultOperation = addOperation(resultClass, op.getName(), op.getReturnType(),
                    ObjectModelModifier.toValue(visibility), ObjectModelModifier.ABSTRACT);

            for (ObjectModelParameter param : op.getParameters()) {
                addParameter(resultOperation, param.getType(), param.getName());
            }

            for (String exception : op.getExceptions()) {
                addException(resultOperation, exception);
            }
        }

        // Add toString operation        
        ObjectModelOperation toString = addOperation(resultClass, "toString", "java.lang.String",
                ObjectModelModifier.PUBLIC); // FIXME manque Override
        addImport(resultClass, "org.apache.commons.lang.builder.ToStringBuilder");
        setOperationBody(toString, ""
                +"\n"
+"                    String result = new ToStringBuilder(this)"+toStringAppend+".\n"
+"                            toString();\n"
+"                    return result;\n"
+"                "
        );

    }

    private void createForDTO(ObjectModelClass resultClass, ObjectModelClass inputClass, List<ObjectModelAttribute> attributes) {

        // Add Serializable implements for DTO generation
        if (!inputClass.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_DTO)) {
            return;

        }

        addInterface(resultClass, "java.io.Serializable");
        String svUID = TopiaGeneratorUtil.findTagValue("dto-serialVersionUID", inputClass, getModel());
        if (svUID != null) {
            addConstant(resultClass, "serialVersionUID", "long", svUID, ObjectModelModifier.PUBLIC);
        }

        for (ObjectModelDependency dependency : inputClass.getDependencies()) {
            ObjectModelClass supplier = (ObjectModelClass) dependency.getSupplier();

            // ENTITY dependency
            // Copy all primitives attributes from the Entity (supplier) to the DTO
            // Prepare a list to future generation of all object generated attributes
            if (supplier.hasStereotype(TopiaGeneratorUtil.STEREOTYPE_ENTITY)) {
                if (log.isDebugEnabled()) {
                    log.debug("Create primitive and date fields in DTO from Entity : "
                            + supplier.getQualifiedName());
                }
                for (ObjectModelAttribute attr : supplier.getAttributes()) {
                    if (TopiaGeneratorUtil.isPrimitiveType(attr) || TopiaGeneratorUtil.isDateType(attr)) {
                        attributes.add(attr);
                    }
                }
            }
        }
    }

    protected void createListeners(ObjectModelClass resultClass, ObjectModelClass inputClass) {

        addAttribute(resultClass, "pcs", "java.beans.PropertyChangeSupport", "",
                ObjectModelModifier.PROTECTED, ObjectModelModifier.FINAL);

        // Add PropertyListener
        String propType = "java.beans.PropertyChangeListener";
        String strType = String.class.getName();
        String objectType = Object.class.getName();

        ObjectModelOperation addPropertyChangeListener = addOperation(resultClass,
                "addPropertyChangeListener", "void", ObjectModelModifier.PUBLIC);
        addParameter(addPropertyChangeListener, propType, "listener");
        setOperationBody(addPropertyChangeListener, ""
                +"\n"
+"                    pcs.addPropertyChangeListener(listener);\n"
+"                "
        );

        ObjectModelOperation addPropertyChangeListenerPlus = addOperation(resultClass,
                "addPropertyChangeListener", "void", ObjectModelModifier.PUBLIC);
        addParameter(addPropertyChangeListenerPlus, strType, "propertyName");
        addParameter(addPropertyChangeListenerPlus, propType, "listener");
        setOperationBody(addPropertyChangeListenerPlus, ""
                +"\n"
+"                    pcs.addPropertyChangeListener(propertyName, listener);\n"
+"                "
        );

        ObjectModelOperation removePropertyChangeListener = addOperation(resultClass,
                "removePropertyChangeListener", "void", ObjectModelModifier.PUBLIC);
        addParameter(removePropertyChangeListener, propType, "listener");
        setOperationBody(removePropertyChangeListener, ""
                +"\n"
+"                    pcs.removePropertyChangeListener(listener);\n"
+"                "
        );

        ObjectModelOperation removePropertyChangeListenerPlus = addOperation(resultClass,
                "removePropertyChangeListener", "void", ObjectModelModifier.PUBLIC);
        addParameter(removePropertyChangeListenerPlus, strType, "propertyName");
        addParameter(removePropertyChangeListenerPlus, propType, "listener");
        setOperationBody(removePropertyChangeListenerPlus, ""
                +"\n"
+"                    pcs.removePropertyChangeListener(propertyName, listener);\n"
+"                "
        );

        ObjectModelOperation firePropertyChange = addOperation(resultClass,
                "firePropertyChange", "void", ObjectModelModifier.PROTECTED);
        addParameter(firePropertyChange, strType, "propertyName");
        addParameter(firePropertyChange, objectType, "oldValue");
        addParameter(firePropertyChange, objectType, "newValue");
        setOperationBody(firePropertyChange, ""
                +"\n"
+"                    pcs.firePropertyChange(propertyName, oldValue, newValue);\n"
+"                "
        );
    }


}
