package org.nuiton.jredmine;

import org.nuiton.jredmine.model.*;


/**
 * Contract of all redmine operations that needs a login to be performed.
 * <p/>
 * A default implementation is offered in
 * {@link org.nuiton.jredmine.DefaultRedmineService}.
 * <p/>
 * Created: 2 janv. 2010
 *
 * @author Tony Chemit <chemit@codelutin.com> Copyright Code Lutin
 * @version $Revision: 66 $
 *          <p/>
 *          Mise a jour: $Date: 2010-01-03 11:49:37 +0100 (dim., 03 janv. 2010) $ par :
 *          $Author: tchemit $
 * @see RedmineAnonymousService
 * @see RedmineService
 * @since 1.0.3
 */
public interface RedmineLogguedService {

    /**
     * Obtain for the loggued user, all projets where he belongs.
     * <p/>
     * <b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @return all the projects belonged by the loggued user
     * @throws RedmineServiceException if any pb while retriving datas
     * @see org.nuiton.jredmine.model.Project
     * @since 1.0.3
     */
    Project[] getUserProjects() throws RedmineServiceException;

    /**
     * Add a version for a given project.
     * <p/>
     * b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @param projectName the name of the project
     * @param version     the version to add
     * @return the added version
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Version
     */
    Version addVersion(String projectName, Version version) throws RedmineServiceException;

    /**
     * Update a version for a given project.
     * <p/>
     * b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @param projectName the name of the project
     * @param version     the version to update
     * @return the updated version
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Version
     */
    Version updateVersion(String projectName, Version version) throws RedmineServiceException;

    /**
     * Prepare a new version (create it or update it).
     * <p/>
     * If the {@code oldVersionName} is given, then all issues unclosed from this
     * old version will be move to the new version.
     * <p/>
     * b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @param projectName    the name of the project
     * @param oldVersionName the name of the old version (optional)
     * @param newVersion     the newVersion to create or update
     * @return the created version
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Version
     */
    Version nextVersion(String projectName, String oldVersionName, Version newVersion) throws RedmineServiceException;

    /**
     * Add a news for a given project.
     * <p/>
     * b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @param projectName the name of the project
     * @param news        the news to add
     * @return the added news.
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see News
     */
    News addNews(String projectName, News news) throws RedmineServiceException;

    /**
     * Add a attachment for a given version of a given project.
     * <p/>
     * b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @param projectName the name of the project
     * @param versionName the name of the version
     * @param attachement the attachment to add
     * @return the added attachment
     * @throws RedmineServiceException if any pb while sending and retriving datas to redmine server
     * @see Attachment
     */
    Attachment addAttachment(String projectName, String versionName, Attachment attachement) throws RedmineServiceException;

    /**
     * Add a new time entry to the given issue for the given project and
     * return the updated time entry.
     * <p/>
     * b>Note:</b> This method requires to be loggued on redmine server.
     *
     * @param projectName the name of the project
     * @param issueId     the id of the issue to update
     * @param entry       time entry to add
     * @return the created time entry
     * @throws RedmineServiceException if any pb while sending or retrieving datas to redmine server
     * @see Issue
     * @see TimeEntry
     * @since 1.0.3
     */
    TimeEntry addIssueTimeEntry(String projectName, String issueId, TimeEntry entry) throws RedmineServiceException;
}
