package org.nuiton.jaxx.application.swing.util;

/*
 * #%L
 * JAXX :: Application Swing
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.jdesktop.swingx.JXErrorPane;
import org.jdesktop.swingx.error.ErrorInfo;
import org.jdesktop.swingx.error.ErrorReporter;
import org.nuiton.csv.ImportRuntimeException;
import org.nuiton.jaxx.application.ApplicationBusinessException;
import org.nuiton.jaxx.application.swing.ApplicationUIContext;
import org.nuiton.jaxx.application.swing.action.ApplicationActionUI;

import javax.swing.JOptionPane;

import static org.nuiton.i18n.I18n.t;

/**
 * Helper to display errors.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.8
 */
public class ApplicationErrorHelper implements ErrorReporter {

    protected final ApplicationUIContext context;

    public ApplicationErrorHelper(ApplicationUIContext context) {
        this.context = context;
    }

    /**
     * Display a user friendly error frame.
     *
     * @param message message for user
     * @param cause   exception cause
     */
    public void showErrorDialog(String message, Throwable cause) {

        if (cause == null) {
            JOptionPane.showMessageDialog(context.getMainUI(), "<html><body>" + message + "</body></html>",
                                          t("jaxx.application.error.ui.business.error"),
                                          JOptionPane.ERROR_MESSAGE);
        } else if (cause instanceof ApplicationBusinessException) {
            JOptionPane.showMessageDialog(context.getMainUI(), "<html><body>" + cause.getMessage() + "</body></html>",
                                          t("jaxx.application.error.ui.business.error"),
                                          JOptionPane.ERROR_MESSAGE);
        } else if (cause instanceof ImportRuntimeException) {
            JOptionPane.showMessageDialog(context.getMainUI(), cause.getMessage(),
                                          t("jaxx.application.error.ui.business.error"),
                                          JOptionPane.ERROR_MESSAGE);
        } else {

            JXErrorPane pane = new JXErrorPane();
            ErrorInfo info = new ErrorInfo(t("jaxx.application.error.ui.other.error"),
                                           t("jaxx.application.error.errorpane.htmlmessage", message), null, null,
                                           cause, null, null);
            pane.setErrorInfo(info);
            pane.setErrorReporter(this);
            ApplicationActionUI actionUI = context.getActionUI();
            if (actionUI != null) {
                actionUI.close();
            }
            JXErrorPane.showDialog(context.getMainUI(), pane);
        }

    }

    /**
     * Display a user friendly error frame.
     *
     * @param message message for user
     */
    public void showErrorDialog(String message) {
        showErrorDialog(message, null);
    }

    @Override
    public void reportError(ErrorInfo errorInfo) throws NullPointerException {

        showErrorDialog(errorInfo.getBasicErrorMessage(), errorInfo.getErrorException());
    }

    /**
     * Display a user friendly warning frame.
     *
     * @param message message for user
     */
    public void showWarningDialog(String message) {

        JOptionPane.showMessageDialog(context.getMainUI(), "<html><body>" + message + "</body></html>",
                                      t("jaxx.application.error.ui.business.warning"),
                                      JOptionPane.WARNING_MESSAGE);

    }

}
