/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: ParserJspMojo.java 1928 2011-05-13 09:59:47Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.4.1/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/parser/impl/ParserJspMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.i18n.plugin.parser.impl;

import org.apache.commons.lang.StringUtils;
import org.nuiton.i18n.plugin.parser.I18nSourceEntry;
import org.nuiton.i18n.plugin.parser.SourceEntry;
import org.nuiton.io.FileUpdater;
import org.nuiton.io.MirroredFileUpdater;
import org.nuiton.plugin.PluginHelper;
import org.nuiton.processor.ProcessorUtil;
import org.nuiton.processor.filters.DefaultFilter;
import org.nuiton.processor.filters.Filter;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.xpath.XPath;
import java.io.File;
import java.io.IOException;

/**
 * Find i18n keys from jsp files.
 * <p/>
 * <b>Note: </b> this goal must always be invoked before the {@code process-resources}
 * phase, otherwise all files will be considered as uptodate.
 *
 * @author tchemit <chemit@codelutin.com>
 * @goal parserJsp
 * @phase generate-resources
 * @since 2.0
 */
public class ParserJspMojo extends AbstractParserXmlMojo {

    /**
     * Root directory of the default entry.
     *
     * @parameter expression="${i18n.defaultBasedir}" default-value="${basedir}/src/main/webapp"
     * @required
     */
    protected File defaultBasedir;

    /**
     * Default included files to process (ant-like expression).
     *
     * @parameter expression="${i18n.defaultIncludes}" default-value="**\/**.jsp"
     */
    protected String defaultIncludes;

    /**
     * Defines the core rules file used to detect i18n keys in jsp files.
     * <p/>
     * <b>Note :</b> If you do not want to use it, set it to empty and fill the
     * {@link #userRulesFiles} parameter.
     *
     * @parameter expression="${i18n.coreRuleFile}"
     * @since 2.0
     */
    protected String coreRuleFile;

    /**
     * Defines the file name of the getter where to put detected i18n keys
     * while getter phase.
     *
     * @parameter expression="${i18n.outputGetter}" default-value="jsp.getter"
     * @since 2.0
     */
    protected String outputGetter;

    /**
     * Where to generated temporary processed files.
     *
     * @parameter expression="${i18n.workdir}" default-value="${basedir}/target/i18n-workdir"
     * @since 2.0
     */
    protected File workdir;

    @Override
    public String[] getDefaultIncludes() {
        return new String[]{defaultIncludes};
    }

    @Override
    public String[] getDefaultExcludes() {
        return I18nSourceEntry.EMPTY_STRING_ARRAY;
    }

    @Override
    public File getDefaultBasedir() {
        return defaultBasedir;
    }

    MirroredFileUpdater entryUpdater;

    @Override
    protected boolean onEnterEntry(I18nSourceEntry entry) {
        boolean b = super.onEnterEntry(entry);
        if (!b) {

            // no skipped entry
            // keep the file updater
            entryUpdater = (MirroredFileUpdater) entry.getUpdater();
        }
        return b;
    }

    @Override
    public FileUpdater newFileUpdater(SourceEntry entry) {
        return new MirroredFileUpdater("", "", entry.getBasedir(), workdir) {

            @Override
            public File getMirrorFile(File f) {
                String file =
                        f.getAbsolutePath().substring(prefixSourceDirecotory);
                return new File(destinationDirectory + File.separator + file + "~");
            }
        };
    }

    @Override
    protected String getOutGetter() {
        return outputGetter;
    }

    @Override
    protected String getCoreRuleFile() {
        return coreRuleFile;
    }

    protected XmlFileParser newXmlFileParser(final XPath xpath,
                                             final DocumentBuilder builder) {

        return new XmlFileParser(getLog(),
                                 encoding,
                                 oldParser,
                                 showTouchedFiles,
                                 rules,
                                 xpath,
                                 builder,
                                 namespaces,
                                 isVerbose()) {

            @Override
            public File prepareFile(File file) throws IOException {

                // clean the jsp to make it xml

                File result = entryUpdater.getMirrorFile(file);
                createDirectoryIfNecessary(result.getParentFile());
                JspFileProcessor processor = new JspFileProcessor();
                processor.process(file, result, getEncoding());
                String resultFileContent =
                        PluginHelper.readAsString(result, getEncoding());
                resultFileContent = resultFileContent.trim();

                if (StringUtils.isBlank(resultFileContent)) {

                    // nothing to scan inside this file
                    result = null;
                } else {

                    if (resultFileContent.startsWith("<html") ||
                        resultFileContent.startsWith("<HTML") ||
                        resultFileContent.startsWith("<?xml")
                            ) {

                        // document should be well-formed...

                    } else {
                        // to be sure, document is well formed, add a start and end tag
                        resultFileContent = "<div>\n" + resultFileContent + "\n</div>";

                        PluginHelper.writeString(result, resultFileContent, getEncoding());
                    }

                }

                return result;
            }

            @Override
            public String extract(String i18nString) {
                String s = null;
                if (!StringUtils.isEmpty(i18nString.trim())) {
                    s = i18nString.trim();
                }
                if (getLog().isDebugEnabled()) {
                    getLog().debug(i18nString + " = " + s);
                }
                return s;
            }

        };
    }

    /**
     * To transform jsp files to valid xml files.
     * <p/>
     * Says :
     * <ul>
     * <li>Remove jsp directive</li>
     * <li>Remove jsp comment</li>
     * <li>Remove any tags in attributes (used for example in struts)</li>
     * <li>? Other thing</li>
     * </ul>
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.0
     */
    public static class JspFileProcessor extends ProcessorHelper.AbstractParserProcessor {

        protected ProcessorHelper.FragmentRemover remover1 =
                new ProcessorHelper.FragmentRemover(
                        "<" + "%" + "-" + "-",
                        "-" + "-" + "%" + ">"
                );

        protected ProcessorHelper.FragmentRemover remover2 =
                new ProcessorHelper.FragmentRemover(
                        "<" + "%",
                        "%" + ">"
                );

        protected ProcessorHelper.FragmentRemover remover3 =
                new ProcessorHelper.FragmentRemover(
                        "<" + "!" + "-" + "-",
                        "-" + "-" + ">"
                );

        public JspFileProcessor() {
            setInputFilter(
                    new Filter[]{
                            remover1,
                            remover2,
                            remover3,
                            new JspAttributeWithTagFilter()
                    }
            );
        }

        /**
         * @param filein   the source file to process
         * @param fileout  the output file to generate
         * @param encoding encoding used to read and write files
         * @throws IOException if any io problems while processing
         * @since 1.0.4
         */
        @Override
        public void process(File filein,
                            File fileout,
                            String encoding) throws IOException {
            ProcessorUtil.doProcess(this, filein, fileout, encoding);
        }

        /**
         * To remove in attributes any sub tags inside it (used for example in
         * struts) from jsp files.
         *
         * @author tchemit <chemit@codelutin.com>
         * @since 2.0
         */
        public static class JspAttributeWithTagFilter extends DefaultFilter {

            private String header = "=" + "\"" + "<";

            private String footer = "\"" + "/" + ">" + "\"";

            @Override
            protected String performInFilter(String ch) {
                return "=\"\"";
            }

            @Override
            protected String performOutFilter(String ch) {
                return ch;
            }

            protected String getHeader() {
                return header;
            }

            protected String getFooter() {
                return footer;
            }
        }
    }
}
