/*
 * #%L
 * I18n :: Maven Plugin
 * 
 * $Id: AbstractMakeI18nBundleMojo.java 1917 2011-05-10 20:16:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/i18n/tags/i18n-2.4.1/maven-i18n-plugin/src/main/java/org/nuiton/i18n/plugin/bundle/AbstractMakeI18nBundleMojo.java $
 * %%
 * Copyright (C) 2007 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.i18n.plugin.bundle;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.MojoFailureException;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.SortedSet;

/**
 * Common mojo to all final bundle maker.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.0
 */
public abstract class AbstractMakeI18nBundleMojo extends AbstractI18nBundleMojo {

    /**
     * Encoding used to load any i18n property files.
     * <p/>
     * If not defined, will use the {@link #encoding} parameter.
     *
     * @parameter expression="${i18n.bundleInputEncoding}"
     * @since 2.4
     */
    protected String bundleInputEncoding;

    /**
     * Encoding used to write any i18n property files.
     * <p/>
     * If not defined, will use the {@link #encoding} parameter.
     *
     * @parameter expression="${i18n.bundleOutputEncoding}"
     * @since 2.4
     */
    protected String bundleOutputEncoding;

    /**
     * Root directory where to generate aggregated bundles (this directory will
     * be added as resources of the project).
     *
     * @parameter expression="${i18n.bundleOutputDir}" default-value="${basedir}/target/generated-sources/resources"
     * @required
     * @since 1.0.0
     */
    protected File bundleOutputDir;

    /**
     * Package name  of the generate aggregated bundles.
     * <p/>
     * <strong>Note:</strong> By default we use the <code>META-INF</code> package
     * since it is the favorite package of <code>I18n</code> runtime initializer.
     * <p/>
     * The package name is dotted as it will be stored as folder like in Java
     * language.
     * <p/>
     * Example :
     * <pre>
     *     package name : foo.bar
     *     directory    : foo/bar
     * </pre>
     *
     * @parameter expression="${i18n.bundleOutputPackage}" default-value="META-INF"
     * @required
     * @since 2.3.2
     */
    protected String bundleOutputPackage;

    /**
     * Name of the bundle to generate.
     *
     * @parameter expression="${i18n.bundleOutputName}" default-value="${project.artifactId}-i18n"
     * @required
     * @since 1.0.2
     */
    protected String bundleOutputName;

    /**
     * A flag to generate a bundle with the first locale defined as a default
     * bundle (with no locale specialization).
     *
     * @parameter expression="${i18n.generateDefaultLocale}" default-value="false"
     * @since 2.1
     */
    protected boolean generateDefaultLocale;

    /**
     * A flag to check that bundles are complete (no missing i18n translations).
     * <p/>
     * <b>Note :</b> This behaviour will be activated is {@link #failsIfWarning} is on.
     *
     * @parameter expression="${i18n.checkBundle}" default-value="true"
     * @since 1.0.0
     */
    protected boolean checkBundle;

    /**
     * A flag to show missing i18n translation.
     * <p/>
     * <b>Note :</b> Need the {@link #checkBundle} to be activated).
     *
     * @parameter expression="${i18n.showEmpty}" default-value="false"
     * @since 1.0.0
     */
    protected boolean showEmpty;

    /**
     * A flag to make the build fails if there is some warnings while generating
     * bundle, says when it misses some translations.
     * <p/>
     * <b>Note :</b> This parameter should be used in a release profile to
     * ensure bundles are complete.
     *
     * @parameter expression="${i18n.failsIfWarning}" default-value="false"
     * @since 2.0
     */
    protected boolean failsIfWarning;

    /** to keep all none translated i18n keys by locale. */
    protected Map<Locale, SortedSet<String>> unsafeMapping;

    /**
     * The definitive directory where to generate the bundles (includes the
     * package of bunlde).
     *
     * @since 2.3.2
     */
    protected File outputFolder;

    @Override
    public void init() throws Exception {
        super.init();

        if (failsIfWarning) {

            // check bundle if wants to fail on unsafe bundles
            checkBundle = true;

            unsafeMapping = new HashMap<Locale, SortedSet<String>>();
        } else {
            unsafeMapping = null;
        }

        // get the definitive folder where to generate bundles (including
        // bundle package)

        outputFolder = getBundleOutputFolder();

        if (isVerbose()) {
            getLog().info("Will generates bundles in " + outputFolder);
        }
        createDirectoryIfNecessary(outputFolder);

        if (StringUtils.isEmpty(bundleInputEncoding)) {

            // use the default encoding
            bundleInputEncoding = getEncoding();
            if (getLog().isDebugEnabled()) {
                getLog().debug("Use as input encoding the default one : " +
                               bundleInputEncoding);
            }
        }

        if (StringUtils.isEmpty(bundleOutputEncoding)) {

            // use the default encoding
            bundleOutputEncoding = getEncoding();

            if (getLog().isDebugEnabled()) {
                getLog().debug("Use as output encoding the default one : " +
                               bundleOutputEncoding);
            }
        }
    }

    protected void failsIfWarning() throws MojoFailureException {
        if (!failsIfWarning) {

            // no check
            return;
        }

        if (unsafeMapping != null && !unsafeMapping.isEmpty()) {

            // there is at least one not complete bundle, faisl the build
            throw new MojoFailureException(
                    "Bundles for locale(s) " + unsafeMapping.keySet() +
                    " are not complete. Use the -Di18n.showEmpty to see " +
                    "missing translations.");
        }
    }

    /**
     * Gets the bundle file for the given parameters.
     *
     * @param root       the root directory where bundles are stored
     * @param artifactId the artifactId (says the prefix of bundle)
     * @param locale     the locale used in bundle ({@code null} means no locale specialized)
     * @param create     a flag to create the file if none existing
     * @return the bundle file
     * @throws IOException if any IO problem while creating it (if needed).
     * @since 2.1
     */
    protected abstract File getBundleFile(File root,
                                          String artifactId,
                                          Locale locale,
                                          boolean create) throws IOException;

    /**
     * Generates the default bundle, says the bundle with no locale specialized.
     * <p/>
     * This bundle is a copy of the bundle of the first locale (which in fact
     * is considered as the main locale).
     *
     * @throws IOException if any IO problem while the copy.
     * @since 2.1
     */
    protected void generateDefaultBundle() throws IOException {

        File bundleFirstLocale = getBundleFile(outputFolder,
                                               bundleOutputName,
                                               locales[0],
                                               false
        );

        File bundleWithoutLocale = getBundleFile(outputFolder,
                                                 bundleOutputName,
                                                 null,
                                                 false
        );

        if (!isSilent()) {
            getLog().info("Generate default bundle at " + bundleWithoutLocale);
        }

        FileUtils.copyFile(bundleFirstLocale, bundleWithoutLocale);
    }

    protected File getBundleOutputFolder() {
        File result = bundleOutputDir;
        if (StringUtils.isNotEmpty(bundleOutputPackage)) {
            String[] paths = bundleOutputPackage.split("\\.");
            for (String path : paths) {
                result = new File(result, path);
            }
        }
        return result;
    }

    public String getBundleOutputEncoding() {
        return bundleOutputEncoding;
    }

    public String getBundleInputEncoding() {
        return bundleInputEncoding;
    }

}
