package fr.reseaumexico.model.parser;

/*
 * #%L
 * JMexico :: Model
 * $Id: ScenarioXmlParser.java 86 2012-10-07 16:22:58Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/jmexico/tags/jmexico-0.7/jmexico-model/src/main/java/fr/reseaumexico/model/parser/ScenarioXmlParser.java $
 * %%
 * Copyright (C) 2011 - 2012 Réseau Mexico, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import fr.reseaumexico.model.Factor;
import fr.reseaumexico.model.Scenario;
import fr.reseaumexico.model.ScenarioImpl;
import fr.reseaumexico.model.ValueType;
import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;

import java.io.IOException;
import java.text.ParseException;
import java.util.Map;

import static fr.reseaumexico.model.MexicoXmlConstant.FACTOR;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO_FACTOR_VALUES;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO_NAME;
import static fr.reseaumexico.model.MexicoXmlConstant.SCENARIO_ORDER_NUMBER;

/**
 * Parser for input design xml files, build a {@link Scenario} model.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.7
 */
public class ScenarioXmlParser extends MexicoXmlParser<Scenario> {

    /**
     * Factors in input design model indexed by their id.
     *
     * @since 0.7
     */
    protected final Map<String, Factor> factors;

    /**
     * Flag to know if parser is used standalone (says to import a scenario) or
     * as part of input designer parser.
     *
     * @since 0.7
     */
    protected final boolean standalone;

    public ScenarioXmlParser(Map<String, Factor> factors) {
        this(factors, true);
    }

    public ScenarioXmlParser(Map<String, Factor> factors, boolean standalone) {
        this.factors = factors;
        this.standalone = standalone;
    }

    @Override
    protected Scenario parseModel(XmlPullParser parser) throws IOException, XmlPullParserException, ParseException {

        if (standalone) {
            // file must start with input design tag
            checkStartFile(
                    parser, SCENARIO,
                    "Scenario file must start with " + SCENARIO + " tag");
        }

        Scenario scenario = new ScenarioImpl();

        // name
        String name = parser.getAttributeValue(null, SCENARIO_NAME);
        scenario.setName(name);

        // orderValue
        String orderValue = parser.getAttributeValue(null, SCENARIO_ORDER_NUMBER);
        scenario.setOrderNumber(Integer.parseInt(orderValue));

        // factor values
        if (testNextStartTag(parser, SCENARIO_FACTOR_VALUES)) {

            Map<Factor, Object> factorValues = Maps.newLinkedHashMap();
            while (!(testCurrentEndTag(parser, SCENARIO))) {

                // get factor
                String factorId = parser.getAttributeValue(null, FACTOR);
                Factor factor = factors.get(factorId);

                if (factor == null) {
                    throw new IOException(
                            "Could not find factor named '" +
                            factorId + "' at line " + parser.getLineNumber());
                }
                ValueType valueType = factor.getDomain().getValueType();

                // factor value
                String value = parser.nextText();

                factorValues.put(factor, getTypedValue(valueType, value));

                // read next tag
                parser.nextTag();
            }
            scenario.setFactorValues(factorValues);
        }

        if (!standalone) {

            // read next tag
            parser.nextTag();
        }

        return scenario;
    }
}
