package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIUtil.java 1171 2013-09-17 09:25:46Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.6.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/TuttiUIUtil.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.TuttiBusinessException;
import fr.ifremer.tutti.TuttiTechnicalException;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;

import java.awt.Color;
import java.awt.Desktop;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import static org.nuiton.i18n.I18n._;

/**
 * Created: 14/06/12
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public final class TuttiUIUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIUtil.class);

    public static final double EARTH_RADIUS = 6378288.0;

    private TuttiUIUtil() {
        // never instanciate util class
    }

    public static void setProperty(Object bean, String property, Object value) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            PropertyUtils.setSimpleProperty(bean, property, value);
        } catch (Exception e) {
            throw new TuttiTechnicalException(_("tutti.property.set.error", property, bean.getClass().getName()), e);
        }
    }

    public static Object getProperty(Object bean, String property) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            return PropertyUtils.getSimpleProperty(bean, property);
        } catch (Exception e) {
            throw new TuttiTechnicalException(_("tutti.property.get.error", property, bean.getClass().getName()), e);
        }
    }

    public static Highlighter newBackgroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, false);
    }

    public static Highlighter newForegroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, true);
    }

    public static int computeDistanceInMeters(Float startLatitude,
                                              Float startLongitude,
                                              Float endLatitude,
                                              Float endLongitude) {

        double sLat = startLatitude * Math.PI / 180.0;
        double sLong = startLongitude * Math.PI / 180.0;
        double eLat = endLatitude * Math.PI / 180.0;
        double eLong = endLongitude * Math.PI / 180.0;

        Double d = EARTH_RADIUS *
                   (Math.PI / 2 - Math.asin(Math.sin(eLat) * Math.sin(sLat)
                                            + Math.cos(eLong - sLong) * Math.cos(eLat) * Math.cos(sLat)));
        return d.intValue();
    }

    public static String getDistanceInMilles(Float distance) {
        String distanceText;
        if (distance != null) {
            Float distanceInMilles = distance / 1852;
            distanceText = String.format("%.3f", distanceInMilles);

        } else {
            distanceText = "";
        }
        return distanceText;
    }

    private static DecimalFormatSymbols symbols;

    public static DecimalFormatSymbols getDecimalFormatSymbols() {
        if (symbols == null) {
            symbols = new DecimalFormatSymbols();
            symbols.setDecimalSeparator('.');
            symbols.setGroupingSeparator(' ');
        }
        return symbols;
    }

    private static DecimalFormat decimalFormat;

    public static DecimalFormat getDecimalFormat(int minDecimal, int maxDecimal) {
        if (decimalFormat == null) {
            decimalFormat = new DecimalFormat();
            decimalFormat.setDecimalFormatSymbols(getDecimalFormatSymbols());
            decimalFormat.setGroupingUsed(false);
        }
        decimalFormat.setMinimumFractionDigits(minDecimal);
        decimalFormat.setMaximumFractionDigits(maxDecimal);
        return decimalFormat;
    }

    public static String getWeightStringValue(Float weight) {
        String textValue;
        if (weight != null) {
            DecimalFormat weightDecimalFormat = getDecimalFormat(1, 3);
            textValue = weightDecimalFormat.format(weight);

        } else {
            textValue = "";
        }
        return textValue;
    }

    public static void openLink(URL url) {
        try {
            openLink(url.toURI());
        } catch (URISyntaxException e) {
            throw new TuttiTechnicalException(_("swing.error.cannot.open.link", url), e);
        }
    }

    public static Desktop getDesktopForBrowse() {

        if (!Desktop.isDesktopSupported()) {
            throw new TuttiTechnicalException(
                    _("swing.error.desktop.not.supported"));
        }

        Desktop desktop = Desktop.getDesktop();

        if (!desktop.isSupported(Desktop.Action.BROWSE)) {

            throw new TuttiTechnicalException(
                    _("swing.error.desktop.browse.not.supported"));
        }

        return desktop;
    }

    public static void openLink(URI uri) {

        Desktop desktop = getDesktopForBrowse();

        try {

            desktop.browse(uri);
        } catch (Exception e) {

            throw new TuttiTechnicalException(
                    _("swing.error.cannot.open.link", uri), e);
        }
    }

    public static Desktop getDesktopForMail() {

        if (!Desktop.isDesktopSupported()) {
            throw new TuttiTechnicalException(
                    _("swing.error.desktop.not.supported"));
        }

        Desktop desktop = Desktop.getDesktop();

        if (!desktop.isSupported(Desktop.Action.MAIL)) {

            throw new TuttiTechnicalException(
                    _("swing.error.desktop.mail.not.supported"));
        }

        return desktop;
    }

    public static void mail(String subject, String body) {

        Desktop desktop = getDesktopForMail();

        try {
            URI mailtoURI = new URI("mailto", null, null, "subject=" + subject + "&body=" + body, null);
            desktop.mail(mailtoURI);

        } catch (Exception e) {

            throw new TuttiTechnicalException(
                    _("swing.error.cannot.mail"), e);
        }
    }

    public static void tryToConnectToUpdateUrl(String urlAsString,
                                               String badUrlFormatI18nKey,
                                               String notReachI18nKey,
                                               String notFoundI18nKey) {
        URL url;
        // get url
        try {
            url = new URL(urlAsString);
        } catch (MalformedURLException e) {
            if (log.isDebugEnabled()) {
                log.debug("Bad url syntax at " + urlAsString, e);
            }
            throw new TuttiBusinessException(_(badUrlFormatI18nKey, urlAsString));
        }

        URLConnection urlConnection;
        // try to connect (fail if network or remote host does not exists)
        try {
            urlConnection = url.openConnection();
            urlConnection.setConnectTimeout(10000);
            urlConnection.connect();
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not connect to " + urlAsString, e);
            }
            throw new TuttiBusinessException(_(notReachI18nKey, urlAsString));
        }

        // try to open the resource (fail if resources does not exist)
        try {
            urlConnection.setReadTimeout(1000);
            InputStream inputStream = null;
            try {
                inputStream = urlConnection.getInputStream();
            } finally {
                IOUtils.closeQuietly(inputStream);
            }
        } catch (Exception e) {
            if (log.isDebugEnabled()) {
                log.debug("Could not found file at to " + urlAsString, e);
            }
            throw new TuttiBusinessException(_(notFoundI18nKey, urlAsString));
        }
    }

    public static float getRoundedLengthStep(float lengthStep, boolean aroundUp) {
        int intValue = (int) ((lengthStep + (aroundUp ? 0.001f : 0f)) * 10);
        float result = intValue / 10f;
        return result;
    }

}
