package fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesFrequencyUIHandler.java 138 2013-01-02 10:07:34Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/frequency/SpeciesFrequencyUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.ezware.oxbow.swingbits.util.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchUIHandler;
import fr.ifremer.tutti.ui.swing.util.Cancelable;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.validator.swing.SwingValidatorMessageTableRenderer;
import jaxx.runtime.validator.swing.SwingValidatorUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesFrequencyUIHandler extends AbstractTuttiTableUIHandler<SpeciesFrequencyRowModel, SpeciesFrequencyUIModel> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesFrequencyUIHandler.class);

    /**
     * UI.
     *
     * @since 0.2
     */
    private final SpeciesFrequencyUI ui;

    public SpeciesFrequencyUIHandler(TuttiUIContext context,
                                     SpeciesFrequencyUI ui) {
        super(context,
              SpeciesFrequencyRowModel.PROPERTY_LENGTH_STEP,
              SpeciesFrequencyRowModel.PROPERTY_NUMBER,
              SpeciesFrequencyRowModel.PROPERTY_WEIGHT,
              SpeciesFrequencyRowModel.PROPERTY_COMPUTED_WEIGHT);
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected SpeciesFrequencyTableModel getTableModel() {
        return (SpeciesFrequencyTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SpeciesFrequencyRowModel row) {
        return row.getLengthStepCaracteristic() != null &&
               row.getLengthStep() != null &&
               row.getNumber() != null;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SpeciesFrequencyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        recomputeRowValidState(row);

        if (SpeciesFrequencyRowModel.PROPERTY_NUMBER.equals(propertyName)) {

            // Need to recompute the computedWeight
            computeComputedWeight(row);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SpeciesFrequencyRowModel> rowMonitor,
                                             SpeciesFrequencyRowModel row) {
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          SpeciesFrequencyRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);
        ui.getValidator().doValidate();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    protected SpeciesFrequencyUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {

        SpeciesFrequencyUIModel model = new SpeciesFrequencyUIModel();

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Caracteristic> lengthStepCaracterics =
                ui.getContextValue(
                        List.class,
                        SpeciesBatchUIHandler.SPECIES_FREQUENCY_LENGHTS);
        Preconditions.checkNotNull(lengthStepCaracterics);

        SpeciesFrequencyUIModel model = getModel();

        //TODO Use protocol to have lengthStepCaracteristic to use (if any protocol)
        initBeanComboBox(ui.getLengthStepCaracteristicComboBox(),
                         lengthStepCaracterics,
                         model.getLengthStepCaracteristic());

        //TODO Should it come from PROTOCOL or config ?
        model.setStep(.5f);

        model.setMinStep(10f);
        model.setMaxStep(20f);

        //TODO Configure this ?
        model.setConfigurationMode(SpeciesFrequencyUIModel.ConfigurationMode.SIMPLE);

        ui.getRafaleStepField().getTextField().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    e.consume();
                    Float step = (Float) ui.getRafaleStepField().getModel();

                    applyRafaleStep(step);
                }
            }
        });

        // when lengthStepCaracteristic changed, let's updates all row with the new value
        model.addPropertyChangeListener(SpeciesFrequencyUIModel.PROPERTY_LENGHT_STEP_CARACTERISTIC, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Caracteristic newValue = (Caracteristic) evt.getNewValue();
                for (SpeciesFrequencyRowModel rowModel : getModel().getRows()) {
                    rowModel.setLengthStepCaracteristic(newValue);
                    recomputeRowValidState(rowModel);
                }
                ui.getValidator().doValidate();
            }
        });

        // create table column model
        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        { // LengthStep

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.LENGTH_STEP,
                                  TuttiUI.DECIMAL1_PATTERN);
        }

        { // Number

            addIntegerColumnToModel(columnModel,
                                    SpeciesFrequencyTableModel.NUMBER,
                                    TuttiUI.INT_6_DIGITS_PATTERN);
        }

        { // Weight

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // ComputedWeight

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.COMPUTED_WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        // create table model
        SpeciesFrequencyTableModel tableModel =
                new SpeciesFrequencyTableModel(columnModel, model);

        JXTable table = getTable();

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        installTableKeyListener(columnModel, table);

        initTable(table);

        SwingValidatorUtil.installUI(ui.getErrorTable(),
                                     new SwingValidatorMessageTableRenderer());

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isInfoEnabled()) {
            log.info("Cancel UI " + ui);
        }

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model (in that way)
        getModel().setValid(false);

        // close dialog
        closeDialog(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void generateLengthSteps() {

        SpeciesFrequencyUIModel model = getModel();
        SpeciesFrequencyTableModel tableModel = getTableModel();

        Map<Float, SpeciesFrequencyRowModel> rowsByStep =
                getTableModel().getRowCache();

        Float minStep = model.getLengthStep(model.getMinStep());
        Float maxStep = model.getLengthStep(model.getMaxStep());

        for (float i = minStep, step = model.getStep(); i <= maxStep; i += step) {
            if (!rowsByStep.containsKey(i)) {

                // add it
                SpeciesFrequencyRowModel newRow = tableModel.createNewRow();
                newRow.setLengthStep(i);
                rowsByStep.put(i, newRow);
            }
        }

        List<SpeciesFrequencyRowModel> rows =
                Lists.newArrayList(rowsByStep.values());

        model.setRows(rows);
    }

    public void applyRafaleStep(Float step) {

        if (log.isInfoEnabled()) {
            log.info("Will apply rafale step: " + step);
        }
        SpeciesFrequencyUIModel model = getModel();
        SpeciesFrequencyTableModel tableModel = getTableModel();

        Map<Float, SpeciesFrequencyRowModel> rowsByStep = tableModel.getRowCache();

        float aroundLengthStep = model.getLengthStep(step);

        SpeciesFrequencyRowModel row = rowsByStep.get(aroundLengthStep);

        int rowIndex;

        if (row != null) {

            // increments current row
            Integer number = row.getNumber();
            row.setNumber((number == null ? 0 : number) + 1);
            rowIndex = tableModel.updateRow(row);

        } else {

            // create a new row

            row = tableModel.createNewRow();
            row.setLengthStep(aroundLengthStep);
            row.setNumber(1);

            // get new index
            List<Float> steps = Lists.newArrayList(rowsByStep.keySet());
            steps.add(aroundLengthStep);

            Collections.sort(steps);

            rowIndex = steps.indexOf(aroundLengthStep);

            tableModel.addNewRow(rowIndex, row);
        }

        getTable().scrollRowToVisible(rowIndex);
    }

    public void editBatch(SpeciesBatchRowModel speciesBatch) {

        List<SpeciesFrequencyRowModel> frequency = null;

        Caracteristic lengthStepCaracteristic = null;
        if (speciesBatch != null) {
            frequency = speciesBatch.getFrequency();
        }

        List<SpeciesFrequencyRowModel> editFrequency = Lists.newArrayList();

        if (frequency != null) {

            SpeciesFrequencyTableModel tableModel = getTableModel();

            for (SpeciesFrequencyRowModel rowModel : frequency) {

                SpeciesFrequencyRowModel newRow = tableModel.createNewRow();
                newRow.setLengthStepCaracteristic(rowModel.getLengthStepCaracteristic());
                newRow.setLengthStep(rowModel.getLengthStep());
                newRow.setNumber(rowModel.getNumber());
                newRow.setComputedWeight(rowModel.getComputedWeight());
                editFrequency.add(newRow);
            }

            if (CollectionUtils.isNotEmpty(frequency)) {
                lengthStepCaracteristic =
                        frequency.get(0).getLengthStepCaracteristic();
            }
        }

        if (log.isInfoEnabled()) {
            log.info("Will edit batch row: " + speciesBatch + " with " +
                     editFrequency.size() + " frequency");
        }

        SpeciesFrequencyUIModel model = getModel();

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setRows(editFrequency);

        if (lengthStepCaracteristic == null) {

            // no lengthStep caracteristic to apply make sure it is not setted.
            if (log.isInfoEnabled()) {
                log.info("No lengthStepCaracteristic to set.");
            }
            model.setLengthStepCaracteristic(null);
        } else {

            // apply existing lengthStepCaracteristic

            if (log.isInfoEnabled()) {
                log.info("Use lengthStepCaracteristic: " +
                         lengthStepCaracteristic.getName());
            }
            model.setLengthStepCaracteristic(lengthStepCaracteristic);
        }

        // keep batch (will be used to push back editing entry)
        model.setBatch(speciesBatch);
    }

    public void save() {

        if (log.isInfoEnabled()) {
            log.info("Save UI " + ui);
        }

        closeDialog(ui);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeComputedWeight(SpeciesFrequencyRowModel row) {

        if (log.isInfoEnabled()) {
            log.info("Will recompute computed weight for frequency: " + row);
        }
    }

}