package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIContext.java 148 2013-01-03 17:52:22Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiUIContext.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.TuttiService;
import fr.ifremer.tutti.service.TuttiServiceContext;
import fr.ifremer.tutti.ui.swing.config.TuttiConfig;
import fr.ifremer.tutti.ui.swing.util.UIMessageNotifier;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.widget.SwingSession;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.util.Set;

/**
 * UI application context.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class TuttiUIContext extends AbstractBean implements Closeable, UIMessageNotifier {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIContext.class);

    public static final String PROPERTY_PROGRAM_ID = "programId";

    public static final String PROPERTY_CRUISE_ID = "cruiseId";

    public static final String PROPERTY_PROTOCOL_ID = "protocolId";

    public static final Set<String> ID_TO_SAVE_PROPERTIES = Sets.newHashSet(
            PROPERTY_PROGRAM_ID,
            PROPERTY_CRUISE_ID,
            PROPERTY_PROTOCOL_ID);

    public static final String PROPERTY_SCREEN = "screen";

    public static final String PROPERTY_CRUISE_CONTEXT_FILLED = "cruiseContextFilled";

    public static final String PROPERTY_PROTOCOL_FILLED = "protocolFilled";

    /**
     * Application context (only one for all the application).
     *
     * @since 0.1
     */
    private static TuttiUIContext applicationContext;

    /**
     * Application global configuration.
     *
     * @since 0.1
     */
    protected final TuttiConfig config;

    /**
     * Service context used by any service.
     *
     * @since 0.1
     */
    protected final TuttiServiceContext serviceContext;

    /**
     * Swing session used to save ui states.
     *
     * @since 0.1
     */
    protected final SwingSession swingSession;

    /**
     * Id of last selected program (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String programId;

    /**
     * Id of last selected cruise (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String cruiseId;

    /**
     * Id of last selected protocol (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String protocolId;

    /**
     * Current screen displayed in ui.
     *
     * @since 0.1
     */
    protected TuttiScreen screen;

    /**
     * Message notifiers.
     *
     * @since 0.3
     */
    protected final Set<UIMessageNotifier> messageNotifiers;

    public static TuttiUIContext newContext(TuttiConfig config) {
        Preconditions.checkNotNull(config);
        Preconditions.checkState(applicationContext == null,
                                 "Application context was already opened!");
        applicationContext = new TuttiUIContext(config);
        return applicationContext;
    }

    protected TuttiUIContext(TuttiConfig config) {
        this.config = config;
        this.serviceContext = new TuttiServiceContext(config.getServiceConfig());
        this.swingSession = new SwingSession(getConfig().getUIConfigFile(), false);
        UIMessageNotifier logMessageNotifier = new UIMessageNotifier() {

            @Override
            public void showInformationMessage(String message) {
                if (log.isInfoEnabled()) {
                    log.info(message);
                }
            }
        };
        this.messageNotifiers = Sets.newHashSet();
        addMessageNotifier(logMessageNotifier);
    }

    public <S extends TuttiService> S getService(Class<S> serviceType) {
        return serviceContext.getService(serviceType);
    }

    public TuttiConfig getConfig() {
        return config;
    }

    public SwingSession getSwingSession() {
        return swingSession;
    }

    public String getProgramId() {
        return programId;
    }

    public String getCruiseId() {
        return cruiseId;
    }

    public String getProtocolId() {
        return protocolId;
    }

    public boolean isCruiseContextFilled() {
        return StringUtils.isNotBlank(programId) &&
               StringUtils.isNotBlank(cruiseId);
    }

    public boolean isProtocolFilled() {
        return StringUtils.isNotBlank(protocolId);
    }

    public TuttiScreen getScreen() {
        return screen;
    }

    public void setProgramId(String programId) {
        boolean oldValue = isCruiseContextFilled();

        this.programId = programId;

        // always propagate the change
        firePropertyChange(PROPERTY_PROGRAM_ID, -1, programId);
        firePropertyChange(PROPERTY_CRUISE_CONTEXT_FILLED,
                           oldValue, isCruiseContextFilled());
    }

    public void setCruiseId(String cruiseId) {
        boolean oldValue = isCruiseContextFilled();

        this.cruiseId = cruiseId;

        // always propagate the change
        firePropertyChange(PROPERTY_CRUISE_ID, -1, cruiseId);
        firePropertyChange(PROPERTY_CRUISE_CONTEXT_FILLED,
                           oldValue, isCruiseContextFilled());
    }

    public void setProtocolId(String protocolId) {
        boolean oldValue = isProtocolFilled();
        this.protocolId = protocolId;

        // always propagate the change
        firePropertyChange(PROPERTY_PROTOCOL_ID, -1, protocolId);
        firePropertyChange(PROPERTY_PROTOCOL_FILLED,
                           oldValue, isProtocolFilled());
    }

    public void setScreen(TuttiScreen screen) {
        Object oldValue = getScreen();
        this.screen = screen;
        firePropertyChange(PROPERTY_SCREEN, oldValue, screen);
    }

    public void open() {

        if (programId == null) {

            // load it from config
            setProgramId(getConfig().getProgramId());
        }

        if (cruiseId == null) {

            // load it from config
            setCruiseId(getConfig().getCruiseId());
        }

        if (protocolId == null) {

            // load it from config
            setProtocolId(getConfig().getProtocolId());
        }

        //check if programId is sane
        PersistenceService persistenceService =
                getService(PersistenceService.class);

        if (protocolId != null) {
            TuttiProtocol protocol = persistenceService.getProtocol(protocolId);
            if (protocol == null) {

                // not found in this db

                if (log.isWarnEnabled()) {
                    log.warn("Remove invalid protocolId: " + protocolId);
                }

                setProtocolId(null);
            }
        }

        if (programId != null) {

            Program program = persistenceService.getProgram(programId);
            if (program == null) {

                // not found in this db

                if (log.isWarnEnabled()) {
                    log.warn("Remove invalid programId: " + programId);
                }

                setProgramId(null);
                setCruiseId(null);

            } else {

                if (log.isInfoEnabled()) {
                    log.info("ProgramId valid: " + programId);
                }

                setProgramId(programId);

                // test cruiseId
                if (cruiseId != null) {

                    Cruise cruise = persistenceService.getCruise(cruiseId);

                    if (cruise != null &&
                        !cruise.getProgram().getId().equals(programId)) {

                        // not matchin program, reset cruise id
                        cruise = null;
                    }

                    if (cruise == null) {

                        // not found in this db

                        if (log.isWarnEnabled()) {
                            log.warn("Remove invalid cruiseId: " + cruiseId);
                        }
                        setCruiseId(null);

                    } else {

                        if (log.isInfoEnabled()) {
                            log.info("CruiseId valid: " + cruiseId);
                        }
                    }
                }
            }
        }

        // save back to config
        saveContextToConfig();

        // list when programId or campaingId change to save the configuration
        addPropertyChangeListener(new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                if (ID_TO_SAVE_PROPERTIES.contains(evt.getPropertyName())) {
                    saveContextToConfig();
                }
            }
        });
    }

    @Override
    public void close() {

        // Clear data references
        messageNotifiers.clear();

        programId = null;
        cruiseId = null;
        protocolId = null;

        setScreen(null);

        IOUtils.closeQuietly(serviceContext);

        // remove listeners
        PropertyChangeListener[] listeners = getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            if (log.isDebugEnabled()) {
                log.debug("Remove listener: " + listener);
            }
            removePropertyChangeListener(listener);
        }
    }

    protected void saveContextToConfig() {
        if (log.isInfoEnabled()) {
            log.info("Save config (programId: " + programId + ", cruiseId: " +
                     cruiseId + ", protocolId: " + protocolId + ')');
        }
        config.setProgramId(programId);
        config.setCruiseId(cruiseId);
        config.setProtocolId(protocolId);
        config.save();
    }

    public void addMessageNotifier(UIMessageNotifier messageNotifier) {
        this.messageNotifiers.add(messageNotifier);
    }

    @Override
    public void showInformationMessage(String message) {
        for (UIMessageNotifier messageNotifier : messageNotifiers) {
            messageNotifier.showInformationMessage(message);
        }
    }
}
