package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: MainUIHandler.java 132 2012-12-30 19:39:00Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2.5/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/MainUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.config.TuttiConfig;
import fr.ifremer.tutti.ui.swing.config.TuttiConfigUI;
import fr.ifremer.tutti.ui.swing.content.cruise.EditCruiseUI;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUI;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.content.program.EditProgramUI;
import fr.ifremer.tutti.ui.swing.content.protocol.EditProtocolUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.widget.SwingSession;

import javax.swing.JComponent;
import java.awt.Desktop;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.util.Calendar;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class MainUIHandler extends AbstractTuttiUIHandler<TuttiUIContext> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(MainUIHandler.class);

    protected MainUI ui;

    protected JComponent currentBody;

    protected final PersistenceService persistenceService;

    protected MainUIHandler(TuttiUIContext context, MainUI ui) {
        super(context);
        this.ui = ui;
        this.persistenceService = context.getService(PersistenceService.class);
        context.addMessageNotifier(this);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {
        context.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String propertyName = evt.getPropertyName();
                if (TuttiUIContext.ID_TO_SAVE_PROPERTIES.contains(propertyName)) {

                    // change the ui title
                    changeTitle();

                } else if (propertyName.equals(TuttiUIContext.PROPERTY_SCREEN)) {
                    setScreen((TuttiScreen) evt.getNewValue());
                }
            }
        });
    }

    @Override
    public void afterInitUI() {

        // Sync to error dialog
        ErrorDialogUI.init(ui);

        // Init SwingSession
        SwingSession swingSession = context.getSwingSession();
        swingSession.add(ui);
        swingSession.save();

        //SwingUtil.getLayer(ui.getBody()).setUI(new BetaLayerUI());

        changeTitle();

        TuttiScreen screen;

        if (context.isCruiseContextFilled()) {

            // direct fill catches screen
            screen = TuttiScreen.EDIT_FISHING_OPERATION;
        } else {

            // go to select cruise screen
            screen = TuttiScreen.SELECT_CRUISE;
        }

        context.setScreen(screen);
    }

    @Override
    public void onCloseUI() {

//        closeCurrentBody();

        context.close();
    }

    @Override
    protected TuttiUIContext getModel() {
        return ui.getModel();
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void closeTutti() {
        RunTutti.closeTutti(ui, true);
    }

    public void reloadTutti() {

        // Close the application
        RunTutti.closeTutti(ui, false);

        // Re-open the context
//        try {
//            context.open();
//        } catch (Exception ex) {
//            ErrorDialogUI.showError(ex);
//        }

        RunTutti.startTutti(context);
    }

    public void reloadApplication() {
        //TODO
    }

    public void reloadShortcuts() {
        //TODO
    }

    public void showConfig() {

        TuttiConfigUI configUI = new TuttiConfigUI(ui);
        configUI.createUI();
    }

    public void showManageProtocol() {
        context.setScreen(TuttiScreen.EDIT_PROTOCOL);
    }

    public void showSelectCampaign() {
        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void showFillCatches() {
        context.setScreen(TuttiScreen.EDIT_FISHING_OPERATION);
    }

    public void showImportScreen() {

    }

    public void showExportScreen() {

    }

    public void showAllegroScreen() {

    }

    public void showAbout() {

        ApplicationConfig applicationConfig =
                context.getConfig().getApplicationConfig();

        String iconPath = applicationConfig.getOption("application.icon.path");
        String name = "tutti-ui-swing";
        String licensePath = "META-INF/" + name + "-LICENSE.txt";
        String thirdPartyPath = "META-INF/" + name + "-THIRD-PARTY.txt";

        AboutPanel about = new AboutPanel();
        about.setTitle(_("tutti.title.about"));
        about.setAboutText(_("tutti.about.message"));

        TuttiConfig config = context.getConfig();
        int currentYear = Calendar.getInstance().get(Calendar.YEAR);
        int inceptionYear = config.getInceptionYear();
        String years;
        if (currentYear != inceptionYear) {
            years = inceptionYear + "-" + currentYear;
        } else {
            years = inceptionYear + "";
        }

        about.setBottomText(_("tutti.about.bottomText",
                              config.getOrganizationName(),
                              years,
                              config.getVersion()));
        about.setIconPath(iconPath);
        about.setLicenseFile(licensePath);
        about.setThirdpartyFile(thirdPartyPath);
        about.buildTopPanel();
        about.init();
        about.showInDialog(ui, true);

        // register on swing session
        context.getSwingSession().add(about);
    }

    public void gotoSite() {
        TuttiConfig config = context.getConfig();

        URL siteURL = config.getSiteUrl();

        if (log.isDebugEnabled()) {
            log.debug("goto " + siteURL);
        }
        if (Desktop.isDesktopSupported() &&
            Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception e) {
                log.error("Failed to open '" + siteURL + "' in browser", e);
                ErrorDialogUI.showError(e);
            }
        }
    }

    @Override
    public final void showInformationMessage(String message) {
        ui.getStatus().setStatus(message);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void setScreen(TuttiScreen screen) {

        try {

            // busy ui
            TuttiUIUtil.updateBusyState(ui, true);

            // close current body (if any)
            if (currentBody != null) {
                TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
                body.getHandler().onCloseUI();

                context.getSwingSession().save();

                ui.getBody().remove(currentBody);

                currentBody = null;
            }

            if (screen != null) {

                String programId = context.getProgramId();
                String cruiseId = context.getCruiseId();
                String protocolId = context.getProtocolId();

                JComponent screenUI;
                String screenTitle;

                switch (screen) {
                    default:
                    case SELECT_CRUISE:

                        screenUI = new SelectCruiseUI(context);
                        screenTitle = _("tutti.title.home");
                        break;

                    case EDIT_PROGRAM:

                        if (programId == null) {
                            screenTitle = _("tutti.title.create.program");
                        } else {
                            screenTitle = _("tutti.title.edit.program");
                        }

                        screenUI = new EditProgramUI(context);
                        break;

                    case EDIT_CRUISE:

                        if (cruiseId == null) {
                            screenTitle = _("tutti.title.create.cruise");
                        } else {
                            screenTitle = _("tutti.title.edit.cruise");
                        }

                        screenUI = new EditCruiseUI(context);
                        break;

                    case EDIT_PROTOCOL:

                        if (protocolId == null) {
                            screenTitle = _("tutti.title.create.protocol");
                        } else {
                            screenTitle = _("tutti.title.edit.protocol");
                        }

                        screenUI = new EditProtocolUI(context);
                        break;

                    case EDIT_FISHING_OPERATION:

                        screenTitle = _("tutti.title.edit.operations", getSelectedCruiseTitle());
                        screenUI = new FishingOperationsUI(context);
                        break;
                }

                this.currentBody = screenUI;
                context.getSwingSession().add(currentBody);
                ui.getBody().setTitle(screenTitle);
                ui.getBody().add(currentBody);
            }
        } catch (Exception e) {

            ErrorDialogUI.showError(e);

            // go back to home
            context.setScreen(TuttiScreen.SELECT_CRUISE);
        } finally {

            // free ui
            TuttiUIUtil.updateBusyState(ui, false);
        }
    }

//    protected void closeTutti(boolean exit) {
//
//        context.getSwingSession().save();
//
//        ui.setVisible(false);
//        ui.dispose();
//
//        onCloseUI();
//
//        if (exit) {
//            System.exit(0);
//        }
//    }

    protected void changeTitle() {

        String title = getSelectedCruiseTitle();

        ui.setTitle("Tutti - v " + getConfig().getVersion() + " [ " + title + " ]");
    }

    protected String getSelectedCruiseTitle() {

        String programId = context.getProgramId();

        String title;

        if (programId == null) {

            // no program selected (so neither cruise)

            title = _("tutti.title.noSelectedProgram");

        } else {

            // selected program

            Program program = persistenceService.getProgram(programId);

            title = _("tutti.title.selectedProgram", program.getName()) + " / ";

            String cruiseId = context.getCruiseId();

            if (cruiseId == null) {

                // no selected cruise

                title += _("tutti.title.noSelectedCruise");

            } else {

                // selected cruise
                Cruise cruise;
                cruise = persistenceService.getCruise(cruiseId);

                title += _("tutti.title.selectedCruise", cruise.getName());
            }

            title += " / ";

            String protocolId = context.getProtocolId();

            if (protocolId == null) {

                // no selected protocol

                title += _("tutti.title.noSelectedProtocol");


            } else {

                // selected protocol
                TuttiProtocol protocol = persistenceService.getProtocol(protocolId);

                title += _("tutti.title.selectedProtocol", protocol.getName());
            }


        }
        return title;
    }

}
