/*
 * #%L
 * IsisFish
 * 
 * $Id: SensitivityExportStorage.java 3431 2011-07-06 14:17:27Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 Ifremer, CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Gestion des fichers VCS de type {@link SensitivityExport}
 * (appartenant au module exports).
 *
 * Created: 18 août 2005 15:07:36 CEST
 *
 * @author chatellier eric <chatellier@codelutin.com>
 * @version $Revision: 3431 $
 *
 * Last update: $Date: 2011-07-06 16:17:27 +0200 (mer., 06 juil. 2011) $
 * by : $Author: chatellier $
 */
public class SensitivityExportStorage extends ExportStorage implements Docable {

    public static final String SENSITIVITY_EXPORT_PATH = "sensitivityexports";

    /** Template freemarker pour les scripts d'export . */
    public static final String SENSITIVITY_EXPORT_TEMPLATE = "templates/script/sensitivityexport.ftl";

    @SuppressWarnings("unchecked")
    private static Map<String, SensitivityExportStorage> sensitivityExportsCache = (Map<String, SensitivityExportStorage>) new ReferenceMap();

    /**
     * Constructeur.
     *
     * @param rootSrc   le repertoire root de stockage des exports
     * @param directory le repertoire de l'export
     * @param name      le nom de l'export
     */
    protected SensitivityExportStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    /**
     * Get context (official VCS) sensitivity export directory.
     * 
     * @return context sensitivity export directory
     */
    public static File getSensitivityExportDirectory() {
        File result = new File(getContextDatabaseDirectory(), SENSITIVITY_EXPORT_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Get community VCS sensitivity export directory.
     * 
     * @return community sensitivity export directory
     */
    public static File getCommunitySensitivityExportDirectory() {
        File result = new File(getCommunityDatabaseDirectory(), SENSITIVITY_EXPORT_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne une nouvelle instance de l'export. Compile le fichier si besoin.
     *
     * @return une nouvelle instance de la classe d'export
     * @throws IsisFishException
     */
    public SensitivityExport getNewSensitivityExportInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (SensitivityExport) result;
    }

    /**
     * Retourne le storage pour l'export demandé.
     *
     * @param name le nom de la export souhaitée
     * @return Le storage pour l'export
     */
    public static SensitivityExportStorage getSensitivityExport(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        SensitivityExportStorage result = sensitivityExportsCache.get(cacheName);
        if (result == null) {
            result = new SensitivityExportStorage(getContextDatabaseDirectory(),
                    getSensitivityExportDirectory(), name);
            sensitivityExportsCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne le storage pour l'export demandé.
     *
     * @param name le nom de l'export souhaitée
     * @return Le storage pour l'export
     */
    public static SensitivityExportStorage getCommunitySensitivityExport(String name) {
        String cacheName = getCommunityDatabaseCacheKey(name);
        SensitivityExportStorage result = sensitivityExportsCache.get(cacheName);
        if (result == null) {
            result = new SensitivityExportStorage(getCommunityDatabaseDirectory(),
                    getCommunitySensitivityExportDirectory(), name);
            sensitivityExportsCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local.
     *
     * @return la liste des noms de toutes les régions disponible en local
     */
    public static List<String> getSensitivityExportNames() {
        List<String> result = getStorageNames(getSensitivityExportDirectory());
        result.addAll(getStorageNames(getCommunitySensitivityExportDirectory()));
        return result;
    }

    public static void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), SENSITIVITY_EXPORT_PATH);
    }
}
