/*
 * #%L
 * IsisFish
 * 
 * $Id: RuleStorage.java 3445 2011-07-15 13:12:45Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2011 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Class permettant de representer une regle de gestion.
 * Une regle de gestion est un fichier Java que l'on compile si besoin.
 * Il se trouve dans le package portant le nom de la region de cette maniere
 * plusieurs region peuvent avoir des regles avec le meme nom mais pas le
 * meme code.
 * <p/>
 * Gere les fichiers VCS de type {@link Rule} (package rules)
 *
 * Created: 17 août 2005 11:11:51 CEST
 *
 * @author Benjamin POUSSIN <poussin@codelutin.com>
 *
 * @version $Revision: 3445 $
 *
 * Last update: $Date: 2011-07-15 15:12:45 +0200 (ven., 15 juil. 2011) $ by : $Author: chatellier $
 */
public class RuleStorage extends JavaSourceStorage implements Docable { // RulesStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(RuleStorage.class);
    
    public static final String RULE_PATH = "rules";

    /** Template freemarker pour les regles. */
    public static final String RULE_TEMPLATE = "templates/script/rule.ftl";

    @SuppressWarnings("unchecked")
    private static Map<String, RuleStorage> rulesCache =
            (Map<String, RuleStorage>) new ReferenceMap();

    /**
     * Contruit un nouveau rule storage
     *
     * @param rootSrc   La region auquelle est attaché la rule
     * @param directory le repertoire ou devrait se trouver la rule
     * @param name      le nom de la rule
     */
    protected RuleStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    /**
     * Get context (official VCS) rule directory.
     * 
     * @return context rule directory
     */
    static public File getRuleDirectory() {
        File result = new File(getContextDatabaseDirectory(), RULE_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Get community VCS rule directory.
     * 
     * @return community rule directory
     */
    public static File getCommunityRuleDirectory() {
        File result = new File(getCommunityDatabaseDirectory(), RULE_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne le nom de toutes les regles existantes pour cette region
     *
     * @return all rule names found in local user database
     */
    static public List<String> getRuleNames() {
        List<String> rules = getStorageNames(getRuleDirectory());
        rules.addAll(getStorageNames(getCommunityRuleDirectory()));
        return rules;
    }

    /**
     * Retourne le storage pour la regle demandée
     *
     * @param name le nom de la regle souhaitée
     * @return Le storage pour la regle
     */
    static public RuleStorage getRule(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        RuleStorage result = rulesCache.get(cacheName);
        if (result == null) {
            result = new RuleStorage(getContextDatabaseDirectory(),
                    getRuleDirectory(), name);
            rulesCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne le storage pour la regle demandée
     *
     * @param name le nom de la regle souhaitée
     * @return Le storage pour la regle
     */
    static public RuleStorage getCommunityRule(String name) {
        String cacheName = getCommunityDatabaseCacheKey(name);
        RuleStorage result = rulesCache.get(cacheName);
        if (result == null) {
            result = new RuleStorage(getCommunityDatabaseDirectory(),
                    getCommunityRuleDirectory(), name);
            rulesCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne une nouvelle instance de la regle. Compile le fichier si besoin
     *
     * @return a new rule instance
     * @throws IsisFishException if any exception while instanciation
     */
    public Rule getNewRuleInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (Rule) result;
    }

    static public void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), RULE_PATH);
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local qui
     * ne sont pas encore sur le serveur VCS
     *
     * @return liste de noms de regions
     */
    static public List<String> getNewRuleNames() {
        List<String> result = getRuleNames();
        result.removeAll(getRemoteRuleNames());
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur VCS
     *
     * @return la liste des noms de toutes les régions disponible sur le serveur
     *         VCS. Si le serveur n'est pas disponible la liste retournée est
     *         vide.
     */
    static public List<String> getRemoteRuleNames() {
        File dir = getRuleDirectory();
        return getRemoteStorageNames(dir);
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible sur le
     * serveur VCS qui ne sont pas encore en local
     *
     * @return liste de noms de regions
     */
    static public List<String> getNewRemoteRuleNames() {
        List<String> result = getRemoteRuleNames();
        result.removeAll(getRuleNames());
        return result;
    }

    /**
     * <b>Be ware this method require to instanciate a Rule, so
     * it would be better to call as often as possible.</b>
     *
     * @return the descript of the instanciate Rule
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        String result = null;
        try {
            Rule rule = getNewRuleInstance();
            result = rule == null ? null : rule.getDescription();
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description",this));
        }
        return result;
    }

} // RuleStorage

