/*
 * #%L
 * IsisFish
 * 
 * $Id: ExportStorage.java 3420 2011-06-17 13:55:14Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2005 - 2011 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.util.Doc;
import fr.ifremer.isisfish.util.Docable;
import fr.ifremer.isisfish.vcs.VCSException;

/**
 * Gestion des fichers VCS de type {@link Export}
 * (appartenant au module exports).
 *
 * Created: 18 août 2005 15:07:36 CEST
 *
 * @author Grégoire DESSARD <dessard@codelutin.com>
 * @version $Revision: 3420 $
 *
 * Last update: $Date: 2011-06-17 15:55:14 +0200 (ven., 17 juin 2011) $
 * by : $Author: chatellier $
 */
public class ExportStorage extends JavaSourceStorage {

    /** Class logger. */
    private static Log log = LogFactory.getLog(ExportStorage.class);

    public static final String EXPORT_PATH = "exports";

    /** Template freemarker pour les scripts d'export . */
    public static final String EXPORT_TEMPLATE = "templates/script/export.ftl";

    @SuppressWarnings("unchecked")
    private static Map<String, ExportStorage> scriptsCache = (Map<String, ExportStorage>) new ReferenceMap();

    /**
     * Constructeur.
     *
     * @param rootSrc   le repertoire root de stockage des exports
     * @param directory le repertoire de l'export
     * @param name      le nom de l'export
     */
    protected ExportStorage(File rootSrc, File directory, String name) {
        super(rootSrc, directory, name);
    }

    /**
     * Get context (official VCS) export directory.
     * 
     * @return context export directory
     */
    public static File getExportDirectory() {
        File result = new File(getContextDatabaseDirectory(), EXPORT_PATH);
        result.mkdirs();
        return result;
    }
    
    /**
     * Get community VCS export directory.
     * 
     * @return community export directory
     */
    public static File getCommunityExportDirectory() {
        File result = new File(getCommunityDatabaseDirectory(), EXPORT_PATH);
        result.mkdirs();
        return result;
    }

    /**
     * Retourne une nouvelle instance de la regle. Compile le fichier si besoin
     *
     * @return une nouvelle instance de la classe d'export
     * @throws IsisFishException
     */
    public Export getNewExportInstance() throws IsisFishException {
        Object result = getNewInstance();
        return (Export) result;
    }

    /**
     * Retourne le storage pour l'export demandée
     *
     * @param name le nom de l'export souhaitée
     * @return Le storage pour l'export
     */
    public static ExportStorage getExport(String name) {
        String cacheName = getContextDatabaseCacheKey(name);
        ExportStorage result = scriptsCache.get(cacheName);
        if (result == null) {
            result = new ExportStorage(getContextDatabaseDirectory(),
                    getExportDirectory(), name);
            scriptsCache.put(cacheName, result);
        }
        return result;
    }
    
    /**
     * Retourne le storage pour l'export demandée
     *
     * @param name le nom de l'export souhaitée
     * @return Le storage pour l'export
     */
    public static ExportStorage getCommunityExport(String name) {
        String cacheName = getCommunityDatabaseCacheKey(name);
        ExportStorage result = scriptsCache.get(cacheName);
        if (result == null) {
            result = new ExportStorage(getCommunityDatabaseDirectory(),
                    getCommunityExportDirectory(), name);
            scriptsCache.put(cacheName, result);
        }
        return result;
    }

    /**
     * Retourne la liste des noms de toutes les régions disponible en local
     *
     * @return la liste des noms de toutes les régions disponible en local
     */
    public static List<String> getExportNames() {
        List<String> result = getStorageNames(getExportDirectory());
        result.addAll(getStorageNames(getCommunityExportDirectory()));
        return result;
    }

    public static void checkout() throws VCSException {
        checkout(IsisFish.config.getDatabaseDirectory(), EXPORT_PATH);
    }

    /**
     * <b>Be ware this method require to instanciate a Export, so
     * it would be better to call as often as possible.</b>
     *
     * @return the description of the instanciate Export
     * @see Doc
     * @see Docable
     */
    public String getDescription() {
        String result = null;
        try {
            Export export = getNewExportInstance();
            if (export != null) {
                result = export.getDescription();
            }
        } catch (Exception e) {
            log.warn(_("isisfish.error.not.found.description",this));
        }
        return result;
    }
}
