/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: ThesaurusSelectionManager.java 1807 2010-11-24 15:11:33Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-swing/src/main/java/com/jurismarches/vradi/ui/thesaurus/helpers/ThesaurusSelectionManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.thesaurus.helpers;

import com.jurismarches.vradi.VradiContext;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.Thesaurus;
import com.jurismarches.vradi.services.VradiException;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.tree.VradiDataProvider;
import com.jurismarches.vradi.ui.tree.VradiTreeNode;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.nuiton.wikitty.WikittyService;

import java.util.*;

/**
 * Manager of selections models. All tree and helper are register here.
 * On register, selection listener are attach. For thesaurus, when one
 * is selected, all other thesaurus panel must select the same.
 *
 * It manage too the proposition list.
 *
 * @author sletellier <letellier@codelutin.com>
 */
public class ThesaurusSelectionManager {

    /** Logger */
    static private final Log log = LogFactory.getLog(ThesaurusSelectionManager.class);

    protected JList propositionList;
    protected PropositionSelectionModel propositionSelectionModel;
    protected ThesaurusListener propositionListener;
    protected ThesaurusTreeHelper rootHelper;
    protected RootThesaurusListener rootListener;
    protected Map<ThesaurusTreeHelper, ThesaurusListener> helpers;
    protected Form form;
    protected boolean isRootListening = true;

    /**
     * This class manage the proposition list pass in param and must
     * know form concerned to select thesaurus
     *
     * @param propositionList proposition list
     * @param form            form concerned
     */
    public ThesaurusSelectionManager(JList propositionList, Form form) {
        this.form = form;
        this.propositionList = propositionList;

        // Create selection model and register it
        this.propositionSelectionModel = new PropositionSelectionModel();
        this.propositionList.setSelectionModel(propositionSelectionModel);

        this.helpers = new HashMap<ThesaurusTreeHelper, ThesaurusListener>();

        // Register listeners
        propositionListener = new ThesaurusListener();
        propositionList.getSelectionModel()
                .addListSelectionListener(propositionListener);
    }

    public VradiDataProvider getDataProvider() {
        return rootHelper.getDataProvider();
    }

    public Form getForm() {
        return form;
    }

    /**
     * Return root helper
     *
     * @return root helper
     */
    public ThesaurusTreeHelper getRootHelper() {
        return rootHelper;
    }

    /**
     * Return all helper less root one
     *
     * @return list of helper
     */
    public List<ThesaurusTreeHelper> getHelpers() {
        return new ArrayList<ThesaurusTreeHelper>(helpers.keySet());
    }

    /**
     * Register the tree and root helper and attach {@link RootThesaurusListener}
     * and select associated thesaurus
     *
     * @param rootHelper root helper to register
     * @param tree       tree to register
     */
    public void registerRootTreeHelper(ThesaurusTreeHelper rootHelper, JTree tree) {
        this.rootHelper = rootHelper;
        useMultipleSelectionMode(tree);
        rootListener = new RootThesaurusListener();
        rootHelper.setUI(tree, true, false, rootListener);
        selectRootThesaurus(rootHelper);
    }

    /**
     * Register the tree and helper and attach {@link ThesaurusListener}
     *
     * @param helper     helper to register
     * @param tree       tree to register
     */
    public void registerTreeHelper(ThesaurusTreeHelper helper, JTree tree) {
        registerListener(helper, tree);
        selectThesaurus(helper);
    }

    protected void registerListener(ThesaurusTreeHelper helper, JTree tree) {
        useMultipleSelectionMode(tree);
        ThesaurusListener listener = new ThesaurusListener();
        this.helpers.put(helper, listener);
        helper.setUI(tree, true, false, listener);
    }

    protected void useMultipleSelectionMode(JTree tree) {
        tree.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
    }

    protected void desactiveRootListener() {
        isRootListening = false;
    }

    protected void activeRootListener() {
        isRootListening = true;
    }

    protected void desactiveListener(ThesaurusTreeHelper helper) {
        helpers.get(helper).desactiveListener();
    }

    protected void activeListener(ThesaurusTreeHelper helper) {
        helpers.get(helper).activeListener();
    }

    protected void selectRootThesaurus(ThesaurusTreeHelper helper) {
        try {
            List<Thesaurus> thesauruses = VradiService.getVradiDataService().getThesaurusAttachedToForm(form);

            if (thesauruses != null) {
                desactiveRootListener();
                helper.tryToSelect(thesauruses);
                activeRootListener();
                propositionSelectionModel.tryToSelectObjects(thesauruses);
            }
        } catch (VradiException ex) {
            throw new RuntimeException("Can't get form thesaurus", ex);
        }
    }

    protected void selectThesaurus(ThesaurusTreeHelper helper) {
        desactiveListener(helper);
        List<VradiTreeNode> nodes = rootHelper.getSelectedNodes();
        List<String> thesaurusIds = extractIds(nodes);
        helper.tryToSelect(thesaurusIds);
        activeListener(helper);
    }

    protected List<String> extractIds(List<VradiTreeNode> nodes) {
        List<String> ids = new ArrayList<String>();
        for (VradiTreeNode node : nodes) {
            ids.add(node.getId());
        }
        return ids;
    }

    public void tryToSelect(Thesaurus thesaurus) {
        if (thesaurus != null) {
            rootHelper.tryToSelect(thesaurus);
        }
    }

    public void tryToUnSelect(Thesaurus thesaurus) {
        if (thesaurus != null) {
            rootHelper.tryToUnselect(thesaurus);
        }
    }

    protected class RootThesaurusListener implements TreeSelectionListener {

        @Override
        public void valueChanged(TreeSelectionEvent e) {
            if (!isRootListening) {
                return;
            }
            TreePath[] paths = e.getPaths();

            List<String> toAdd = new ArrayList<String>();
            List<String> toRemove = new ArrayList<String>();
            for(TreePath path : paths){
                VradiTreeNode node = (VradiTreeNode) path.getLastPathComponent();
                String id = node.getId();
                if (e.isAddedPath(path)){
                    if (log.isDebugEnabled()) {
                        Thesaurus thesaurus = ThesaurusDataHelper.restoreThesaurus(id);
                        log.debug("[Root listener] try to add thesaurus : " + thesaurus.getName());
                    }
                    toAdd.add(id);
                } else {
                    if (log.isDebugEnabled()) {
                        Thesaurus thesaurus = ThesaurusDataHelper.restoreThesaurus(id);
                        log.debug("[Root listener] try to remove thesaurus : " + thesaurus.getName());
                    }
                    toRemove.add(id);
                }
            }

            for (ThesaurusTreeHelper helper : helpers.keySet()){
                if (log.isDebugEnabled()) {
                    VradiTreeNode rootNode = helper.getRootNode();
                    if (Thesaurus.class.equals(rootNode.getInternalClass())) {
                        Thesaurus rootThesaurus = ThesaurusDataHelper.restoreThesaurus(rootNode.getId());
                        log.debug("[Root listener] Apply selection for helper with root " + rootThesaurus.getName());
                    }
                }
                desactiveListener(helper);
                helper.tryToSelect(toAdd);
                helper.tryToUnselect(toRemove);
                activeListener(helper);
            }

            propositionListener.desactiveListener();
            propositionSelectionModel.tryToSelect(toAdd);
            propositionSelectionModel.tryToUnSelect(toRemove);
            propositionListener.activeListener();
        }
    }

    protected class ThesaurusListener implements TreeSelectionListener, ListSelectionListener {
        
        protected List<Object> oldsPropositionsSelected;
        protected boolean isThesaurusListening = true;

        @Override
        public void valueChanged(TreeSelectionEvent e) {
            if (!isThesaurusListening) {
                return;
            }
            TreePath[] paths = e.getPaths();

            List<String> toAdd = new ArrayList<String>();
            List<String> toRemove = new ArrayList<String>();
            for(TreePath path : paths){
                VradiTreeNode node = (VradiTreeNode) path.getLastPathComponent();
                String id = node.getId();
                if (e.isAddedPath(path)){
                    if (log.isDebugEnabled()) {
                        Thesaurus thesaurus = ThesaurusDataHelper.restoreThesaurus(id);
                        log.debug("[Listener] try to add thesaurus : " + thesaurus.getName());
                    }
                    toAdd.add(id);
                } else {
                    if (log.isDebugEnabled()) {
                        Thesaurus thesaurus = ThesaurusDataHelper.restoreThesaurus(id);
                        log.debug("[Listener] try to remove thesaurus : " + thesaurus.getName());
                    }
                    toRemove.add(id);
                }
            }
            rootHelper.tryToSelect(toAdd);
            rootHelper.tryToUnselect(toRemove);
        }

        @Override
        public void valueChanged(ListSelectionEvent e) {
            if (!isThesaurusListening) {
                return;
            }

            List<Object> newPropositionsSelected = Arrays.asList(propositionList.getSelectedValues());
            List<String> idsToAdd = new ArrayList<String>();
            List<String> idsToRemove = new ArrayList<String>();
            for (Object o : newPropositionsSelected){
                if (oldsPropositionsSelected == null || !oldsPropositionsSelected.contains(o)){
                    Thesaurus proposition = (Thesaurus)o;
                    idsToAdd.add(proposition.getWikittyId());
                }
            }
            if (oldsPropositionsSelected != null){
                for (Object o : oldsPropositionsSelected){
                    if (!newPropositionsSelected.contains(o)){
                        Thesaurus proposition = (Thesaurus)o;
                        idsToRemove.add(proposition.getWikittyId());
                    }
                }
            }

            rootHelper.tryToSelect(idsToAdd);
            rootHelper.tryToUnselect(idsToRemove);

            // Cache
            oldsPropositionsSelected = newPropositionsSelected;
        }

        public void desactiveListener() {
            isThesaurusListening = false;
        }

        public void activeListener() {
            isThesaurusListening = true;
        }
    }

    protected class PropositionSelectionModel extends DefaultListSelectionModel {

        protected void tryToSelect(List<String> thesaurusId) {

            List<Thesaurus> thesauruss = ThesaurusDataHelper.restoreThesaurus(thesaurusId);
            tryToSelectObjects(thesauruss);
        }

        protected void tryToSelectObjects(List<Thesaurus> thesauruses) {

            for (Thesaurus thesaurus : thesauruses)  {
                int toSelect = ((DefaultListModel) propositionList.getModel()).indexOf(thesaurus);
                super.addSelectionInterval(toSelect, toSelect);
            }
        }

        protected void tryToUnSelect(List<String> thesaurusId) {

            List<Thesaurus> thesauruss = ThesaurusDataHelper.restoreThesaurus(thesaurusId);
            for (Thesaurus thesaurus : thesauruss)  {
                int toSelect = ((DefaultListModel) propositionList.getModel()).indexOf(thesaurus);
                super.removeSelectionInterval(toSelect, toSelect);
            }
        }
    }
}
