/*
 * #%L
 * Vradi :: Services
 * 
 * $Id: FormTypeManager.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.1/vradi-services/src/main/java/com/jurismarches/vradi/services/managers/FormTypeManager.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.services.managers;

import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyException;
import org.nuiton.wikitty.WikittyExtension;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyUtil;

import com.jurismarches.vradi.VradiServiceConfiguration;
import com.jurismarches.vradi.VradiConstants.FormTypeTemplateEnum;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.services.VradiException;

/**
 * Class containing the metods to manage the form types :
 * - creation, update, retrieving
 * - cache filling
 *
 * @author schorlet
 * @date 2010-01-28 08:08:53
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 */
public class FormTypeManager {
    private static final Log log = LogFactory.getLog(FormTypeManager.class);

    public final static String TYPE_TEMPLATE = "template";
    
    protected WikittyProxy wikittyProxy;

    public FormTypeManager(WikittyProxy wikittyProxy) {
        this.wikittyProxy = wikittyProxy;
    }

    /**
     * Retrieves the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve
     * @return the form type whose name is the parameter
     */
    public WikittyExtension getFormType(String name) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getFormType(" + name + ")");
        }
        
        try {
            WikittyExtension wikittyExtension = wikittyProxy.restoreExtensionLastVersion(name);
            return wikittyExtension;
            
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't get form type", eee);
            }
            throw new VradiException("Can't get form type", eee);
        }
    }

    /**
     * Retrieves the fields of the form type whose name is the parameter 'name'
     *
     * @param name the name of the form type we want to retrieve the fields
     * @return a map containing the field names and their type
     */
    public Map<String, FieldType> getFormTypeFields(String name) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("getFormTypeFields(" + name + ")");
        }
        
        try {
            WikittyExtension extension = getFormType(name);
            Map<String, FieldType> fields = new HashMap<String, FieldType>();
            
            for (String fieldName : extension.getFieldNames()) {
                fields.put(fieldName, extension.getFieldType(fieldName));
            }
    
            return fields;
            
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't get form type fields", eee);
            }
            throw new VradiException("Can't get form type fields", eee);
        }
    }

    /**
     * Updates the form type whose name is 'name'
     *
     * @param name the name of the form type to update
     * @param fields the new fields of the form type
     * @param requires the new requires of the form type
     * @param tagValues the new tag values of the form type
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields,
           String requires, Map<String, String> tagValues) throws VradiException {
        
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + name + ", fields, requires, tagValues)");
        }
        
        if (name != null) {
            try {
                WikittyExtension lastVersion = getFormType(name);
                String newVersion = null;
                
                if (lastVersion != null) {
                    newVersion = WikittyUtil.incrementMajorRevision(lastVersion.getVersion());
                } else {
                    newVersion = WikittyUtil.DEFAULT_VERSION;
                }
                
                WikittyExtension extension = new WikittyExtension(name, newVersion,
                        requires, new LinkedHashMap<String, FieldType>(fields));
                
                if (tagValues != null) {
                    for (Map.Entry<String, String> entry : tagValues.entrySet()) {
                        extension.addTagValue(entry.getKey(), entry.getValue());
                    }
                }
                
                List<WikittyExtension> extensions = Arrays.asList(extension);
                wikittyProxy.storeExtension(extensions);
                
                log.info(String.format("FormType named %s saved with id: %s and version: %s",
                        name, extension.getId(), newVersion));
                
                return extension;
            
            } catch (Exception eee) {
                if (log.isErrorEnabled()) {
                    log.error("Can't update form type", eee);
                }
                throw new VradiException("Can't update form type", eee);
            }
        }
        
        return null;
    }

    /**
     * Update the form type given in parameter
     *
     * @param extension the form type to update
     * @return the form type up to date
     */
    public WikittyExtension updateFormType(WikittyExtension extension, String templateName) throws VradiException {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType(" + extension.toDefinition() + ")");
        }
        try {

            if (templateName != null) {
                extension.addTagValue(FormTypeManager.TYPE_TEMPLATE, templateName);
            }

            Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();

            for (String fieldName : extension.getFieldNames()) {
                FieldType fieldType = extension.getFieldType(fieldName);
                fields.put(fieldName, fieldType);
            }

            WikittyExtension updated = updateFormType(extension.getName(), fields,
                    extension.getRequires(), extension.getTagValues());

            return updated;

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't update form type", eee);
            }
            throw new VradiException("Can't update form type", eee);
        }
    }

    /**
     * 
     * @param extensionName
     * @param templateName
     * @return
     */
    public File getTemplate(String extensionName, String templateName) {
        if (templateName == null) {
            return null;
        }
        File templatesDir = VradiServiceConfiguration.getInstance().getTemplatesDir();
        File template = new File(new File(templatesDir, extensionName), templateName);
        return template.exists() ? template : null;
    }

    /**
     * Lists the template files associated with the specified extension
     * @param extension
     * @return
     */
    @Deprecated
    public File[] getTemplates(WikittyExtension extension) {
        if(extension == null) {
            return null;
        }
        
        File templatesDir = VradiServiceConfiguration.getInstance().getTemplatesDir();
        File templateDir = new File(templatesDir, extension.getName());
        File[] result = templateDir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return !file.isDirectory() && !file.getName().endsWith(".xml");
            }
        });
        return result;
    }
    
    /**
     * Get the template file names associated with the specified extension.
     * 
     * @param extension extension
     * @return extension's template file names
     */
    public Collection<String> getTemplateFilenames(WikittyExtension extension) {
        
        Collection<String> filesNames = null;
        
        File templatesDir = VradiServiceConfiguration.getInstance().getTemplatesDir();
        File templateDir = new File(templatesDir, extension.getName());
        
        File[] result = templateDir.listFiles(new FileFilter() {
            // TODO EC20100518 maybe extension is always ".odt" ?
            @Override
            public boolean accept(File file) {
                return !file.isDirectory() && !file.getName().endsWith(".xml");
            }
        });

        if (result != null) {
            filesNames = new ArrayList<String>();
            for (File file : result) {
                filesNames.add(file.getName());
            }
        }
        return filesNames;
    }

    /**
     * Get association between template fields and extension fields.
     * 
     * @param extensionName
     * @param templateName
     * @return association map or {@code null} is there is no association yet
     * @throws VradiException
     */
    public Map<String, String> getAssociatedFields(String extensionName,
            String templateName) throws VradiException {

        File templatesDir = VradiServiceConfiguration.getInstance().getTemplatesDir();
        File extensionTemplateDir = new File(templatesDir, extensionName);
        File templateFieldsFile = new File(extensionTemplateDir, templateName + ".xml");

        // no association file found, return null
        if (!templateFieldsFile.isFile()) {
            return null;
        }

        // here file does exists, read contents
        Map<String, String> fieldMap = new HashMap<String, String>();
        Document document = null;
        try {
            SAXBuilder sxb = new SAXBuilder();
            document = sxb.build(templateFieldsFile);
            
            Element root = document.getRootElement();
            for(Object child : root.getChildren(FormTypeTemplateEnum.FIELD_ELEMENT.getValue())) {
                Element field = (Element) child;
                String templateField =
                        field.getAttributeValue(FormTypeTemplateEnum.FIELD_TEMPLATE_ATTRIBUTE.getValue());
                String extensionField =
                        field.getAttributeValue(FormTypeTemplateEnum.FIELD_EXTENSION_ATTRIBUTE.getValue());
                fieldMap.put(templateField, extensionField);
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Impossible d'associer les champs", eee);
            }
            throw new VradiException("Can't read association file", eee);
        }
        
        return fieldMap;
    }

    /**
     * Save association between template fields and user selected fields
     * into external xml field next to template.
     * 
     * Xml file is created a first call.
     * 
     * @param extensionName
     * @param templateName
     * @param fieldMap
     * @throws VradiException
     */
    public void setAssociatedFields(String extensionName, String templateName,
            Map<String, String> fieldMap) throws VradiException {
        
        File templatesDir = VradiServiceConfiguration.getInstance().getTemplatesDir();
        File templateDir = new File(templatesDir, extensionName);
        templateDir.mkdirs();
        File targetFile = new File(templateDir, templateName + ".xml");

        // load content
        Document document = null;
        try {
            
            // load or create dom tree
            if (targetFile.isFile()) {
                SAXBuilder sxb = new SAXBuilder();
                document = sxb.build(targetFile);
            }
            else {
                Element root = new Element(FormTypeTemplateEnum.FIELDS_ELEMENT.getValue());
                document = new Document(root);
            }
            
            // write new content
            Element root = document.getRootElement();
            root.removeChildren(FormTypeTemplateEnum.FIELD_ELEMENT.getValue());
            
            for(Map.Entry<String, String> field : fieldMap.entrySet()) {
                Element child = new Element(FormTypeTemplateEnum.FIELD_ELEMENT.getValue());
                child.setAttribute(FormTypeTemplateEnum.FIELD_TEMPLATE_ATTRIBUTE.getValue(),
                        field.getKey());
                if(field.getValue() != null) {
                    child.setAttribute(FormTypeTemplateEnum.FIELD_EXTENSION_ATTRIBUTE.getValue(),
                            field.getValue());
                }
                root.addContent(child);
            }
            
            XMLOutputter xmlOuputter = new XMLOutputter();
            Writer writer = new FileWriter(targetFile);
            xmlOuputter.output(document, writer);

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't save extension fields template association", eee);
            }
            throw new VradiException("Can't save extension fields template association", eee);
        }
    }

    /**
     * Check if a form type name already exists.
     * 
     * @param formTypeName form type name
     * @return new created wikitty extension
     * @throws VradiException if name is not valid
     */
    public boolean isFormTypeExists(String formTypeName) throws VradiException {
        boolean result = false;
        WikittyExtension extension = wikittyProxy.restoreExtensionLastVersion(formTypeName);
        if (extension != null) {
            result = true;
        }
        return result;
    }

    /**
     * Create new form type.
     * 
     * @param formTypeName form type name
     * @return new created wikitty extension
     * @throws VradiException if name is not valid
     */
    public WikittyExtension createFormType(String formTypeName) throws VradiException {

        // check name characters
        if (formTypeName == null) {
            throw new VradiException("Can't create new form type with null name !");
        }
        
        // check name characters
        if (!formTypeName.matches("\\w+")) {
            throw new VradiException("Form name contains invalid characters : " + formTypeName);
        }
        
        // check unicity
        if (isFormTypeExists(formTypeName)) {
            throw new VradiException("Form name \"" + formTypeName + "\" already exists !");
        }

        WikittyExtension formType = null;

        try {
            formType = new WikittyExtension(formTypeName,
                    WikittyUtil.DEFAULT_VERSION,
                    Form.EXT_FORM, new LinkedHashMap<String, FieldType>());
    
            wikittyProxy.storeExtension(formType);
        }
        catch (WikittyException ex) {
            throw new VradiException("Can't create form type", ex);
        }
        
        return formType;
    }

    public String getFormTypeTemplateName(WikittyExtension extension) {
        String result = extension == null ?
                null : extension.getTagValue(TYPE_TEMPLATE);
        return result;
    }
}
