/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package com.jurismarches.vradi.services;

import com.jurismarches.vradi.entities.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdom.Document;
import org.jdom.input.SAXBuilder;
import org.nuiton.util.MD5;
import org.sharengo.exceptions.TechnicalException;
import org.sharengo.wikitty.*;
import org.sharengo.wikitty.jdbc.WikittyExtensionStorageJDBC;
import org.sharengo.wikitty.jdbc.WikittyServiceJDBC;
import org.sharengo.wikitty.search.Element;
import org.sharengo.wikitty.search.Search;

import java.math.BigDecimal;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author morin
 */
public class VradiStorageServiceImpl implements VradiStorageService {

    static private final Log log = LogFactory.getLog(VradiStorageServiceImpl.class);

    protected WikittyProxy proxy;
    protected static String fieldPattern = "[\\p{Graph}\\p{Blank}\\p{L}\\p{Lu}&&[^\\(\\)(AND)(OR)]]+:";
    protected static String valuePattern = "[\\p{Graph}\\p{Blank}\\p{L}\\p{Lu}&&[^\\(\\)]]+";
    protected static String notPattern = "(NOT\\p{Blank})?";
    protected static Pattern queryPattern = Pattern.compile(
            "(" + fieldPattern + ")?" +
                    "(" +
                    "\\(" + notPattern + valuePattern +
                    "(" +
                    "\\p{Blank}((AND)|(OR))\\p{Blank}" + notPattern + valuePattern +
                    ")*" + "\\)" +
                    ")" +
                    "(\\p{Blank}((AND)|(OR))\\p{Blank}" +
                    "(" + fieldPattern + ")?" +
                    "(" +
                    "\\(" + notPattern + valuePattern +
                    "(" +
                    "\\p{Blank}((AND)|(OR))\\p{Blank}" + notPattern + valuePattern +
                    ")*" + "\\)" +
                    ")" +
                    ")*?");
    public static String ROOT_THESAURUS_NAME = "Thesaurus";

    protected static String item = "item";
    protected static String channel = "channel";

    public static SimpleDateFormat RSS_DATE_FORMAT = new SimpleDateFormat("E, d MMM yyyy HH:mm:ss Z", java.util.Locale.US);

    protected List<String> queriesInTheDB = null;

    public VradiStorageServiceImpl() {
        proxy = new WikittyProxy();
        proxy.setWikittyService(new WikittyServiceJDBC());
    }

    @Override
    public BusinessEntity getEntity(String id, Class clazz) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getEntity");
        }
        BusinessEntity result = null;
        if (id != null) {
            result = proxy.restore(clazz, id);
        }
        return result;
    }

    @Override
    public BusinessEntity updateEntity(BusinessEntity entity) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateEntity");
        }
        if (entity != null) {
            if (QueryMaker.class.isAssignableFrom(entity.getClass()) && queriesInTheDB != null) {
                Set<String> queries = ((QueryMaker) entity).getQueries();
                if (queries != null) {
                    for (String query : queries) {
                        if (!queriesInTheDB.contains(query)) {
                            queriesInTheDB.add(query);
                        }
                    }
                }
            }
            entity.setWikittyVersion(
                    WikittyUtil.incrementMajorRevision(entity.getWikittyVersion()));
            BusinessEntity result = proxy.store(entity);
            return result;
        }
        return null;
    }

    @Override
    public void deleteEntity(String id) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("deleteEntity");
        }
        if (id != null) {
            proxy.delete(id);
        }
    }

    @Override
    public User getUser(String userId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getUser");
        }
        User result = null;
        if (userId != null) {
            result = proxy.restore(User.class, userId);
        }
        return result;
    }

    @Override
    public Client getClient(String clientId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getClient");
        }
        Client result = null;
        if (clientId != null) {
            result = proxy.restore(Client.class, clientId);
            //TC-20091103 : corrige un possible NPE si le client n'est pas trouve...
            proxy.restore(User.class, new ArrayList<String>(result.getUser()));
        }

        return result;
    }

    @Override
    public Group getGroup(String groupId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroup");
        }
        Group result = null;
        if (groupId != null) {
            result = proxy.restore(Group.class, groupId);
        }
        return result;
    }

    @Override
    public List<User> getGroupUsers(String groupId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupUsers");
        }
        List<User> result = new ArrayList<User>();
        Group group = getGroup(groupId);
        if (group != null) {
            for (String userId : group.getUser()) {
                User user = getUser(userId);
                result.add(user);
            }
        }
        return result;
    }

    @Override
    public List<User> getClientUsers(String clientId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getClientUsers");
        }
        List<User> result = new ArrayList<User>();
        Client client = getClient(clientId);
        if (client != null) {
            for (String userId : client.getUser()) {
                User user = getUser(userId);
                result.add(user);
            }
        }
        return result;
    }

    @Override
    public List<Client> getGroupClients(String groupId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupClients");
        }
        List<Client> result = new ArrayList<Client>();
        Group group = getGroup(groupId);
        if (group != null) {
            for (String clientId : group.getClient()) {
                Client client = getClient(clientId);
                result.add(client);
            }
        }
        return result;
    }

    @Override
    public Client getClientByUserId(String userId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getClientByUserId");
        }
        Criteria criteria = Search.query().eq(Client.FQ_FIELD_USER, userId).criteria();
        return proxy.findAllByCriteria(Client.class, criteria).get(0);
    }

    @Override
    public List<Group> getGroupsByUserId(String userId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupsByUserId");
        }
        Criteria criteria = Search.query().eq(Group.FQ_FIELD_USER, userId).criteria();
        return proxy.findAllByCriteria(Group.class, criteria).getAll();
    }

    @Override
    public List<Group> getGroupsByClientId(String clientId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getGroupsByClientId");
        }
        Criteria criteria = Search.query().eq(Group.FQ_FIELD_CLIENT, clientId).criteria();
        return proxy.findAllByCriteria(Group.class, criteria).getAll();
    }

    @Override
    public List<Client> getAllClients() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllClients");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Client.EXT_CLIENT).criteria();
        return new ArrayList<Client>(proxy.findAllByCriteria(Client.class, criteria).getAll());
    }

    @Override
    public List<User> getAllUsers() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllUsers");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, User.EXT_USER).criteria();
        return new ArrayList<User>(proxy.findAllByCriteria(User.class, criteria).getAll());
    }

    @Override
    public List<Group> getAllGroups() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllGroups");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Group.EXT_GROUP).criteria();
        return new ArrayList<Group>(proxy.findAllByCriteria(Group.class, criteria).getAll());
    }

    /**
     * Saves a form
     *
     * @param form the form to insert or update
     * @return the form stored
     */
    @Override
    public Form updateForm(Form form) {
        if (log.isDebugEnabled()) {
            log.debug("updateForm");
        }
        if (form != null) {
            if (form.getDatePub() == null) {
                form.setDatePub(new Date());
            }
            form.setWikittyVersion(
                    WikittyUtil.incrementMajorRevision(form.getWikittyVersion()));
            return proxy.store(form);
        }
        return null;
    }

    /**
     * Deletes a Form
     *
     * @param formId the id of the Form to delete
     */
    @Override
    public void deleteForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("deleteForm");
        }
        if (formId != null) {
            proxy.delete(formId);
        }
    }

    /**
     * Find the forms filtered by the parameter query
     *
     * @param query the query to filter the forms
     * @return the forms filtered by query
     */
    @Override
    public List<Form> findForms(String query) {
        if (log.isDebugEnabled()) {
            log.debug("findForms");
        }
        Search search = Search.query();
        Map<String, String> caught = parseQuery(query);
        if (caught != null) {
            createQuery(search, caught, "_search" + (caught.size() - 1) + "_", null);
        }
        Criteria criteria = search.criteria();
        PagedResult<Form> result = proxy.findAllByCriteria(Form.class, criteria);
        return new ArrayList<Form>(result.getAll());
    }

    /**
     * Find the forms filtered by the parameter query
     *
     * @param query     the query to filter the forms
     * @param beginDate
     * @param endDate
     * @return the forms filtered by query
     */
    @Override
    public List<Form> findForms(String query, WikittyExtension extension,
                                String dateType, Date beginDate, Date endDate, List[] thesaurus) {
        if (log.isDebugEnabled()) {
            log.debug("findForms");
        }
        Search search = Search.query();
        Map<String, String> caught = parseQuery(query);
        if (caught != null) {
            createQuery(search, caught, "_search" + (caught.size() - 1) + "_", null);
        }
        if (extension != null) {
            search.eq(Element.ELT_EXTENSION, extension.getName());
        }
        if (dateType != null) {
            if (beginDate != null && endDate != null) {
                log.info(dateType);
                String beginDateS = WikittyUtil.solrDateFormat.format(beginDate);
                String endDateS = WikittyUtil.solrDateFormat.format(endDate);
                if (!beginDate.equals(endDate)) {
                    search.bw(dateType, beginDateS, endDateS + "+1DAY");
                } else {
                    search.bw(dateType, beginDateS, beginDateS + "+1DAY");
                }
            }
        }
        if (thesaurus != null) {
            for (int i = 0; i < thesaurus.length; i++) {
                if (thesaurus[i] != null) {
                    Search subSearch = search.or();
                    for (Object th : thesaurus[i]) {
                        subSearch.eq(Form.FQ_FIELD_THESAURUS, (String) th);
                    }
                }
            }
        }
        Criteria criteria = search.criteria();
        PagedResult<Form> result = proxy.findAllByCriteria(Form.class, criteria);
        return new ArrayList<Form>(result.getAll());
    }

    @Override
    public List<String> getQueriesReturningForm(Form form) throws TechnicalException {
        List<String> result = new ArrayList<String>();
        if (queriesInTheDB == null) {
            List<QueryMaker> queryMakers = new ArrayList<QueryMaker>();
            queryMakers.addAll(getAllClients());
            queryMakers.addAll(getAllGroups());
            queryMakers.addAll(getAllUsers());

            queriesInTheDB = new ArrayList<String>();
            for (QueryMaker queryMaker : queryMakers) {
                if (queryMaker.getQueries() != null) {
                    for (String query : queryMaker.getQueries()) {
                        if (!queriesInTheDB.contains(query)) {
                            queriesInTheDB.add(query);
                        }
                    }
                }
            }
        }
        for (String query : queriesInTheDB) {
            Map<String, String> caught = parseQuery(query);
            if (caught != null) {
                //createQuery(search, caught, "_search" + (caught.size() - 1) + "_", null);
                for (Map.Entry<String, String> stringStringEntry : caught.entrySet()) {
                    log.info(stringStringEntry.getKey() + " : " + stringStringEntry.getValue());
                }
                Boolean formFound = isQueryReturningForm(caught, "_search" + (caught.size() - 1) + "_", null, form);
                if (formFound != null && formFound) {
                    result.add(query);
                }
            }
        }
        return result;
    }

    protected Map<String, String> parseQuery(String query) {
        if (query != null && !query.isEmpty()) {
            String s = query;
            boolean hasMatched = true;
            int k = 0;
            //map of the query elements found
            Map<String, String> caught = new HashMap<String, String>();
            //while the matcher matches the pattern
            while (hasMatched) {
                hasMatched = false;
                Matcher m = queryPattern.matcher(s);
                while (m.find()) {
                    hasMatched = true;
                    //"_search" + k + "_" is a kind of id for the pattern detected
                    caught.put("_search" + k + "_", m.group(0).trim());
                    //replace the pattern by the id
                    s = s.replace(m.group(0).trim(), "_search" + k++ + "_");
                }
            }
            caught.put("_search" + k + "_", "(" + s + ")");
            return caught;
        }
        return null;
    }

    /**
     * Get all the form types
     *
     * @return the list of the wikittyExtension representing the form types
     */
    @Override
    public List<WikittyExtension> getAllFormTypes() {
        if (log.isDebugEnabled()) {
            log.debug("getAllFormTypes");
        }
        List<WikittyExtension> result = new ArrayList<WikittyExtension>();
//        List<String> extNames = new ArrayList<String>();
        WikittyService wikittyService = proxy.getWikittyService();
        result.addAll(wikittyService.getAllExtensions(true));
        List<WikittyExtension> formTypes = new ArrayList<WikittyExtension>();
        for (WikittyExtension extension : result) {
            if (extension != null && Form.EXT_FORM.equals(extension.getRequires())) {
                formTypes.add(extension);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Number of FormType found : " + result.size());
        }
        return formTypes;
    }

    /**
     * Get the last version of the extension named name
     *
     * @param name the name of the extension
     * @return the last version of the extension
     */
    @Override
    public WikittyExtension getFormType(String name) {
        if (log.isDebugEnabled()) {
            log.debug("getFormType");
        }
        if (name != null) {
            WikittyExtension result = null;
            String extLastVersion = null;
            List<String> extIds = proxy.getWikittyService().getAllExtensionIds();
            for (String extId : extIds) {
                if (name.equals(WikittyExtension.computeName(extId))) {
                    String extVersion = WikittyExtension.computeVersion(extId);
                    if (WikittyUtil.versionGreaterThan(extVersion, extLastVersion)) {
                        extLastVersion = extVersion;
                    }
                }
            }
            if (extLastVersion != null) {
                result = proxy.getWikittyService()
                        .restoreExtension(WikittyExtension.computeId(name, extLastVersion));
                return result;
            } else {
                return null;
            }
        } else {
            return null;
        }
    }

    /**
     * Get the fields of a form type
     *
     * @param name the name of the field type
     * @return a map containing the names of the fields and their type
     */
    @Override
    public Map<String, FieldType> getFormTypeFields(String name) {
        if (log.isDebugEnabled()) {
            log.debug("getFormTypeFields");
        }
        WikittyExtension ext = getFormType(name);
        Map<String, FieldType> result = new HashMap<String, FieldType>();
        for (String fieldName : ext.getFieldNames()) {
            //FieldTypeEnum fieldType = new FieldTypeEnum()
            result.put(fieldName, ext.getFieldType(fieldName));
        }

        return result;
    }

    /**
     * Inserts a form type. If the form type is modified, then the version
     * is automaticaly incremented.
     *
     * @param name   the name of the form type
     * @param fields the fields of the form type
     * @return the WikittyExtension corresponding to the form type
     */
    @Override
    public WikittyExtension updateFormType(String name, Map<String, FieldType> fields, String requires) {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType");
        }
        if (name != null) {
            WikittyExtension result = null;
            WikittyExtension lastVersion = getFormType(name);
            if (log.isDebugEnabled()) {
                log.debug("lastVersion : " + lastVersion);
            }
            if (lastVersion != null) {
                result = new WikittyExtension(name,
                        WikittyUtil.incrementMajorRevision(lastVersion.getVersion()),
                        requires, (LinkedHashMap) fields);
            } else {
                result = new WikittyExtension(name, WikittyUtil.DEFAULT_VERSION,
                        requires, (LinkedHashMap) fields);
            }
            List<WikittyExtension> extensions = Arrays.asList(result);
            proxy.getWikittyService().storeExtension(extensions);
            if (log.isDebugEnabled()) {
                log.debug("formType : " + result.getId());
            }
            return result;
        } else {
            return null;
        }
    }

    /**
     * Inserts a form type. If the form type is modified, then the version
     * is automaticaly incremented.
     *
     * @param extention to store
     * @return the WikittyExtension corresponding to the form type
     */
    @Override
    public WikittyExtension updateFormType(WikittyExtension extention) {
        if (log.isDebugEnabled()) {
            log.debug("updateFormType");
        }
        Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
        for (String fieldName : extention.getFieldNames()) {
            log.debug(fieldName);
            fields.put(fieldName, extention.getFieldType(fieldName));
        }

        return updateFormType(extention.getName(), fields, extention.getRequires());
    }

    /**
     * Get the form whose id is id.
     *
     * @param formId the id of the form
     * @return the form whose id is id
     */
    @Override
    public Form getForm(String formId) {
        if (log.isDebugEnabled()) {
            log.debug("getForm");
        }
        Form result = null;
        if (formId != null) {
            result = proxy.restore(Form.class, formId);
        }
        return result;
    }

    /**
     * Get all the forms recorded
     *
     * @return the list of all the forms
     */
    @Override
    public List<Form> getAllForms() {
        if (log.isDebugEnabled()) {
            log.debug("getAllForms");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Form.EXT_FORM).criteria();
        return new ArrayList<Form>(proxy.findAllByCriteria(Form.class, criteria).getAll());
    }

    /**
     * Add the AND or OR to the search
     *
     * @param search the Search instance to add the conditions
     * @param caught the map of the patterns with their id
     * @param sss    the id of the pattern matched
     * @param field  the field which must (or not) be equals to the values
     */
    protected void createQuery(Search search, Map<String, String> caught,
                               String sss, String field) {
        if (log.isDebugEnabled()) {
            log.debug("createQuery");
        }
        String si = caught.get(sss);
        String expression;
        if (si.startsWith("(")) {
            expression = si.substring(1, si.length() - 1);
        } else {
            field = si.substring(0, si.indexOf(":")).replace(" ", WikittyExtensionStorageJDBC.WORD_SEPARATOR);
            expression = si.substring(si.indexOf("(") + 1, si.length() - 1);
        }
        String[] orSplitted = expression.split(" OR ");
        if (orSplitted.length > 1) {
            addTokens(search.or(), orSplitted, field, caught);
        } else {
            String[] andSplitted = expression.split(" AND ");
            if (andSplitted.length > 1) {
                addTokens(search.and(), andSplitted, field, caught);
            } else {
                addTokens(search, new String[]{expression}, field, caught);
            }
        }
    }

    /**
     * Add the equals or not equals conditions to the search
     *
     * @param search the Search instance to add the conditions
     * @param tokens the list of values that must equals or not the search
     * @param field  the field which must (or not) be equals to the values
     * @param caught the map of the patterns with their id
     */
    protected void addTokens(Search search, String[] tokens, String field,
                             Map<String, String> caught) {
        if (log.isDebugEnabled()) {
            log.debug("equalsFieldValue");
        }
        for (String token : tokens) {
            if (token.trim().startsWith("NOT ")) {
                Search subSearch = search.and();
                subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                        + field + Criteria.SEPARATOR
                        + FieldType.TYPE.STRING, "*" + token.trim().substring(4) + "*");
                try {
                    subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + FieldType.TYPE.DATE,
                            WikittyUtil.solrDateFormat.format(
                                    DateFormat.getInstance().parse(token.trim().substring(4))));
                } catch (ParseException eee) {
                    if (log.isDebugEnabled()) {
                        log.debug(token + " cannot be a date.");
                    }
                }
                try {
                    Boolean.parseBoolean(token.trim().substring(4));
                    subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + FieldType.TYPE.BOOLEAN, token.trim().substring(4));
                } catch (Exception eee) {
                    if (log.isDebugEnabled()) {
                        log.debug(token + " cannot be a boolean.");
                    }
                }
                try {
                    Double.valueOf(token.trim().substring(4));
                    subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + FieldType.TYPE.NUMERIC, token.trim().substring(4));
                } catch (NumberFormatException eee) {
                    if (log.isDebugEnabled()) {
                        log.debug(token + " cannot be a number.");
                    }
                }
                subSearch.neq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                        + field + Criteria.SEPARATOR
                        + FieldType.TYPE.WIKITTY, token.trim().substring(4));
            } else {
                if (caught.containsKey(token.trim())) {
                    createQuery(search, caught, token.trim(), field);
                } else {
                    Search subSearch = search.or();
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + FieldType.TYPE.STRING, "*" + token.trim() + "*");
                    try {
                        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + FieldType.TYPE.DATE,
                                WikittyUtil.solrDateFormat.format(
                                        DateFormat.getInstance().parse(token.trim())));
                    } catch (ParseException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a date.");
                        }
                    }
                    try {
                        Boolean.parseBoolean(token.trim());
                        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + FieldType.TYPE.BOOLEAN, token.trim());
                    } catch (Exception eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a boolean.");
                        }
                    }
                    try {
                        Double.valueOf(token.trim());
                        subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                                + field + Criteria.SEPARATOR
                                + FieldType.TYPE.NUMERIC, token.trim());
                    } catch (NumberFormatException eee) {
                        if (log.isDebugEnabled()) {
                            log.debug(token + " cannot be a number.");
                        }
                    }
                    subSearch.eq(Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR
                            + field + Criteria.SEPARATOR
                            + FieldType.TYPE.WIKITTY, token.trim());
                }
            }
        }
    }

    /**
     * Add the AND or OR to the search
     *
     * @param caught the map of the patterns with their id
     * @param sss    the id of the pattern matched
     * @param field  the field which must (or not) be equals to the values
     */
    protected Boolean isQueryReturningForm(Map<String, String> caught,
                                           String sss, String field, Form form) {
        Boolean result;
        if (log.isDebugEnabled()) {
            log.debug("createQuery");
        }
        String si = caught.get(sss);
        String expression;
        if (si.startsWith("(")) {
            expression = si.substring(1, si.length() - 1);
        } else {
            field = si.substring(0, si.indexOf(":")).replace(" ", WikittyExtensionStorageJDBC.WORD_SEPARATOR);
            expression = si.substring(si.indexOf("(") + 1, si.length() - 1);
        }
        String[] orSplitted = expression.split(" OR ");
        if (orSplitted.length > 1) {
            result = equalsFieldValue(orSplitted, field, caught, form, false);
        } else {
            String[] andSplitted = expression.split(" AND ");
            if (andSplitted.length > 1) {
                result = equalsFieldValue(andSplitted, field, caught, form, true);
            } else {
                result = equalsFieldValue(new String[]{expression}, field, caught, form, true);
            }
        }
        return result;
    }

    /**
     * Add the equals or not equals conditions to the search
     *
     * @param tokens the list of values that must equals or not the search
     * @param field  the field which must (or not) be equals to the values
     * @param caught the map of the patterns with their id
     */
    protected Boolean equalsFieldValue(String[] tokens, String field,
                                       Map<String, String> caught, Form form, Boolean and) {
        if (log.isDebugEnabled()) {
            log.debug("equalsFieldValue");
        }
        Boolean result = null;
        Map<String, FieldType> extensions = new HashMap<String, FieldType>();
        if (form != null)
            for (WikittyExtension ext : form.getExtensions()) {
                if (ext.getFieldType(field) != null) {
                    extensions.put(ext.getName(), ext.getFieldType(field));
                }
            }
        for (String token : tokens) {
            Boolean tokenResult = null;
            if (token.trim().startsWith("NOT ")) {
                for (String extensionName : extensions.keySet()) {
                    switch (extensions.get(extensionName).getType()) {
                        case DATE:
                            Date formDate = (Date) form.getField(extensionName, field);
                            try {
                                Date queryDate = DateFormat.getInstance().parse(token.trim().substring(4));
                                tokenResult = !formDate.equals(queryDate);
                            } catch (ParseException eee) {
                                if (log.isDebugEnabled()) {
                                    log.debug(token + " cannot be a date.");
                                }
                            }
                            break;
                        case BOOLEAN:
                            Boolean formBoolean = (Boolean) form.getField(extensionName, field);
                            try {
                                Boolean queryBoolean = Boolean.parseBoolean(token.trim().substring(4));
                                tokenResult = !formBoolean.equals(queryBoolean);
                            } catch (Exception eee) {
                                if (log.isDebugEnabled()) {
                                    log.debug(token + " cannot be a boolean.");
                                }
                            }
                            break;
                        case NUMERIC:
                            BigDecimal formNumber = (BigDecimal) form.getField(extensionName, field);
                            try {
                                BigDecimal queryNumber = BigDecimal.valueOf(
                                        Double.valueOf(token.trim().substring(4)));
                                tokenResult = !formNumber.equals(queryNumber);

                            } catch (NumberFormatException eee) {
                                if (log.isDebugEnabled()) {
                                    log.debug(token + " cannot be a number.");
                                }
                            }
                            break;
                        case WIKITTY:
                            String formString = (String) form.getField(extensionName, field);
                            String queryString = token.trim().substring(4);
                            tokenResult = !formString.equals(queryString);
                            break;
                        default:
                            formString = (String) form.getField(extensionName, field);
                            queryString = token.trim().substring(4);
                            tokenResult = !formString.contains(queryString);
                    }
                }
            } else {
                if (caught.containsKey(token.trim())) {
                    tokenResult = isQueryReturningForm(caught, token.trim(), field, form);
                } else {
                    for (String extensionName : extensions.keySet()) {
                        switch (extensions.get(extensionName).getType()) {
                            case DATE:
                                Date formDate = (Date) form.getField(extensionName, field);
                                try {
                                    Date queryDate = DateFormat.getInstance().parse(token.trim());
                                    tokenResult = formDate.equals(queryDate);
                                } catch (ParseException eee) {
                                    if (log.isDebugEnabled()) {
                                        log.debug(token + " cannot be a date.");
                                    }
                                }
                                break;
                            case BOOLEAN:
                                Boolean formBoolean = (Boolean) form.getField(extensionName, field);
                                try {
                                    Boolean queryBoolean = Boolean.parseBoolean(token.trim());
                                    tokenResult = formBoolean.equals(queryBoolean);
                                } catch (Exception eee) {
                                    if (log.isDebugEnabled()) {
                                        log.debug(token + " cannot be a boolean.");
                                    }
                                }
                                break;
                            case NUMERIC:
                                BigDecimal formNumber = (BigDecimal) form.getField(extensionName, field);
                                try {
                                    BigDecimal queryNumber = BigDecimal.valueOf(
                                            Double.valueOf(token.trim()));
                                    tokenResult = formNumber.equals(queryNumber);

                                } catch (NumberFormatException eee) {
                                    if (log.isDebugEnabled()) {
                                        log.debug(token + " cannot be a number.");
                                    }
                                }
                                break;
                            case WIKITTY:
                                String formString = (String) form.getField(extensionName, field);
                                String queryString = token.trim();
                                tokenResult = formString.equals(queryString);
                                break;
                            default:
                                formString = (String) form.getField(extensionName, field);
                                queryString = token.trim();
                                tokenResult = formString.contains(queryString);
                        }
                    }
                }
            }
            if (result == null) {
                result = tokenResult;
            } else if (and) {
                result = result && tokenResult;
            } else {
                result = result || tokenResult;
            }
        }
        return result;
    }

    @Override
    public XmlStream updateXmlStream(XmlStream stream) {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlStream");
        }
        if (stream != null) {
            return proxy.store(stream);
        }
        return null;
    }

    @Override
    public void deleteXmlStream(String streamId) {
        if (log.isDebugEnabled()) {
            log.debug("deleteXmlStream");
        }
        if (streamId != null) {
            proxy.delete(streamId);
        }
    }

    @Override
    public List<XmlStream> getAllXmlStreams() {
        if (log.isDebugEnabled()) {
            log.debug("getAllXmlStreams");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, XmlStream.EXT_XMLSTREAM).criteria();
        return new ArrayList<XmlStream>(proxy.findAllByCriteria(XmlStream.class, criteria).getAll());
    }

    @Override
    public List<XmlFieldBinding> updateXmlFieldBindings(List<XmlFieldBinding> bindings) {
        if (log.isDebugEnabled()) {
            log.debug("updateXmlFieldBindings");
        }
        if (bindings != null) {
            List<XmlFieldBinding> result = new ArrayList<XmlFieldBinding>();
            for (XmlFieldBinding binding : bindings) {
                if (binding != null) {
                    result.add(binding);
                }
            }
            result = proxy.store(result);
            return new ArrayList<XmlFieldBinding>(result);
        }
        return null;
    }

    @Override
    public TreeNode getRootThesaurus() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getRootThesaurus");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE).eq(TreeNode.FQ_FIELD_NAME, ROOT_THESAURUS_NAME).criteria();
        List<TreeNode> rootThesaurus = proxy.findAllByCriteria(TreeNode.class, criteria).getAll();

        log.debug("Thesaurus find " + rootThesaurus);
        if (rootThesaurus.isEmpty()) {
            TreeNode thesaurusImpl = new TreeNodeImpl();
            thesaurusImpl.setName(ROOT_THESAURUS_NAME);
            updateThesaurus(thesaurusImpl);

            return thesaurusImpl;
        }
        return rootThesaurus.get(0);
    }

    @Override
    public List<TreeNode> getAllThesaurus() throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getAllThesaurus");
        }
        Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, TreeNode.EXT_TREENODE).criteria();

        List<TreeNode> thesaurus = proxy.findAllByCriteria(TreeNode.class, criteria).getAll();
        log.debug(thesaurus.size() + " thesaurus found " + thesaurus);
        return new ArrayList<TreeNode>(thesaurus);
    }

    @Override
    public TreeNode getThesaurus(String ThesaurusId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("getThesaurus");
        }
        TreeNode result = null;
        if (ThesaurusId != null) {
            result = proxy.restore(TreeNode.class, ThesaurusId);
        }
        return result;
    }

    @Override
    public TreeNode updateThesaurus(TreeNode thesaurus) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("updateThesaurus");
        }
        if (thesaurus != null) {
            thesaurus.setWikittyVersion(
                    WikittyUtil.incrementMajorRevision(thesaurus.getWikittyVersion()));
            return proxy.store(thesaurus);
        }
        return null;
    }

    @Override
    public void deleteThesaurus(String thesaurusId) throws TechnicalException {
        if (log.isDebugEnabled()) {
            log.debug("deleteThesaurus");
        }
        if (thesaurusId != null) {
            proxy.delete(thesaurusId);
        }
    }

    @Override
    public XmlStream getXmlStream(String xmlStreamId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlStream");
        }
        XmlStream result = null;
        if (xmlStreamId != null) {
            result = proxy.restore(XmlStream.class, xmlStreamId);
        }
        return result;
    }

    @Override
    public XmlFieldBinding getXmlFieldBinding(String xmlFieldBindingId) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBinding");
        }
        XmlFieldBinding result = null;
        if (xmlFieldBindingId != null) {
            result = proxy.restore(XmlFieldBinding.class, xmlFieldBindingId);
        }
        return result;
    }

    @Override
    public List<XmlFieldBinding> getXmlFieldBindings(XmlStream xmlStream) {
        if (log.isDebugEnabled()) {
            log.debug("getXmlFieldBindings");
        }
        if (xmlStream != null && xmlStream.getXmlFieldBinding() != null) {
            List<String> bindings = new ArrayList<String>();
            bindings.addAll(xmlStream.getXmlFieldBinding());
            return proxy.restore(XmlFieldBinding.class, bindings);
        }
        return null;
    }

    @Override
    public void bindFormsToClients() {
        List<Sending> toSend = new ArrayList<Sending>();
        try {
            List<QueryMaker> queryMakers = new ArrayList<QueryMaker>();
            List<Client> clients = getAllClients();
            for (Client client : clients) {
                Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                        .eq(Sending.FQ_FIELD_CLIENT, client.getWikittyId()).criteria();
                List<Sending> sendings = proxy.findAllByCriteria(Sending.class, criteria).getAll();
                List<String> alreadySentFormIds = new ArrayList<String>();
                for (Sending sending : sendings) {
                    alreadySentFormIds.add(sending.getForm());
                }
                List<Form> alreadySentForms = proxy.restore(Form.class, alreadySentFormIds);

                queryMakers.add(client);
                List<User> users = getClientUsers(client.getWikittyId());
                queryMakers.addAll(users);
                for (QueryMaker queryMaker : queryMakers) {
                    for (String query : queryMaker.getQueries()) {
                        List<Form> forms = findForms(query);
                        for (Form form : forms) {
                            if (!alreadySentForms.contains(form)) {
                                alreadySentForms.add(form);
                                Sending sending = new SendingImpl();
                                sending.setClient(client.getWikittyId());
                                sending.setForm(form.getWikittyId());
                                sending.setSentDate(null);
                                sending.setReceptionDate(null);
                                toSend.add(sending);
                            }
                        }
                    }
                }
                queryMakers.clear();
            }
            proxy.store(toSend);

        } catch (TechnicalException e) {
            e.printStackTrace();
        }
    }

    @Override
    public Map<Client, List<Form>> getFormsByClients() throws TechnicalException {
        Map<Client, List<Form>> result = new HashMap<Client, List<Form>>();
        List<Client> clients = getAllClients();
        for (Client client : clients) {
            List<String> formIds = new ArrayList<String>();
            Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                    .eq(Sending.FQ_FIELD_CLIENT, client.getWikittyId()).criteria();
            List<Sending> sendings = proxy.findAllByCriteria(Sending.class, criteria).getAll();
            for (Sending sending : sendings) {
                formIds.add(sending.getForm());
            }
            result.put(client, proxy.restore(Form.class, formIds));
        }
        return result;
    }

    @Override
    public Map<Form, List<Client>> getClientsByForms() {
        Map<Form, List<Client>> result = new HashMap<Form, List<Client>>();
        List<Form> forms = getAllForms();
        for (Form form : forms) {
            List<String> clientIds = new ArrayList<String>();
            Criteria criteria = Search.query().eq(Element.ELT_EXTENSION, Sending.EXT_SENDING)
                    .eq(Sending.FQ_FIELD_FORM, form.getWikittyId()).criteria();
            List<Sending> sendings = proxy.findAllByCriteria(Sending.class, criteria).getAll();
            for (Sending sending : sendings) {
                clientIds.add(sending.getForm());
            }
            result.put(form, proxy.restore(Client.class, clientIds));
        }
        return result;
    }

    @Override
    public String getFormsFromXmlStream(XmlStream xmlStream, String lastItemRecorded) {
        Map<String, Set<String>> streamBinding = new HashMap<String, Set<String>>();
        WikittyExtension formType = null;
        List<Form> forms = new ArrayList<Form>();
        log.info(xmlStream.getName());
        List<XmlFieldBinding> bindings = proxy.restore(XmlFieldBinding.class,
                new ArrayList<String>(xmlStream.getXmlFieldBinding()));
        for (XmlFieldBinding binding : bindings) {
            log.info(binding.getFormField());
            String formTypeName = binding.getFormField().substring(0, binding.getFormField().indexOf('.'));
            if (formType == null && !formTypeName.equals(Form.EXT_FORM)) {
                formType = getFormType(formTypeName);
            }
            streamBinding.put(binding.getFormField(), binding.getXmlField());
        }
        try {
            String result = null;
            SAXBuilder sxb = new SAXBuilder();
            URL rssUrl = new URL(xmlStream.getUrl());
            Document document = sxb.build(rssUrl);
            org.jdom.Element racine = document.getRootElement();
            List<org.jdom.Element> itemElt = racine.getChild(channel).getChildren(item);
            for (org.jdom.Element element : itemElt) {
                StringBuffer sb = new StringBuffer();
                List<org.jdom.Element> fields = element.getChildren();
                for (org.jdom.Element field : fields) {
                    sb.append(field.getText());
                }
                MD5 lastItem = new MD5(sb.toString());
                if (result == null) {
                    result = MD5.asHex(lastItem.Final());
                }
                if (lastItemRecorded != null
                        && lastItemRecorded.equals(MD5.asHex(lastItem.Final()))) {
                    break;
                }
                Form form = new FormImpl();
                form.addExtension(formType);
                for (String field : streamBinding.keySet()) {
                    int dot = field.indexOf('.');
                    String extName = field.substring(0, dot);
                    String fieldName = field.substring(dot + 1);
                    FieldType.TYPE fieldType = null;
                    if (extName.equals(Form.EXT_FORM)) {
                        fieldType = FormImpl.extensionForm.getFieldType(fieldName).getType();
                    } else if (formType != null) {
                        fieldType = formType.getFieldType(fieldName).getType();
                    }
                    if (fieldType != null && streamBinding.get(field) != null) {
                        switch (fieldType) {
                            case DATE:
                                for (String xmlField : streamBinding.get(field)) {
                                    form.setField(extName, fieldName,
                                            WikittyUtil.solrDateFormat.format(
                                                    RSS_DATE_FORMAT.parse(
                                                            element.getChild(xmlField).getText())));
                                }
                                break;
                            default:
                                for (String xmlField : streamBinding.get(field)) {
                                    form.setField(extName, fieldName,
                                            element.getChild(xmlField).getText());
                                }
                                break;
                        }
                    }
                }
                if (form.getName() == null) {
                    form.setName(form.getWikittyId());
                }
                forms.add(updateForm(form));
            }
            return result;
        } catch (Exception eee) {
            eee.printStackTrace();
        }
        return null;
    }

    public List<String> getRSSFields(String url) {
        List<String> result = new ArrayList<String>();
        try {
            SAXBuilder sxb = new SAXBuilder();
            URL rssUrl = new URL(url);
            Document document = sxb.build(rssUrl);
            org.jdom.Element racine = document.getRootElement();
            org.jdom.Element itemElt = racine.getChild(channel).getChild(item);
            for (org.jdom.Element elt : (List<org.jdom.Element>) itemElt.getChildren()) {
                result.add(elt.getName());
            }
        } catch (Exception eee) {
            eee.printStackTrace();
        }
        return result;
    }

    @Override
    /**
     * Changes the data directory
     * @param newDataDir the new data directory path
     * @param oldDataDir the old data directory path.
     *          If null, the data in the old directory will not be copied.
     */
    public void changeDataDir(String newDataDir, String oldDataDir) {
        proxy.changeDataDir(newDataDir, oldDataDir);
    }
}
