/*
 * #%L
 * SGQ :: Business
 * $Id: LabelErrorDAOImpl.java 446 2014-04-08 08:39:46Z echatellier $
 * $HeadURL: https://svn.codelutin.com/sgq-ch/tags/sgq-ch-1.1.6/sgq-business/src/main/java/com/herbocailleau/sgq/entities/LabelErrorDAOImpl.java $
 * %%
 * Copyright (C) 2012 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.entities;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.nuiton.topia.TopiaException;

public class LabelErrorDAOImpl<E extends LabelError> extends LabelErrorDAOAbstract<E> {

    /**
     * Renvoie le nombre d'erreur totales.
     * 
     * @throws TopiaException 
     */
    public long findLabelErrorsCount() throws TopiaException {
        String query = "select count(L) from " + LabelError.class.getName() +
                " L WHERE L." + LabelError.PROPERTY_MODIFIED + " = false" +
                " AND L." + LabelError.PROPERTY_DELETED + " = false";
        
        Number result = (Number)context.findUnique(query);
        return result.longValue();
    }

    /**
     * Recherche les erreurs par zone source ordonnée par la donnée.
     * 
     * @param source zone source
     * @throws TopiaException 
     */
    public List<LabelError> findLabelErrorsBySource(Zone source) throws TopiaException {
        String query = "from " + LabelError.class.getName() +
                " where " + LabelError.PROPERTY_SOURCE + " = :zone" +
                " AND " + LabelError.PROPERTY_MODIFIED + " = false" +
                " AND " + LabelError.PROPERTY_DELETED + " = false" +
                " order by " + LabelError.PROPERTY_LINE;
        
        List<LabelError> result = (List<LabelError>)context.findAll(query, "zone", source);
        return result;
    }

    /**
     * Recherche les erreurs par zone source ordonnée par la donnée.
     * 
     * @param beginDate begin date
     * @param endDate end date
     * @return count history
     * @throws TopiaException 
     */
    public List<LabelError> findAllLabelErrors(Date beginDate, Date endDate) throws TopiaException {
        String suffix = " order by L." + LabelError.PROPERTY_LABEL_DATE;
        List<LabelError> result = (List<LabelError>)findLabelErrors("", suffix, beginDate, endDate, 0, -1);
        return result;
    }

    /**
     * Recherche les erreurs par zone source ordonnée par la donnée.
     * 
     * @param beginDate begin date
     * @param endDate end date
     * @param offset offset
     * @param limit limit
     * @return count history
     * @throws TopiaException 
     */
    public List<LabelError> findLabelErrors(Date beginDate, Date endDate, int offset, int limit) throws TopiaException {
        String suffix = " order by L." + LabelError.PROPERTY_LABEL_DATE;
        List<LabelError> result = (List<LabelError>)findLabelErrors("", suffix, beginDate, endDate, offset, limit);
        return result;
    }
    
    /**
     * Recherche les erreurs par zone source ordonnée par la donnée.
     * 
     * @param beginDate begin date
     * @param endDate end date
     * @return count history
     * @throws TopiaException 
     */
    public long findLabelErrorsCount(Date beginDate, Date endDate) throws TopiaException {
        String prefix = "SELECT count(L) ";
        Number count = (Number)findLabelErrors(prefix, "", beginDate, endDate, -1, -1);
        return count.longValue();
    }

    /**
     * Recherche les erreurs par zone source ordonnée par la donnée.
     * 
     * @param prefix query prefix
     * @param suffix query suffix
     * @param beginDate begin date
     * @param endDate end date
     * @param offset offset
     * @param limit limit
     * @return count history
     * @throws TopiaException 
     */
    protected Object findLabelErrors(String prefix, String suffix, Date beginDate,
            Date endDate, int offset, int limit) throws TopiaException {
        String query = prefix + "from " + LabelError.class.getName() +
                " L where (L." + LabelError.PROPERTY_MODIFIED + " = true" +
                " OR L." + LabelError.PROPERTY_DELETED + " = true)";

        List<Object> params = new ArrayList<Object>();
        if (beginDate != null) {
            query += " AND :beginDate <= L." + LabelError.PROPERTY_LABEL_DATE;
            params.add("beginDate");
            params.add(beginDate);
        }
        if (endDate != null) {
            query += " AND L." + LabelError.PROPERTY_LABEL_DATE + " <= :endDate";
            params.add("endDate");
            params.add(endDate);
        }
        query += suffix;

        Object result;
        if (offset == -1) {
            result = context.findUnique(query, params.toArray());
        } else if (limit == -1) {
            result = context.findAll(query, params.toArray());
        } else {
            result = context.find(query, offset, offset + limit - 1, params.toArray()); 
        }
        return result;
    }

    /**
     * Lots probablement expirés et pour lesquels une étiquette a été émise.
     * 
     * @return batch in error
     * @throws TopiaException
     */
    public List<Batch> getBatchWithLabelAfterExpiration() throws TopiaException {
        String query = "SELECT DISTINCT " + LabelError.PROPERTY_AFTER_BATCH_EXPIRATION +
                " from " + LabelError.class.getName() +
                " where " + LabelError.PROPERTY_AFTER_BATCH_EXPIRATION + " != null" +
                " AND " + LabelError.PROPERTY_MODIFIED + " = false" +
                " AND " + LabelError.PROPERTY_DELETED + " = false";
        
        List<Batch> result = context.findAll(query);
        return result;
    }
} //LabelErrorDAOImpl<E extends LabelError>
