/*
 * #%L
 * SGQ :: Business
 * $Id: SgqBusinessConfig.java 428 2013-07-26 12:58:47Z echatellier $
 * $HeadURL: https://svn.codelutin.com/sgq-ch/tags/sgq-ch-1.1.6/sgq-business/src/main/java/com/herbocailleau/sgq/business/SgqBusinessConfig.java $
 * %%
 * Copyright (C) 2012 - 2013 Herboristerie Cailleau
 * %%
 * Herboristerie Cailleau - Tous droits réservés
 * #L%
 */

package com.herbocailleau.sgq.business;

import java.util.Properties;

import org.nuiton.util.config.ApplicationConfig;
import org.nuiton.util.config.ArgumentsParserException;

/**
 * SGQ Business config.
 * 
 * @author echatellier
 */
public class SgqBusinessConfig {

    protected ApplicationConfig config;

    public SgqBusinessConfig() {
        config = new ApplicationConfig();
        for (SgqBusinessOption o : SgqBusinessOption.values()) {
            if (o.defaultValue != null) {
                config.setDefaultOption(o.key, o.defaultValue);
            }
        }
    }

    public ApplicationConfig getConfig() {
        return config;
    }

    public void parse() {
        try {
            config.parse();
        } catch (ArgumentsParserException ex) {
            throw new SgqBusinessException("Can't load configuration", ex);
        }
    }

    public Properties getFlatOptions() {
        return config.getFlatOptions();
    }

    public String getApplicationVersion() {
        return config.getOption("application.version");
    }

    public String getDocumentationUrl() {
        return config.getOption(SgqBusinessOption.SGQ_DOCUMENTATION_URL.key);
    }

    public int getBatchRangeNonBioStart() {
        return config.getOptionAsInt(SgqBusinessOption.BATCH_RANGE_NONBIO_START.key);
    }

    public int getBatchRangeNonBioEnd() {
        return config.getOptionAsInt(SgqBusinessOption.BATCH_RANGE_NONBIO_END.key);
    }

    public int getBatchRangeBioStart() {
        return config.getOptionAsInt(SgqBusinessOption.BATCH_RANGE_BIO_START.key);
    }

    public int getBatchRangeBioEnd() {
        return config.getOptionAsInt(SgqBusinessOption.BATCH_RANGE_BIO_END.key);
    }

    public String getLabelProductionFilename() {
        return config.getOption(SgqBusinessOption.LABEL_PRODUCTION_FILENAME.key);
    }

    public String getLabelExpeditionFilename() {
        return config.getOption(SgqBusinessOption.LABEL_EXPEDITION_FILENAME.key);
    }

    public String getImportCsvFileEncoding() {
        return config.getOption(SgqBusinessOption.IMPORT_CSV_FILE_ENCODING.key);
    }
    
    public String getImportDbfFileEncoding() {
        return config.getOption(SgqBusinessOption.IMPORT_DBF_FILE_ENCODING.key);
    }

    public String getBackupDevicesCommand() {
        return config.getOption(SgqBusinessOption.BACKUP_DEVICES_COMMAND.key);
    }

    public String getBackupPerformCommand() {
        return config.getOption(SgqBusinessOption.BACKUP_PERFORM_COMMAND.key);
    }

    public String getDatabaseUsername() {
        return config.getOption(SgqBusinessOption.HIBERNATE_CONNETION_USERNAME.key);
    }

    public boolean isDevMode() {
        return config.getOptionAsBoolean(SgqBusinessOption.DEV_MODE.key);
    }

    static enum SgqBusinessOption {

        CONFIG_FILE(ApplicationConfig.CONFIG_FILE_NAME, "sgqweb.properties"),
        SGQ_DIRECTORY("sgq.directory", "${java.io.tmpdir}/sgq"),
        SGQ_DOCUMENTATION_URL("sgq.documentation.url", "http://maven-site.forge.codelutin.com/sgq-ch/"),

        HIBERNATE_DIALECT("hibernate.dialect", "org.hibernate.dialect.H2Dialect"),
        HIBERNATE_CONNECTION_URL("hibernate.connection.url", "jdbc:h2:file:${sgq.directory}/data/sgq"),
        HIBERNATE_CONNETION_USERNAME("hibernate.connection.username", "sa"),
        HIBERNATE_CONNECTION_PASSWORD("hibernate.connection.password", ""),
        
        IMPORT_CSV_FILE_ENCODING("import.csv.file.encoding", "UTF-8"),
        IMPORT_DBF_FILE_ENCODING("import.dbf.file.encoding", "Cp437"),

        BATCH_RANGE_NONBIO_START("batch.range.nonbio.start", "22105"),
        BATCH_RANGE_NONBIO_END("batch.range.nonbio.end", "69999"),
        BATCH_RANGE_BIO_START("batch.range.bio.start", "71716"),
        BATCH_RANGE_BIO_END("batch.range.bio.end", "89999"),

        LABEL_PRODUCTION_FILENAME("label.production.filename", "FIF_HIST.txt"),
        LABEL_EXPEDITION_FILENAME("label.expedition.filename", "FIC_HIST.txt"),

        BACKUP_DEVICES_SCRIPT("backup.devices.script", "/var/local/sgq/sgq-backup-devices.sh"),
        BACKUP_DEVICES_COMMAND("backup.devices.command", "${backup.devices.script}"),
        BACKUP_PERFORM_SCRIPT("backup.perform.script", "/var/local/sgq/sgq-backup-perform.sh"),
        BACKUP_PERFORM_COMMAND("backup.perform.command", "${backup.perform.script} %d sa sgq %f %m"),
        
        DEV_MODE("dev.mode", "false");

        protected String key;
        protected String defaultValue;

        private SgqBusinessOption(String key, String defaultValue) {
            this.key = key;
            this.defaultValue = defaultValue;
        }
    }
}