package com.franciaflex.faxtomail.ui.swing.content.demande;

/*
 * #%L
 * FaxToMail :: UI
 * %%
 * Copyright (C) 2014 Mac-Groupe, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import static org.nuiton.i18n.I18n.t;

import java.awt.*;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.table.TableCellEditor;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import com.franciaflex.faxtomail.persistence.entities.Configuration;
import com.franciaflex.faxtomail.persistence.entities.EmailFilter;
import com.franciaflex.faxtomail.persistence.entities.MailAction;
import com.franciaflex.faxtomail.services.service.EmailService;
import com.franciaflex.faxtomail.ui.swing.FaxToMailUIContext;
import com.franciaflex.faxtomail.ui.swing.actions.OpenMailFolderChooserFromListAction;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.swing.table.filter.TableFilter;
import jaxx.runtime.swing.table.filter.TableRowFilterSupport;
import jaxx.runtime.validator.swing.SwingValidator;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ColorHighlighter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.swing.action.ApplicationUIAction;
import org.nuiton.jaxx.application.swing.table.AbstractApplicationTableModel;
import org.nuiton.jaxx.application.swing.table.ColumnIdentifier;
import org.nuiton.jaxx.application.swing.util.CloseableUI;

import com.franciaflex.faxtomail.persistence.entities.Attachment;
import com.franciaflex.faxtomail.persistence.entities.AttachmentFile;
import com.franciaflex.faxtomail.persistence.entities.DemandStatus;
import com.franciaflex.faxtomail.persistence.entities.Email;
import com.franciaflex.faxtomail.persistence.entities.FaxToMailUser;
import com.franciaflex.faxtomail.persistence.entities.History;
import com.franciaflex.faxtomail.persistence.entities.HistoryImpl;
import com.franciaflex.faxtomail.persistence.entities.HistoryType;
import com.franciaflex.faxtomail.persistence.entities.MailField;
import com.franciaflex.faxtomail.persistence.entities.MailFolder;
import com.franciaflex.faxtomail.services.FaxToMailServiceUtils;
import com.franciaflex.faxtomail.ui.swing.actions.ArchiveFromListAction;
import com.franciaflex.faxtomail.ui.swing.actions.ComputeQuantitiesByRangeAction;
import com.franciaflex.faxtomail.ui.swing.actions.LoadFolderEmailsAction;
import com.franciaflex.faxtomail.ui.swing.actions.PrintOnDefaultPrinterAction;
import com.franciaflex.faxtomail.ui.swing.actions.SaveDemandeFromListAction;
import com.franciaflex.faxtomail.ui.swing.content.reply.ReplyFormUI;
import com.franciaflex.faxtomail.ui.swing.content.reply.ReplyFormUIModel;
import com.franciaflex.faxtomail.ui.swing.util.AbstractFaxToMailDemandListHandler;
import com.franciaflex.faxtomail.ui.swing.util.FaxToMailUIUtil;
import com.franciaflex.faxtomail.ui.swing.util.FolderTreeNode;
import com.franciaflex.faxtomail.ui.swing.util.PaginationComboModel;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.nuiton.util.pagination.PaginationParameter;

/**
 * Handler of UI {@link com.franciaflex.faxtomail.ui.swing.content.demande.DemandeListUIHandler}.
 *
 * @author kmorin - morin@codelutin.com
 */
public class DemandeListUIHandler extends AbstractFaxToMailDemandListHandler<DemandeListUIModel, DemandeListUI> implements CloseableUI {

    /** Logger. */
    private static final Log log = LogFactory.getLog(DemandeListUIHandler.class);
    public static final String REFRESH_KEY = "F5";
    public static final String REFRESH_FOLDER_ACTION = "refreshFolder";

    protected Configuration config;

    protected TableFilter<JXTable> tableFilter;

    public TableFilter<JXTable> getTableFilter() {
        return tableFilter;
    }

    public final PropertyChangeListener selectedDemandeChangeListener = new PropertyChangeListener() {
        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            String propertyName = evt.getPropertyName();
            if (Email.PROPERTY_PRIORITY.equals(propertyName)) {
                if (evt.getNewValue() != null || evt.getOldValue() != null) {
                    SaveDemandeFromListAction saveAction =
                            getContext().getActionFactory().createLogicAction(DemandeListUIHandler.this,
                                                                              SaveDemandeFromListAction.class);
                    saveAction.setModifiedProperties(propertyName);
                    getContext().getActionEngine().runAction(saveAction);
                }
            }

        }
    };

    public Configuration getConfiguration() {
        return config;
    }

    @Override
    public void beforeInit(DemandeListUI ui) {
        super.beforeInit(ui);

        config = getContext().newServiceContext().getConfigurationService().getConfiguration();

        DemandeListUIModel model = new DemandeListUIModel();
        Collection<MailFolder> folders = getContext().newServiceContext().getMailFolderService()
                                                .getRootMailFoldersWithReadingRights(getContext().getCurrentUser());
        model.setFolders(new ArrayList<MailFolder>(folders));

        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(DemandeListUI ui) {

        initUI(ui);

        DemandeListUIModel model = getModel();

        // init table
        final JXTable dataTable = getUI().getDataTable();

        initDemandeTable(dataTable, false);

        tableFilter = new DemandeListTableFilter(dataTable, this);

        initTableFilter();

        dataTable.getSelectionModel().addListSelectionListener(new ListSelectionListener() {

            @Override
            public void valueChanged(ListSelectionEvent e) {
                ListSelectionModel source = (ListSelectionModel) e.getSource();

                DemandeListUIModel model = getModel();
                if (source.isSelectionEmpty()) {
                    model.setSelectedEmails(null);
                } else {
                    List<DemandeUIModel> selectedRows = new ArrayList<DemandeUIModel>();
                    AbstractApplicationTableModel<DemandeUIModel> dataTableModel = (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
                    for (int i = source.getMinSelectionIndex(); i <= source.getMaxSelectionIndex(); i++) {
                        if (source.isSelectedIndex(i)) {
                            selectedRows.add(dataTableModel.getEntry(i));
                        }
                    }
                    model.setSelectedEmails(selectedRows);
                }
            }
        });

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_EMAILS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<DemandeUIModel> emails = (List<DemandeUIModel>) evt.getNewValue();
                AbstractApplicationTableModel<DemandeUIModel> dataTableModel = (AbstractApplicationTableModel<DemandeUIModel>) dataTable.getModel();
                dataTableModel.setRows(emails);

                int quotationNb = 0;
                int pfNb = 0;
                int savNb = 0;
                for (DemandeUIModel email : emails) {
                    quotationNb += email.getQuotationNb();
                    pfNb += email.getPfNb();
                    savNb += email.getSavNb();
                }

                DemandeListUIModel model = (DemandeListUIModel) evt.getSource();
                model.setQuotationNb(quotationNb);
                model.setPfNb(pfNb);
                model.setSavNb(savNb);
            }
        });

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_SELECTED_EMAILS, new PropertyChangeListener() {

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<DemandeUIModel> oldDemands = (List<DemandeUIModel>) evt.getOldValue();
                if (oldDemands != null) {
                    for (DemandeUIModel demand : oldDemands) {
                        demand.removePropertyChangeListener(selectedDemandeChangeListener);
                    }
                }

                List<DemandeUIModel> newDemands = (List<DemandeUIModel>) evt.getNewValue();
                if (newDemands != null) {
                    for (DemandeUIModel demand : newDemands) {
                        demand.addPropertyChangeListener(selectedDemandeChangeListener);
                    }
                }
            }
        });

        // init tree
        final JTree navigationTree = ui.getNavigationTree();

        final Map<MailFolder, FolderTreeNode> nodesByFolder =
                FaxToMailUIUtil.initFolderTree(getContext(), navigationTree, model.getFolders(), true);

        getContext().addPropertyChangeListener(FaxToMailUIContext.PROPERTY_ACTION_IN_PROGRESS, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                boolean inProgress = (boolean) evt.getNewValue();
                navigationTree.setEnabled(!inProgress);
            }
        });

        navigationTree.addTreeSelectionListener(new TreeSelectionListener() {
            @Override
            public void valueChanged(TreeSelectionEvent e) {
                TableCellEditor cellEditor = dataTable.getCellEditor();
                if (cellEditor != null) {
                    cellEditor.stopCellEditing();
                }

                FolderTreeNode folderNode = (FolderTreeNode) e.getPath().getLastPathComponent();
                if (folderNode.isCanSelect()) {
                    MailFolder folder = folderNode.getMailFolder();

                    getModel().setSelectedFolder(folder);
                }
            }
        });

        model.addPropertyChangeListener(DemandeListUIModel.PROPERTY_SELECTED_FOLDER, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                MailFolder folder = (MailFolder) evt.getNewValue();
                getContext().setCurrentMailFolder(folder);
                loadFolderDemands(folder, nodesByFolder);

                DefaultMutableTreeNode node = nodesByFolder.get(folder);
                if (node != null) {
                    navigationTree.setSelectionPath(new TreePath(node.getPath()));
                }
            }
        });

        // int combo box for result per page
        ui.getResultPerPageCombo().setModel(new PaginationComboModel());
        int resultPerPage = getConfig().getResultPerPage();
        ui.getModel().setResultPerPage(resultPerPage);
        ui.getResultPerPageCombo().setSelectedItem(resultPerPage);
        ui.getResultPerPageCombo().addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                updateResultPerPage(e);
            }
        });

        // add refresh shortcut
        JRootPane rootPane = getContext().getMainUI().getRootPane();
        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(REFRESH_KEY), REFRESH_FOLDER_ACTION);
        ApplicationUIAction<LoadFolderEmailsAction> refreshAction = getContext().getActionFactory().createUIAction(this, LoadFolderEmailsAction.class);
        rootPane.getActionMap().put(REFRESH_FOLDER_ACTION, refreshAction);

    }

    public void initTableFilter() {
        Decorator<Object> decorator = new Decorator<Object>(Object.class) {
            @Override
            public String toString(Object bean) {
                String s = null;
                if (bean != null) {
                    Decorator<?> decorator = getDecorator(bean.getClass(), null);
                    if (decorator != null) {
                        s = decorator.toString(bean);
                    }
                }
                if (s == null) {
                    s = JAXXUtil.getStringValue(bean);
                }
                return s;
            }
        };
        TableRowFilterSupport.forFilter(tableFilter)
                            .searchable(true)
                            .searchDecorator(decorator)
                            .useTableRenderers(true)
                            .setPopupDefaultSize(new Dimension(250, 290))
                            .apply();
    }

    protected void loadFolderDemands(MailFolder folder, Map<MailFolder, FolderTreeNode> nodesByFolder) {
        if (folder != null) {
            FolderTreeNode folderNode = nodesByFolder.get(folder);
            boolean readable = folderNode.isCanRead();

            while (folder.getAllowCreateDemandIntoFolder() == null
                    && folder.getParent() != null) {
                folder = folder.getParent();
            }

            DemandeListUIModel model = getModel();
            model.setComputeQuantitiesByRangeEnabled(readable);
            model.setNewDemandEnabled(readable && folder != null && Boolean.TRUE.equals(folder.getAllowCreateDemandIntoFolder()));
        }

        tableFilter.clear();
    }

    public void goToNextPage() {
        getModel().setPaginationParameter(getModel().getPaginationResult().getNextPage());
        runListAction();
    }
    
    public void goToPreviousPage() {
        getModel().setPaginationParameter(getModel().getPaginationResult().getPreviousPage());
        runListAction();
    }
    
    public void updateResultPerPage(ItemEvent event) {
        if (event.getStateChange() == ItemEvent.SELECTED) {
            int resultPerPage = (Integer)event.getItem();
            getModel().setResultPerPage(resultPerPage);
            getConfig().setResultPerPage(resultPerPage);
            getConfig().save();
            getModel().resetPaginationParameter();
            runListAction();
        }
    }
    
    protected void runListAction() {
        if (getModel().getSelectedFolder() != null) { // can be when update result per page
            LoadFolderEmailsAction loadFolderEmailsAction =
                    getContext().getActionFactory().createLogicAction(DemandeListUIHandler.this,
                                                                      LoadFolderEmailsAction.class);
            getContext().getActionEngine().runAction(loadFolderEmailsAction);
        }
    }

    @Override
    public void initDemandeTable(final JXTable table, boolean sortable) {
        HighlightPredicate reportNotTakenPredicate = new HighlightPredicate() {
            @Override
            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                int viewRow = adapter.row;
                int modelRow = adapter.convertRowIndexToModel(viewRow);
                DemandeUIModel row = ((AbstractApplicationTableModel<DemandeUIModel>) table.getModel()).getEntry(modelRow);
                MailFolder folder = getModel().getSelectedFolder();
                while (folder.getOpenAttachmentReportNoTaken() == null
                        && folder.getParent() != null) {
                    folder = folder.getParent();
                }

                return Boolean.TRUE.equals(folder.getOpenAttachmentReportNoTaken())
                        && row.getLastAttachmentOpener() != null
                        && !row.getLastAttachmentOpener().equals(row.getTakenBy());
            }
        };
        Color color = Color.ORANGE;
        table.addHighlighter(new ColorHighlighter(reportNotTakenPredicate, color, Color.WHITE, color.darker(), Color.WHITE));

        super.initDemandeTable(table, sortable);
    }

    @Override
    public List<MailField> getColumns() {
        List<MailField> tableColumns = null;
        MailFolder selectedFolder = getModel().getSelectedFolder();
        if (selectedFolder != null) {
            while (!selectedFolder.isUseCurrentLevelTableColumns() && selectedFolder.getParent() != null) {
                selectedFolder = selectedFolder.getParent();
            }
            tableColumns = selectedFolder.getFolderTableColumns();
        }
        return tableColumns;
    }

    @Override
    protected MailField[] getEditableTableProperties() {
        MailField[] result;
        if (getModel().getSelectedFolder() != null && getModel().getSelectedFolder().isFolderWritable()) {
            result = new MailField[] { MailField.PRIORITY, MailField.ATTACHMENT, MailField.REPLIES };
        } else {
            result = new MailField[] { MailField.ATTACHMENT, MailField.REPLIES };
        }
        return result;
    }

    @Override
    protected void onDoubleClickOnDemande(DemandeUIModel selectedEmail) {
        super.onDoubleClickOnDemande(selectedEmail);
        selectedEmail.removePropertyChangeListener(selectedDemandeChangeListener);
        getContext().setCurrentPaginationParameter(getModel().getPaginationParameter());
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getNavigationTree();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        JTree tree = getUI().getNavigationTree();
        TreeModel treeModel = tree.getModel();
        Enumeration<TreePath> paths = tree.getExpandedDescendants(new TreePath(treeModel.getRoot()));
        List<MailFolder> folders = new ArrayList<MailFolder>();
        if (paths != null) {
            while (paths.hasMoreElements()) {
                TreePath path = paths.nextElement();
                Object lastPathComponent = path.getLastPathComponent();
                if (FolderTreeNode.class.isAssignableFrom(lastPathComponent.getClass())) {
                    folders.add(((FolderTreeNode) lastPathComponent).getMailFolder());
                }
            }
        }
        getContext().setExpandedFolders(folders);

        // remove refresh shortcut
        JRootPane rootPane = getContext().getMainUI().getRootPane();
        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).remove(KeyStroke.getKeyStroke(REFRESH_KEY));
        rootPane.getActionMap().remove(REFRESH_FOLDER_ACTION);

        clearValidators();
    }

    @Override
    public boolean quitUI() {
        return true;
    }

    @Override
    public SwingValidator<DemandeListUIModel> getValidator() {
        return null;
    }

    @Override
    protected void beforeOpenPopup(int rowIndex, int columnIndex) {
        super.beforeOpenPopup(rowIndex, columnIndex);

        int selectedRowCount = getUI().getDataTable().getSelectedRowCount();


        DemandeListUIModel model = getModel();

        model.setReplyEnabled(selectedRowCount == 1
                                      && model.getSelectedEmails().get(0).isEditable()
                                      && isActionEnabled(model.getSelectedEmails().get(0), MailAction.REPLY, config));

        model.setArchiveEnabled(selectedRowCount > 0);
        model.setTransmitEnabled(selectedRowCount > 0);
        model.setPrintEnabled(selectedRowCount > 0);
    }

    /**
     * During rigth clic, auto-select located under mouse position (not done by default by java) and
     * display contextual menu.
     * 
     * @param e event
     * @param popup popup menu to display
     */
    public void autoSelectNodeInTree(MouseEvent e, JPopupMenu popup) {

        boolean rightClick = SwingUtilities.isRightMouseButton(e);

        if (rightClick) {

            JTree source = (JTree) e.getSource();

            // get the row index at this point
            int rowIndex = source.getClosestRowForLocation(e.getX(), e.getY());

            if (log.isDebugEnabled()) {
                log.debug("At point [" + e.getPoint() + "] found Row " + rowIndex);
            }

            // select row (could empty selection)
            if (rowIndex == -1) {
                source.clearSelection();
            } else {
                // set selection
                source.setSelectionRow(rowIndex);
            }

            // on right click show popup
            popup.show(source, e.getX(), e.getY());
        }
    }

    public void computeQuantitiesByRange() {
        QuantitiesByRangeUI dialogContent = new QuantitiesByRangeUI(ui);
        getContext().getActionEngine().runAction(new ComputeQuantitiesByRangeAction(dialogContent.getHandler()));
    }

    public void newDemand() {
        FaxToMailUser currentUser = getContext().getCurrentUser();
        Date now = new Date();

        List<History> histories = new ArrayList<History>();
        DemandeUIModel email = new DemandeUIModel();
        History history = new HistoryImpl();
        history.setFaxToMailUser(currentUser);
        history.setType(HistoryType.CREATION);
        history.setModificationDate(now);
        histories.add(history);

        history = new HistoryImpl();
        history.setFaxToMailUser(currentUser);
        history.setType(HistoryType.OPENING);
        history.setModificationDate(now);
        histories.add(history);

        email.setMailFolder(getModel().getSelectedFolder());
        email.setReceptionDate(now);
        email.setTakenBy(currentUser);
        email.setHistory(histories);
        email.setDemandStatus(DemandStatus.UNTREATED);
        email.setOriginalEmail(""); // can't be null
        openDemand(email);
    }

    public void print() {
        DemandeListUIModel model = getModel();
        MailFolder selectedFolder = model.getSelectedFolder();
        while (selectedFolder.getParent() != null && selectedFolder.getPrintActionEqualTakeAction() == null) {
            selectedFolder = selectedFolder.getParent();
        }
        boolean take = Boolean.TRUE.equals(selectedFolder.getPrintActionEqualTakeAction());

        Multimap<DemandeUIModel, AttachmentFile> attachmentToPrints = HashMultimap.create();

        for (DemandeUIModel demandeUIModel : model.getSelectedEmails()) {

            List<Attachment> attachments = demandeUIModel.getAttachment();
            if (CollectionUtils.isEmpty(attachments)) {
                attachmentToPrints.put(demandeUIModel, null);

            } else {
                for (Attachment attachment : attachments) {
                    // force lazy loading
                    // TODO kmorin 20140813 action ?
                    FaxToMailUIUtil.forceAttachmentFileLoading(getContext(), attachment);

                    AttachmentFile attachmentFile = attachment.getEditedFile();
                    if (attachmentFile == null) {
                        attachmentFile = attachment.getOriginalFile();
                    }
                    attachmentToPrints.put(demandeUIModel, attachmentFile);
                }
            }
        }

        PrintOnDefaultPrinterAction action = new PrintOnDefaultPrinterAction(this,
                                             attachmentToPrints,
                                             take,
                                             true);
        getContext().getActionEngine().runAction(action);
    }

    public void reply() {
        DemandeUIModel demand = getModel().getSelectedEmails().get(0);
        getContext().setCurrentEmail(demand);

        ReplyFormUI dialogContent = new ReplyFormUI(ui);
        ReplyFormUIModel model = dialogContent.getModel();

        model.setOriginalDemand(demand);

        String sender = JAXXUtil.getStringValue(demand.getSender());
        if (demand.isFax()) {
            MailFolder selectedFolder = getModel().getSelectedFolder();
            sender = FaxToMailServiceUtils.addFaxDomainToFaxNumber(sender, selectedFolder);
        }
        model.setTo(sender);

        openFrame(dialogContent, t("faxtomail.reply.title", demand.getTitle()), new Dimension(800, 600));
    }

    public void archive() {
        ArchiveFromListAction action =
                getContext().getActionFactory().createLogicAction(this,
                                                                  ArchiveFromListAction.class);
        getContext().getActionEngine().runAction(action);
    }

    /**
     * Display move popup for user to chose where to move all selected demands.
     */
    public void transmit() {
        OpenMailFolderChooserFromListAction action =
                getContext().getActionFactory().createLogicAction(this,
                                                                  OpenMailFolderChooserFromListAction.class);
        getContext().getActionEngine().runAction(action);
    }
}
