package org.planx.xpath.expr.operator;

import java.util.Iterator;
import org.planx.xpath.Navigator;
import org.planx.xpath.XPathException;
import org.planx.xpath.expr.Expression;
import org.planx.xpath.object.*;


/**
 * Relational operators include equals, not equals, less than,
 * greater then, etc.
 **/
public abstract class EqualityOperator extends Operator {
    public EqualityOperator(Expression e1, Expression e2) {
        super(e1, e2);
    }

    protected XObject evaluate(XObject o1, XObject o2, Navigator navigator)
                                                     throws XPathException {
        boolean result;

        if ((o1 instanceof XBoolean) || (o2 instanceof XBoolean)) {
            result = compare(o1.booleanValue(), o2.booleanValue());
        } else if ((o1 instanceof XNodeSet) && (o2 instanceof XNodeSet)) {
            result = compare((XNodeSet) o1, (XNodeSet) o2, navigator);
        } else if ((o1 instanceof XNodeSet) && (o2 instanceof XSimple)) {
            result = compare((XNodeSet) o1, (XSimple) o2, navigator);
        } else if ((o1 instanceof XSimple) && (o2 instanceof XNodeSet)) {
            result = compare((XNodeSet) o2, (XSimple) o1, navigator);
        } else if ((o1 instanceof XNumber) || (o2 instanceof XNumber)) {
            result = compare(((XSimple) o1).doubleValue(),
                             ((XSimple) o2).doubleValue());
        } else if ((o1 instanceof XString) && (o2 instanceof XString)) {
            result = compare(((XSimple) o1).stringValue(),
                             ((XSimple) o2).stringValue());
        } else {
            throw new XPathException("Unsupported subclass of XObject");
        }
        return result ? XBoolean.TRUE : XBoolean.FALSE;
    }

    /**
     * Compare XNodeSet and XSimple (which is not an XBoolean).
     * True if there is a node in the set such that the comparison
     * is true.
     **/
    private boolean compare(XNodeSet o1, XSimple o2, Navigator navigator)
                                                   throws XPathException {
        if (o2 instanceof XNumber) {
            double v2 = o2.doubleValue();
            for (int i = 0, size = o1.size(); i < size; i++) {
                String s1 = navigator.getStringValue(o1.get(i));
                double v1 = XString.doubleValue(s1);
                if (compare(v1, v2)) return true;
            }
        } else if (o2 instanceof XString) {
            String v2 = o2.stringValue();
            for (int i = 0, size = o1.size(); i < size; i++) {
                String v1 = navigator.getStringValue(o1.get(i));
                if (compare(v1, v2)) return true;
            }
        }
        return false;
    }

    protected abstract boolean compare(XNodeSet o1, XNodeSet o2,
                                            Navigator navigator)
                                          throws XPathException;

    protected abstract boolean compare(boolean v1, boolean v2);

    protected abstract boolean compare(double v1, double v2);

    protected abstract boolean compare(String v1, String v2);
}
