package org.planx.xpath.expr.axis;

import org.planx.xpath.Navigator;
import org.planx.xpath.XPathException;

/**
 * Ancestor axis and ancestor-or-self axis.
 **/
public class AncestorAxis extends Axis {
    private final boolean includeSelf;

    /**
     * Create an ancestor axis.
     **/
    public AncestorAxis() {
        this(false);
    }

    /**
     * If <code>includeSelf</code> is <code>true</code> create
     * an ancestor-or-self axis.
     * If <code>includeSelf</code> is <code>false</code> create
     * an ancestor axis.
     **/
    public AncestorAxis(boolean includeSelf) {
        this.includeSelf = includeSelf;
    }

    public AxisIterator iterator(Object contextNode, Navigator navigator)
                                                  throws XPathException {
        return new AncestorAxisIterator(contextNode, navigator);
    }

    /**
     * The principal node type of this axis is
     * {@link Navigator#ELEMENT}.
     **/
    public int getPrincipalNodeType() {
        return Navigator.ELEMENT;
    }

    public boolean isReverse() {
        return true;
    }

    public String toString() {
        if (includeSelf) return "ancestor-or-self";
        else return "ancestor";
    }

    class AncestorAxisIterator implements AxisIterator {
        private Object currentNode;
        private Navigator navigator;
        private XPathException currentException = null;

        AncestorAxisIterator(Object contextNode, Navigator navigator)
                                              throws XPathException {
            this.currentNode = contextNode;
            this.navigator = navigator;
            if (!includeSelf) next();
        }

        public boolean hasNext() {
            return currentNode != null;
        }

        public Object next() throws XPathException {
            if (currentException != null) throw currentException;
            if (currentNode == null) throw new XPathException(
                               "No more elements in iterator");
            Object result = currentNode;
            try {
                currentNode = navigator.getParent(currentNode);
            } catch (XPathException e) {
                // delay the throwing of the exception until
                // the node is actually requested
                currentException = e;
                currentNode = null;
            }
            return result;
        }
    }
}
