package org.planx.xpath;

import java.util.Map;
import java.util.HashMap;
import org.planx.xpath.function.*;
import org.planx.xpath.object.XObject;

/**
 * An XPath context environment.
 **/
public class Environment {
    private Map functions;
    private Map variables;

    /**
     * Create an empty environment with only core functions
     * and no variables bound.
     **/
    public Environment() {
        functions = new HashMap();
        variables = new HashMap();
        addCoreFunctions();
    }

    private void addCoreFunctions() {
        functions.put("boolean", new BooleanFunction());
        functions.put("number", new NumberFunction());
        functions.put("string", new StringFunction());
        functions.put("position", new PositionFunction());
        functions.put("last", new LastFunction());
    }

    /**
     * Binds the specified function to the specified
     * function name in this environment. If a function is
     * already bound to the specified name, that function
     * is replaced.
     **/
    public void bindFunction(String funcName, Function func) {
        functions.put(funcName, func);
    }

    /**
     * Binds all the functions in the specified map to the
     * function names under which they are stored in the map.
     * The map must consist of (String, Function) key-value pairs.
     **/
    public void bindFunctions(Map functions) {
        this.functions.putAll(functions);
    }

    /**
     * Returns the function bound to the specified name.
     * If no function is bound to the name an XPathException is thrown.
     **/
    public Function getFunction(String funcName) throws XPathException {
        if (functions.containsKey(funcName)) {
            return (Function) functions.get(funcName);
        } else {
            throw new XPathException("Function name not bound: "+funcName);
        }
    }

    /**
     * Binds the specified variable to the specified
     * variable name in this environment. If a variable is
     * already bound to the specified name, the value
     * is replaced.
     **/
    public void bindVariable(String varName, XObject value) {
        variables.put(varName, value);
    }

    /**
     * Binds all the variables in the specified map to the
     * variable names under which they are stored in the map.
     * The map must consist of (String, XObject) key-value pairs.
     **/
    public void bindVariables(Map variables) {
        this.variables.putAll(variables);
    }

    /**
     * Returns the value bound to the specified variable name.
     * If no value is bound to the name an XPathException is thrown.
     **/
    public XObject getVariable(String varName) throws XPathException {
        if (variables.containsKey(varName)) {
            return (XObject) variables.get(varName);
        } else {
            throw new XPathException("Variable name not bound: "+varName);
        }
    }
}

