package org.planx.xmlstore.regions;

import java.io.*;
import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;
import java.util.*;
import org.planx.msd.lang.EquivalenceClass;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nodes.*;
import org.planx.util.*;

/**
 * A <code>Region</code> is a collection of nodes.
 * There are only two ways a <code>Region</code> is changed:
 * When it is first not closed and exists only in memory, new nodes
 * can be added with the <code>save</code> method and outgoing
 * inter-region pointers will be found at once and the incoming
 * sets of the target regions updated. Thus, in this case, when
 * persisting the (non-closed) region it is not necessary to scan
 * for outgoing pointers. Also, in this case, the identifier
 * of the region is not changed.
 * <p>
 * Second, the <code>Sharer</code> may update data through the use
 * of a region node. This is obtained by calling {@link #getRegionRoot}.
 * See the <code>Sharer</code> code for the sequence of steps needed
 * to ensure consistency of incoming and outgoing sets when several
 * regions are shared at the same time.
 * <p>
 * Should be synchronized externally.
 *
 * @author Thomas Ambus
 */
public class Region {
    private RegionManager manager;
    private LocatorListener ioListener = new IOListener();
    private LocatorListener outgoingListener = new OutgoingListener();
    private IndexSet<LocalLocator,InterRegionEdge> incoming;
    private HashMap<InterRegionEdge.Origin,InterRegionEdge> outgoing;
    private LocalLocator onDiskBlock = null;
    private LocalLocator memoryBlock;     // always exist, generates relative locators
    private MemoryFileSystem mfs = null;  // not null while in memory
    private RegionNode regionNode = null; // not null while in memory
    private WeakHashMap<LocalLocator,SoftReference<SystemNode>>
        refMap = null; // not null while in memory
    private Group group = new Group();
    private boolean isClosed = false;   // only true when not persisted
    private boolean isModified = false; // true if flush() should write to disk
    private boolean isDirty = false;
    private int id = 0;    // unique ID in this store assigned by the manager
    int originalSize = 0;  // statistics, input size before sharing
    int originalNodes = 0; // statistics, original number of written nodes
    int currentNodes = 0;  // statistics, number of nodes when last persisted

    /**
     * Create a new non-persisted, in-memory <code>Region</code>.
     */
    Region(RegionManager manager, int id) throws IOException {
        this.manager = manager;
        this.id = id;
        // Create MemoryFileSystem with FileSystemIdentifier of the on-disk FileSystem
        // so that Locators handed out do not need to be changed when mfs is persisted
        FileSystemIdentifier fsi = manager.getFileSystem().currentIdentifier();
        int size = manager.configuration().REGION_SIZE;
        mfs = new MemoryFileSystem(size, fsi);
        memoryBlock = mfs.allocate(size);
        initRefMap();
        initInterRegionSets();

        onDiskBlock = null;
        regionNode = null;
        isClosed = false;
        isModified = false;
        isDirty = true;
    }

    private synchronized void initInterRegionSets() {
        incoming = new IndexSet<LocalLocator,InterRegionEdge>();
        outgoing = new HashMap<InterRegionEdge.Origin,InterRegionEdge>();
    }

    /**
     * Returns <code>true</code> if the specified <code>LocalLocator</code>
     * points to data in this <code>Region</code>.
     */
    public synchronized boolean isContained(LocalLocator loc) {
        if (mfs != null) return mfs.isContained(loc);
        return memoryBlock.isContained(loc);
    }

    /**
     * Returns <code>true</code> if this <code>Region</code> is closed for¨
     * further application writes (the sharer may still modify the region).
     */
    public synchronized boolean isClosed() {
        return isClosed;
    }

    /**
     * Returns <code>true</code> if this <code>Region</code> is currently
     * in memory.
     */
    public synchronized boolean isCached() {
        return mfs != null;
    }

    /**
     * Returns <code>true</code> if the on-disk representation of this
     * <code>Region</code> is up-to-date.
     */
    public synchronized boolean isPersisted() {
        return isClosed() && !isCached();
    }

    /**
     * Closes this <code>Region</code> for further application writes.
     */
    synchronized void close() {
        if (!isClosed) {
            isClosed = true;
            originalSize = memoryBlock.getLen();
        }
    }

    /**
     * Returns <code>true</code> if this <code>Region</code> is not known
     * to be discriminated with respect to the store.
     */
    synchronized boolean isDirty() {
        return isDirty;
    }

    synchronized void setDirty(boolean newDirty) {
        if (newDirty && !isDirty) {
            isDirty = true;
            for (InterRegionEdge in : incoming) {
                if (in.origin.originRegion != null) {
                    in.origin.originRegion.setDirty(true);
                }
            }
        }
        this.isDirty = newDirty;
    }

    public synchronized Group getGroup() {
        return group;
    }

    public synchronized void setGroup(Group group) {
        this.group = group;
    }

    /**
     * A local unique identifier useful for making a total order on regions
     * for consistent canonical node choosing policies.
     */
    public int getLocalId() {
        return id;
    }

    public synchronized FileSystemIdentifier getIdentifier() {
        if (mfs != null) return mfs.currentIdentifier();
        return memoryBlock.getFileSystemId();
    }

    public synchronized LocalLocator getBound() {
        return memoryBlock;
    }

    // LOADING AND SAVING FROM MEMORY FILESYSTEM, i.e. BYTE REPRESENTATION

    /**
     * Loads the node with the specified locator from this region.
     * TODO: The second argument has been deprecated, please remove.
     */
    public synchronized SystemNode load(LocalLocator loc, Reference secondaryRef)
                                   throws IOException, UnknownReferenceException {
        if (!isContained(loc)) throw new UnknownReferenceException
              ("LocalLocator "+loc+" does not belong to this Region");

        // Attempt lookup in refMap, it will hold nodes for incoming that have already been loaded
        SystemNode node = lookupMapping(loc);
        if (node != null) return node;

        cache();
        // Load from memory, will only create proxies to nodes outside this
        // region
        NodeConverter dnav = manager.getConverter();
        return dnav.load(loc, secondaryRef, memoryBlock, mfs, ioListener);
    }

    /**
     * Can only save to non-closed <code>Region</code>.
     */
    public synchronized LocalLocator save(SystemNode node) throws IOException {
        if (isClosed) throw new IOException("Region is closed for further writes");
        if (mfs == null || refMap == null) throw new IOException("Region not in memory");
        isModified = true;

        NodeConverter dnav = manager.getConverter();
        LocalLocator loc = dnav.save(node, null, mfs, outgoingListener);
        memoryBlock = mfs.all();
        if (isContained(loc)) {
            // If the node is, in fact, in this region, add it to refMap
            // in anticipation of it becoming an incoming node
            addMapping(loc, node);
        }

        // If region is full, close it
        if (mfs.size() >= manager.configuration().REGION_SIZE) close();
        return loc;
    }

    /**
     * Handles correct setting of locators' region ID's during loading, and
     * adds shared nodes to refMap during loading and saving.
     */
    private class IOListener implements LocatorListener {
        public LocalLocator locatorLoaded(LocalLocator l, LocalLocator parent,
                                          int childIndex) {
            // Locators are stored without fsi, so set it here
            InterRegionEdge out = lookupOutgoing(parent, childIndex);
            if (out != null) {
                // Child is in another region
                assert assertConnect(out, Region.this, out.targetRegion);
                return out.targetNode;
            } else {
                // Otherwise, the child is in this region
                l.setFileSystemId(getIdentifier());
                return l;
            }
        }

        public void nodeLoaded(LocalLocator l, SystemNode node,
                                           boolean hasOutgoing) {
            assert assertNode(node, l, true);
            // Only store LocalLocator->SystemNode mappings for shared,
            // incoming and outgoing
            if (hasOutgoing || node.isShared() || incomingContains(l)) {
                addMapping(l, node);
            }
        }

        public SystemNode lookup(LocalLocator l) {
            return lookupMapping(l);
        }

        public void nodeSaved(LocalLocator l, SystemNode node,
                                          boolean hasOutgoing) {
            assert assertNode(node, l, true);

            if (!isClosed) originalNodes++;
            else currentNodes++;

            // Add outgoing and shared nodes to refMap. Incoming are added
            // elsewhere.
            if (hasOutgoing || node.isShared()) {
                addMapping(l, node);
            }
        }
    }

    /**
     * Adds to IOListener identification of outgoing edges during saving
     * and updates outgoing and corresponding incoming sets on the fly.
     */
    private class OutgoingListener extends IOListener implements LocatorListener {
        public void nodeSaved(LocalLocator l, SystemNode node, boolean hasOutgoing) {
            super.nodeSaved(l, node, hasOutgoing);

            if (hasOutgoing) {
                // Add to outgoing edges here and incoming edges in target regions
                List<SystemNode> children = node.getChildren();
                for (int i=0, max=children.size(); i<max; i++) {
                    LocalLocator target = children.get(i).getLocator();
                    if (!isContained(target)) {
                        // Node is outside
                        Region r = manager.lookup(target);
                        InterRegionEdge edge =
                            new InterRegionEdge(target, l, i, r, Region.this);
                        addOutgoing(edge);
                        r.addIncoming(edge);
                        assert assertConnect(edge, Region.this, r);
                    }
                }
            }
        }
    }

    // REFERENCE MAP ADMINISTRATION

    private synchronized void addMapping(LocalLocator l, SystemNode node) {
        // Use SystemNode.get() to avoid storing a proxy: not only could
        // it give inefficient chains to proxy references - worse, it could
        // give cycles of proxies!
        node = node.get();
        assert assertNode(node, l, true);
        assert !(node instanceof NodeProxy) : "node is NodeProxy";
        if (refMap != null)
            refMap.put(l, new SoftReference<SystemNode>(node.get()));
    }

    private synchronized SystemNode lookupMapping(LocalLocator l) {
        if (refMap == null) return null;
        SoftReference<SystemNode> wn = refMap.get(l);
        SystemNode node = (wn == null) ? null : wn.get();
        if (node == null) return null;
        assert assertNode(node, l, true);
        assert !(node instanceof NodeProxy) : "node is NodeProxy";
        return node;
    }

    private synchronized void initRefMap() {
        refMap = new WeakHashMap<LocalLocator,SoftReference<SystemNode>>();
    }

    synchronized void clearRefMap() {
        if (refMap != null) refMap.clear();
    }

    // LOADING AND SAVING DISK <-> MEMORY

    /**
     * Load this <code>Region</code> to memory.
     */
    public synchronized void cache() throws IOException, UnknownReferenceException {
        manager.informCached(this);

        if (mfs != null) return; // already in memory
        if (onDiskBlock == null) throw new IOException
            ("Region not persisted");
        if (!isClosed) throw new IOException
            ("Internal error: Region is not closed but persisted");

        // Load all data in one big read
        assert onDiskBlock.getFileSystemId().equals(memoryBlock.getFileSystemId());
        mfs = new MemoryFileSystem(onDiskBlock.getLen(), onDiskBlock.getFileSystemId());
        memoryBlock = mfs.allocate(onDiskBlock.getLen());
        mfs.copy(manager.getFileSystem(), onDiskBlock, memoryBlock);
        initRefMap();
    }

    /**
     * Flushes this <code>Region</code> from memory by persisting it and releasing
     * memory.
     */
    public synchronized void flush() throws IOException, UnknownReferenceException {
        if (regionNode != null) {
            // may not release while region node exist
            manager.informCached(this);
            return;
        }
        persist();

        // Release
        mfs = null;
        refMap = null;
        regionNode = null;
    }

    /**
     * Persists the in-memory version of this <code>Region</code> to disk.
     */
    public synchronized void persist() throws IOException, UnknownReferenceException {
        if (mfs == null) return;
        if (!isClosed && onDiskBlock != null) throw new IllegalStateException
            ("Internal error: Region is not closed but persisted");

        if (isModified) {
            FileSystem fs = manager.getFileSystem();
            if (onDiskBlock != null && regionNode == null) throw new IOException
                ("Previously persisted Region cannot have been modified without a RegionNode");

            // If previously persisted, free the old copy
            if (onDiskBlock != null) {
                fs.free(onDiskBlock);
            }

            if (regionNode != null) {
                // The RegionNode is assumed to have the most recent version of the graph.
                // Write the graph in-memory first and then save to disk (below).

                assert assertGraph(false,false,false)    : "invalid graph before save";

                // Set isShared so that it is correctly written in nodes
                scanShared(regionNode, memoryBlock, new Object());

                // Clear memory file system and generate new ID
                FileSystemIdentifier fsi = fs.currentIdentifier();
                mfs.clear(fsi); // only place (except constructor) that changes fsi
                clearRefMap();
                currentNodes = 0;

                // Inform manager that ID has changed
                manager.change(memoryBlock.getFileSystemId(), this);
                assert manager.lookup(mfs.all()) == this : "manager did not change id";
                assert !mfs.isContained(memoryBlock)     : "fsi should change";

                // Write the graph
                for (SystemNode incomingRoot : regionNode.getChildren()) {
                    // Writes nodes (previously) within memoryBlock and not nodes outside.
                    LocalLocator loc = incomingRoot.getLocator();
                    if (memoryBlock.isContained(loc)) {
                        LocalLocator l = manager.getConverter().save
                             (incomingRoot, memoryBlock, mfs, ioListener);
                        addMapping(l, incomingRoot);
                    }
                }
                memoryBlock = mfs.all();
                regionNode.setLocator(memoryBlock);
                assert mfs.isContained(memoryBlock)      : "memoryBlock mfs mismatch";
            }

            if (memoryBlock.getLen() > 0) {
                // Copy from memory to disk in one big write
                onDiskBlock = fs.allocate(memoryBlock.getLen(), memoryBlock.getFileSystemId());
                assert onDiskBlock.getFileSystemId().equals(memoryBlock.getFileSystemId());
                fs.copy(mfs, memoryBlock, onDiskBlock);
            } else {
                // Region is empty, manager will be informed after mend or flush
                onDiskBlock = null;
            }
        }
        isModified = false;
        isClosed = true;
    }

    /**
     * Sets the isShared status correctly in each node.
     * The isShared status needs to be set so that it is correct when nodes are read
     * back-in. Thus, this must be done before saving (and not while saving), since
     * otherwise the isShared status will be wrong, because it is first known the second
     * time the node is encountered and then the node will already have been written.
     * <p>
     * Will not set isShared <code>true</code> for nodes in incoming set as this may
     * change before the next time the node is loaded, and thus should rather be checked
     * at load time.
     */
    private synchronized void scanShared(SystemNode node, LocalLocator loc, Object visitToken) {
        if (!isContained(loc)) return;
        if (node.getVisitToken() == visitToken) {
            // redundant visit
            node.setShared(true);
            return;
        }
        node.setVisitToken(visitToken);
        node.setShared(false);

        List<SystemNode> children = node.getChildren();
        for (int i=0, max=children.size(); i<max; i++) {
            SystemNode child = children.get(i);
            LocalLocator target = child.getLocator();
            if (isContained(target)) {
                // internal node to this region
                scanShared(child, target, visitToken);
            }
        }
    }

    /**
     * If the argument <code>Region</code> is not null all outgoing pointers
     * to children in the specified region will be looked up in that region's
     * refMap and used instead of a proxy. This should only be done after calling
     * <code>getRegionRoot</code> on that region which will cause all live nodes
     * to be loaded. Thus, the two regions' graphs of live nodes are merged.
     */
    synchronized SystemNode getRegionRoot(Region twin) throws IOException,
                                                UnknownReferenceException {
        releaseRegionRoot();
        cache();  // does nothing if already in memory

        // Convert from MemoryFileSystem to DAGs
        List<SystemNode> roots = new ArrayList<SystemNode>(incoming.size());
        List<InterRegionEdge> ins = new ArrayList<InterRegionEdge>(incoming.size());
        for (InterRegionEdge in : incoming) {
            SystemNode node = load(in.targetNode, null);
            assert node.getLocator() != null
                : "root's locator null";
            assert isContained(node.getLocator())
                : "region does not contain locator in root";
            roots.add(node);
            ins.add(in);
        }

        // Make special root node keeping track of incoming pointers from other regions
        regionNode = new RegionNode(roots, ins);

        // Replace node proxies in twin region with real nodes
        if (twin != null) {
            if (twin.regionNode == null) throw new NullPointerException();
            mergeWithTwin(twin);
            twin.mergeWithTwin(this);
        }

        isModified = true; // Expect to be modified
        return regionNode;
    }

    private synchronized void mergeWithTwin(Region twin) {
        for (InterRegionEdge out : outgoing.values()) {
            assert assertConnect(out, this, out.targetRegion);
            if (twin == out.targetRegion) {
                SystemNode real = twin.lookupMapping(out.targetNode);
                SystemNode parent = lookupMapping(out.origin.originNode);
                if (real != null && parent != null)
                    parent.setChild(out.origin.childIndex, real);
            }
        }
    }

    /**
     * Releases data referenced by the region root (allows garbage collection).
     */
    synchronized void releaseRegionRoot() {
        if (regionNode != null && isModified) throw new IllegalStateException
                 ("Changes made to previous RegionNode has not been persisted");
        regionNode = null;
    }

    // MENDING INCOMING AND OUTGOING SETS

    synchronized void prepareMend(Region twin) {
        // Remove all old entries from this' outgoing set and other's incoming sets
        for (InterRegionEdge out : outgoing.values()) {
            out.targetRegion.removeIncoming(out);
        }
        for (Region r : manager.getRegions()) {
            if (r != this && r != twin) {
                r.clearRefMap();
            }
        }
        initInterRegionSets();
    }

    synchronized boolean mend(Region twin) throws UnknownReferenceException {
        InterRegionEdge oldIn=null, newIn=null;
        SystemNode newRoot=null;
        LocalLocator newLoc=null, oldLoc=null, oldOrigin=null;
        boolean isConsistent=true, isGlobalRoot=false, hadForeignOrigin=false;
        Region oldOriginRegion=null, targetRegion=null;

        if (twin == this) throw new IllegalArgumentException("Regions identical");
        if (regionNode == null) throw new NullPointerException("Region node null");
        if (twin != null && twin.regionNode == null)
            throw new NullPointerException("Region node null in twin");
        if (isModified) throw new IllegalStateException("This region not persisted");
        if (twin != null && twin.isModified)
            throw new IllegalStateException("Twin region not persisted");

        try {
            assert manager.lookup(memoryBlock) == this
                : "manager lookup of memoryBlock does not yield this region";

            List<SystemNode> roots = regionNode.roots;
            List<InterRegionEdge> oldIns = regionNode.ins;
            Object visitToken = new Object();

            for (int i=0, max=roots.size(); i<max; i++) {
                oldIn            = oldIns.get(i);
                newRoot          = roots.get(i);
                newLoc           = newRoot.getLocator();
                oldLoc           = oldIn.targetNode;
                oldOrigin        = oldIn.origin.originNode;
                oldOriginRegion  = oldIn.origin.originRegion;
                isGlobalRoot     = oldOriginRegion == null;
                hadForeignOrigin = oldOriginRegion != this &&
                                  (twin == null || oldOriginRegion != twin);

                assert newRoot != null         : "newRoot is null";
                assert newLoc != null          : "newLoc is null";
                assert !newLoc.equals(oldLoc)  : "new and old locators are equal";
                assert assertEdge(oldIn, oldOriginRegion, this)   : "edge failed";
                assert oldOriginRegion != this : "origin region must not be itself";
                assert (oldOriginRegion==this || (twin!=null && oldOriginRegion==twin))
                        || manager.lookup(oldOrigin) == oldOriginRegion
                    : "foreign origin locator's region has changed;oldOrigin="+oldOrigin+
                      ";oldOriginRegion="+oldOriginRegion+
                      ";manager.lookup="+manager.lookup(oldOrigin);

                if (isContained(newLoc)) {
                    // Root is in this region
                    assert isContained(newLoc) : "new locator is not in this region";
                    assert manager.lookup(newLoc) == this
                        : "manager lookup of new locator does not yield this region";

                    if (isGlobalRoot || hadForeignOrigin) {
                        newIn = new InterRegionEdge(newLoc, oldOrigin, oldIn.origin.childIndex,
                                                                         this, oldOriginRegion);
                        addIncoming(newIn);
                        if (isGlobalRoot) {
                            isConsistent = manager.rootMoved(oldLoc, newLoc) & isConsistent;
                        } else {
                            oldOriginRegion.changeOutgoing(newIn);
                        }
                    }
                } else {
                    // Root is in twin region
                    targetRegion = (twin == null) ? manager.lookup(newLoc) : twin;
                    assert targetRegion != null : "target/twin region is null";
                    assert targetRegion.isContained(newLoc)
                        : "new locator is not in target/twin region"+
                          ";targetRegion.memoryBlock="+targetRegion.memoryBlock;
                    assert manager.lookup(newLoc) == targetRegion
                        : "manager lookup of new locator does not yield target/twin region"+
                          ";targetRegion.memoryBlock="+targetRegion.memoryBlock;

                    if (isGlobalRoot || hadForeignOrigin) {
                        newIn = new InterRegionEdge(newLoc, oldOrigin, oldIn.origin.childIndex,
                                                                 targetRegion, oldOriginRegion);
                        targetRegion.addIncoming(newIn);
                        if (isGlobalRoot) {
                            isConsistent = manager.rootMoved(oldLoc, newLoc) & isConsistent;
                        } else {
                            oldOriginRegion.changeOutgoing(newIn);
                        }
                    }
                }
                scanOutgoing(newRoot, newLoc, visitToken);
            }
            assert assertGraph(true,true,true) : "invalid graph after save";
        } catch (AssertionError e) {
            String graph = (assert_currentNodes<assert_graphSize) ? ";"+assertGraphOutput() : "";
            rethrow(e,";twin="+twin+";oldIn="+oldIn+";newIn="+newIn+
                ";newRoot="+toString(newRoot)+";newLoc="+newLoc+
                ";isGlobalRoot="+isGlobalRoot+";hadForeignOrigin="+hadForeignOrigin+graph+
                ";targetRegion="+targetRegion);
        }

        // Empty region?
        if (memoryBlock.getLen() == 0) manager.remove(this);
        return isConsistent;
    }

    private synchronized void scanOutgoing(SystemNode node,
                       LocalLocator loc, Object visitToken) {
        if (!isContained(loc)) return;
        if (node.getVisitToken() == visitToken) return;
        node.setVisitToken(visitToken);
        assert assertNode(node, false);

        // Add to outgoing edges here and incoming edges in target regions
        List<SystemNode> children = node.getChildren();
        for (int i=0, max=children.size(); i<max; i++) {
            SystemNode child = children.get(i);
            LocalLocator target = child.getLocator();
            assert target != null : "child node has null locator. Probable cause: "+
                                    "node with deprecated locator in refMap?"+
                                    ";node="+toString(node);
            if (isContained(target)) {
                // Internal node to this region
                scanOutgoing(child, target, visitToken);
            } else {
                // Node is outside
                Region r = manager.lookup(target);
                assert r != null : "cannot find target region for child locator."+
                                   "Probable cause: node with deprecated locator in refMap?"+
                                   ";target="+target+";node="+toString(node);
                InterRegionEdge edge = new InterRegionEdge(target, loc, i, r, this);
                addOutgoing(edge);
                r.addIncoming(edge);
                assert assertConnect(edge, Region.this, r);
            }
        }
    }

    synchronized void addIncoming(InterRegionEdge in) {
        assert assertNewEdge(in, in.origin.originRegion, this);
        incoming.add(in.targetNode, in);
    }

    synchronized void removeIncoming(InterRegionEdge in) {
        assert assertEdge(in, in.origin.originRegion, this);
        incoming.remove(in.targetNode, in);
    }

    private synchronized void addOutgoing(InterRegionEdge out) {
        assert assertNewEdge(out, this, out.targetRegion);
        outgoing.put(out.origin, out);
    }

    private synchronized void removeOutgoing(InterRegionEdge out) {
        assert assertEdge(out, this, out.targetRegion);
        outgoing.remove(out.origin);
        if (refMap != null) refMap.remove(out.origin.originNode);
    }

    private synchronized InterRegionEdge lookupOutgoing(
                    LocalLocator origin, int childIndex) {
        InterRegionEdge out = outgoing.get(
            new InterRegionEdge.Origin(origin,childIndex,this));
        if (out == null) return null;
        try {
            assert assertEdge(out, this, out.targetRegion);
            assert out.origin.originNode == null && origin == null ||
                  (out.origin.originNode != null && origin != null &&
                   out.origin.originNode.equals(origin))
                : "looked-up edge does not match argument locator";
            assert out.origin.childIndex == childIndex
                : "looked-up edge does not match argument childIndex";
        } catch (AssertionError e) {
            rethrow(e,";origin="+origin+";childIndex="+childIndex+";edge="+out);
        }
        return out;
    }

    private synchronized InterRegionEdge lookupOutgoing(
                          InterRegionEdge.Origin origin) {
        InterRegionEdge out = outgoing.get(origin);
        if (out == null) return null;
        try {
            assert assertEdge(out, this, out.targetRegion);
            assert out.origin.originNode == null && origin.originNode == null ||
                  (out.origin.originNode != null && origin.originNode != null &&
                   out.origin.originNode.equals(origin.originNode))
                : "looked-up edge does not match argument's origin locator";
            assert out.origin.childIndex == origin.childIndex
                : "looked-up edge does not match argument's childIndex";
        } catch (AssertionError e) {
            rethrow(e,";origin="+origin+";edge="+out);
        }
        return out;
    }

    private synchronized boolean incomingContains(LocalLocator target) {
        return incoming.contains(target);
    }

    /**
     * Called by another <code>Region</code> to inform that a node pointed to
     * has changed.
     */
    private synchronized void changeOutgoing(InterRegionEdge newOut) {
        InterRegionEdge oldOut = lookupOutgoing(newOut.origin);
        assert oldOut != null : "old outgoing edge is null;newOut="+newOut+";"+oldOut;
        removeOutgoing(oldOut);
        addOutgoing(newOut);
    }

    private class RegionNode extends AbstractNode {
        List<SystemNode> roots;
        List<InterRegionEdge> ins;
        int height = -1;
        LocalLocator loc = null;

        RegionNode(List<SystemNode> roots, List<InterRegionEdge> ins) {
            this.roots = roots;
            this.ins = ins;
            loc = memoryBlock;
        }

        public void setChild(int index, SystemNode child) {
            roots.set(index, child);
        }

        public boolean isShared() {
            return false;
        }

        public void setShared(boolean isShared) {}

        public int getHeight() {
            return height;
        }

        public void setHeight(int height) {
            this.height = height;
        }

        public byte getType() {
            return Node.ELEMENT;
        }

        public String getNodeValue() {
            return "RegionNode";
        }

        public List<SystemNode> getChildren() {
            return roots;
        }

        public void addChild(SystemNode child) {
            roots.add(child);
        }

        public List<Attribute> getAttributes() {
            return Collections.emptyList();
        }

        public String getAttribute(String attrName) {
            return null;
        }

        public String[] getAttributeNames() {
            return new String[0];
        }

        public LocalLocator getLocator() {
            return loc;
        }

        public void setLocator(LocalLocator loc) {
            this.loc = loc;
        }

        public String toString() {
            StringBuilder result = new StringBuilder();
            List<SystemNode> children = getChildren();
            List<Attribute> attributes = getAttributes();

            result.append("<" + getNodeValue());

            for (Attribute attr : attributes) {
                result.append(" "+attr.getName()+"=\""+attr.getValue()+"\"");
            }

            if (children.size() > 0) {
                result.append(">\n");
                for (SystemNode node : children) {
                    result.append(node.toString() + "\n");
                }
                result.append("</" + getNodeValue() + ">");
            } else {
                result.append("/>");
            }
            return result.toString();
        }
    }

    // STATISTICS

    public synchronized String toString() {
        return "R"+id+"{"+memoryBlock.getFileSystemId()+"/"+((mfs==null)?"null":mfs.all().getFileSystemId())+"}";
    }

    public synchronized int size() {
        return memoryBlock.getLen();
    }

    public synchronized int originalSize() {
        return originalSize;
    }

    public synchronized int incomingSize() {
        return incoming.size();
    }

    public synchronized int outgoingSize() {
        return outgoing.size();
    }

    public synchronized Statistics statistics() {
        return new Statistics(this);
    }

    public static class Statistics {
        public LocalLocator onDiskBlock, memoryBlock;
        public boolean isClosed, isModified, isDirty, isInMemory, hasRegionRoot;
        public int size, originalSize, refMapSize, incomingSize, outgoingSize;
        public int originalNodes, currentNodes;
        public String region, refMap, incoming, outgoing;

        Statistics(Region r) {
            this.region = r.toString();
            this.onDiskBlock = r.onDiskBlock;
            this.memoryBlock = r.memoryBlock;
            this.isClosed = r.isClosed;
            this.isModified = r.isModified;
            this.isDirty = r.isDirty;
            this.isInMemory = (r.mfs!=null);
            this.hasRegionRoot = (r.regionNode!=null);
            this.size = r.memoryBlock.getLen();
            this.originalSize = r.originalSize;
            this.originalNodes = r.originalNodes;
            this.currentNodes = r.currentNodes;
            this.refMapSize = (r.refMap==null) ? 0 : r.refMap.size();
            this.incomingSize = r.incoming.size();
            this.outgoingSize = r.outgoing.size();
        }

        static String toString(Collection<?> c) {
            StringBuilder sb = new StringBuilder();
            sb.append("{\n");
            for (Object o : c) {
                sb.append("      ");
                sb.append(o.toString());
                sb.append("\n");
            }
            sb.append("    }");
            return sb.toString();
        }

        public String toString() {
            return full();
        }

        public String brief() {
            return region+"{size="+size+",originalSize="+originalSize+
                          ",currentNodes="+currentNodes+",originalNodes="+originalNodes+
                          ",incoming="+incomingSize+",outgoing="+outgoingSize+"}";
        }

        public String full() {
            StringBuilder sb = new StringBuilder();
            sb.append("  Region "+region+"\n");
            sb.append("    onDiskBlock:   "+onDiskBlock+"\n");
            sb.append("    memoryBlock:   "+memoryBlock+"\n");
            sb.append("    size:          "+size+"\n");
            sb.append("    originalSize:  "+originalSize+"\n");
            sb.append("    currentNodes:  "+currentNodes+"\n");
            sb.append("    originalNodes: "+originalNodes+"\n");
            sb.append("    isClosed:      "+isClosed+"\n");
            sb.append("    isModified:    "+isModified+"\n");
            sb.append("    isDirty:       "+isDirty+"\n");
            sb.append("    isInMemory:    "+isInMemory+"\n");
            sb.append("    hasRegionRoot: "+hasRegionRoot+"\n");
            sb.append("    refMapSize:    "+refMapSize+"\n");
            sb.append("    incomingSize:  "+incomingSize+"\n");
            sb.append("    outgoingSize:  "+outgoingSize+"\n");
            sb.append("    refMap:        "+refMap+"\n");
            sb.append("    incoming:      "+incoming+"\n");
            sb.append("    outgoing:      "+outgoing+"\n");
            return sb.toString();
        }
    }

    static boolean assertNewEdge(InterRegionEdge edge, Region origin, Region target) {
        try {
            assertEdge(edge, origin, target);
            assert target.isContained(edge.targetNode)
                : "target node not in target region";
            assert !target.isContained(edge.origin.originNode)
                : "origin node in target region";
            assert origin==null||origin.isContained(edge.origin.originNode)
                : "origin node not in origin region";
            assert origin==null||!origin.isContained(edge.targetNode)
                : "target node in origin region";
            assert origin==null||origin.manager.lookup(edge.origin.originNode)==
                edge.origin.originRegion :
                    "manager lookup of origin does not yield origin region";
            assert target.manager.lookup(edge.targetNode) == edge.targetRegion
                : "manager lookup of target node does not yield target region";
        } catch (AssertionError e) {
            rethrows(e,";edge="+edge+";origin="+origin+";target="+target);
        }
        return true;
    }

    static boolean assertEdge(InterRegionEdge edge, Region origin, Region target) {
        try {
            assert edge != null
                : "edge null";
            assert origin != target
                : "origin and target regions are the same";
            assert edge.targetRegion != null
                : "target region is null";
            assert edge.targetNode != null
                : "target node is null";
            assert edge.targetRegion == target
                : "target region mismatch";
            assert edge.origin.originRegion == origin
                : "origin region mismatch";
            assert edge.origin.childIndex >= 0
                : "child index negative";
            assert edge.origin.originNode == null && edge.origin.originRegion == null ||
                   edge.origin.originNode != null && edge.origin.originRegion != null
                : "origin node and origin region null mismatch";
            assert edge.targetRegion != edge.origin.originRegion
                : "origin and target regions are the same";
            assert edge.targetNode != edge.origin.originNode
                : "origin node and target node are the same";
        } catch (AssertionError e) {
            rethrows(e,";edge="+edge+";origin="+origin+";target="+target);
        }
        return true;
    }

    static boolean assertConnect(InterRegionEdge edge, Region origin, Region target) {
        try {
            assertNewEdge(edge, origin, target);
            assert origin == null || edge.equals(origin.lookupOutgoing(edge.origin))
                : "origin region lookup does not yield edge";
            assert target.incomingContains(edge.targetNode)
                : "target region does not contain edge (1)";
            assert target.incoming.get(edge.targetNode).contains(edge)
                : "target region does not contain edge (2)";
        } catch (AssertionError e) {
            rethrows(e,";edge="+edge+";origin="+origin+";target="+target);
        }
        return true;
    }

    boolean assertIncomingConnect() {
        try {
            for (InterRegionEdge edge : incoming) {
                assert assertConnect(edge, edge.origin.originRegion, this)
                    : "incoming set failed connection assertion";
            }
        } catch (AssertionError e) {
            rethrow(e,";incoming.size="+incoming.size()+
                      ";outgoing.size="+outgoing.size()+
                      ";refMap.size="+((refMap==null)?"null":refMap.size()));
        }
        return true;
    }

    boolean assertOutgoingConnect() {
        try {
            for (InterRegionEdge edge : outgoing.values()) {
                assert assertConnect(edge, this, edge.targetRegion)
                    : "outgoing set failed connection assertion";
            }
        } catch (AssertionError e) {
            rethrow(e,";incoming.size="+incoming.size()+
                      ";outgoing.size="+outgoing.size()+
                      ";refMap.size="+((refMap==null)?"null":refMap.size()));
        }
        return true;
    }

    synchronized boolean assertNode(SystemNode node, boolean verifyManager) {
        try {
            assert node != null
                : "node is null";
            assert node.getLocator() != null
                : "reference in node is null";
            assert mfs != null
                : "memory file system should not be null at this point";
            assert mfs.isContained(node.getLocator())
                : "this region does not contain the node's locator";
            if (verifyManager)
                assert manager.lookup(node.getLocator()) == this
                    : "manager lookup of node's locator does not yield this region";
        } catch (AssertionError e) {
            rethrow(e,";nodeLoc="+toLocator(node)+";node="+toString(node)+
                      ";manager.lookup="+manager.lookup(toLocator(node))+
                      ";refMap.size="+((refMap==null)?"null":refMap.size())+
                      ";incomingContains="+incomingContains(toLocator(node)));
        }
        return true;
    }

    synchronized boolean assertNode(SystemNode node, LocalLocator loc, boolean verifyManager) {
        try {
            assert loc != null
                : "argument locator is null";
            assert loc.equals(toLocator(node))
                : "argument locator does not match node's locator";
            assertNode(node, verifyManager);
        } catch (AssertionError e) {
            rethrow(e,";loc="+loc+";nodeLoc="+toLocator(node)+";node="+toString(node)+
                      ";manager.lookup="+manager.lookup(toLocator(node))+
                      ";refMap.size="+((refMap==null)?"null":refMap.size())+
                      ";incomingContains="+incomingContains(toLocator(node)));
        }
        return true;
    }

    private static int assert_graphSize = 100;
    private int assert_currentNodes = 0;

    synchronized boolean assertGraph(boolean verifyShared,
                                     boolean verifyOutgoing,
                                     boolean verifyNodes) {
        int out = 0;
        assert_currentNodes = 0;
        try {
            assert regionNode != null
                : "must have region node at this point";
            Object token = new Object();
            for (SystemNode node : regionNode.getChildren()) {
                if (isContained(toLocator(node)))
                    out += assertGraph(node, verifyShared, token);
            }
            if (verifyNodes) assert assert_currentNodes==currentNodes
                : "mismatch in number of nodes";
            if (verifyOutgoing) assert out==outgoing.size()
                : "mismatch in outgoing set's size";
        } catch (AssertionError e) {
            String graph = (assert_currentNodes<assert_graphSize) ? ";"+assertGraphOutput() : "";
            rethrow(e,";assertOut="+out+";out="+outgoing.size()+
                      ";assertNodes="+assert_currentNodes+";currentNodes="+currentNodes+
                      ";verifyShared="+verifyShared+";verifyOutgoing="+verifyOutgoing+graph);
        }
        return true;
    }

    private synchronized int assertGraph(SystemNode node, boolean verifyShared,
                                                             Object visitToken) {
        int out = 0;
        if (!isContained(toLocator(node))) return 1;
        assertNode(node, true);

        if (node.getVisitToken() == visitToken) {
            if (verifyShared)
                assert node.isShared()
                    : "node is shared but does not know it;node="+toString(node)+
                      ";nodeLoc="+toLocator(node);
            return 0;
        }
        node.setVisitToken(visitToken);

        LocalLocator loc = toLocator(node);
        List<SystemNode> children = node.getChildren();
        for (int i=0, max=children.size(); i<max; i++) {
            SystemNode child = children.get(i);
            LocalLocator l = toLocator(child);
            assert child != null : "child is null";
            assert child != node : "cyclic node reference";
            assert l != loc      : "cyclic locator reference";
            out += assertGraph(child, verifyShared, visitToken);
        }
        assert_currentNodes++;
        return out;
    }

    private synchronized String assertGraphOutput() {
        if (regionNode == null) return "";
        StringBuilder sb = new StringBuilder();
        assertGraphOutput(regionNode, sb, 0, new Object());
        return sb.toString();
    }

    private synchronized void assertGraphOutput(SystemNode node,
                StringBuilder sb, int indent, Object visitToken) {
        indent(sb, indent);
        if (node == null) {
            sb.append("<NULL/>;");
            return;
        }
        if (!isContained(toLocator(node))) {
            sb.append("<OUTSIDE LOC="+toLocator(node)+">");
            sb.append(node.getNodeValue());
            sb.append("</OUTSIZE>;");
            return;
        }
        String s = (node.getVisitToken() == visitToken) ? " SHARED=TRUE" : "";
        node.setVisitToken(visitToken);

        if (node.getType() == Node.ELEMENT) {
            sb.append("<"+node.getNodeValue()+" LOC="+toLocator(node)+s+">;");
            for (SystemNode child : node.getChildren())
                assertGraphOutput(child, sb, indent+1, visitToken);
            indent(sb, indent);
            sb.append("</"+node.getNodeValue()+">;");
        } else {
            sb.append("<CHARDATA LOC="+toLocator(node)+s+">");
            sb.append(node.getNodeValue());
            sb.append("</CHARDATA>;");
        }
    }

    private static void indent(StringBuilder sb, int indent) {
        for (int i=0; i<indent*2; i++) sb.append(' ');
    }

    synchronized boolean assertInterRegion() {
        assertIncomingConnect();
        assertOutgoingConnect();
        assert assertGraph(true,true,true) : "invalid graph";
        return true;
    }

    static void rethrows(AssertionError e, String detail) {
        detail = format(e.getMessage()+detail);
        AssertionError a = new AssertionError(detail);
        a.setStackTrace(e.getStackTrace());
        throw a;
    }

    void rethrow(AssertionError e, String detail) {
        detail = format(e.getMessage()+";region="+this+detail);
        AssertionError a = new AssertionError(detail);
        a.setStackTrace(e.getStackTrace());
        throw a;
    }

    static String toString(SystemNode node) {
        if (node==null) return "null";
        String cls = node.getClass().getName();
        cls = cls.substring(cls.lastIndexOf('.')+1);
        return cls+":"+node.getNodeValue();
    }

    static LocalLocator toLocator(SystemNode node) {
        if (node == null) return null;
        return node.getLocator();
    }

    static String format(String detail) {
        return detail.replaceAll(";","\n");
    }
}
