package org.planx.xmlstore.regions;

import java.io.*;
import java.util.*;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nodes.*;
import org.planx.xmlstore.references.*;
import org.planx.util.*;
import static org.planx.xmlstore.regions.RegionConfiguration.*;

public class HashSharer extends Sharer {
    private LinkedList<Region> regions;
    private LinkedList<Region> queue;
    private ReferenceListener listener;
    private Map<ContentValueReference,SystemNode> discMap;
    private PersistentMap<ContentValueReference,LocalLocator> refTable;

    HashSharer(RegionManager manager) throws IOException {
        super(manager);
        String name = manager.getXMLStore().toString();
        regions = new LinkedList<Region>();
        queue = new LinkedList<Region>();

        Streamer<ContentValueReference> s1 = ContentValueReference.getStreamer();
        Streamer<LocalLocator> s2 = LocalLocator.getStreamer(false);
        refTable = new PersistentMap<ContentValueReference,LocalLocator>
                                                (name + ".hash", s1, s2);

        listener = new HashListener();
    }

    synchronized void addRegion(Region region) throws IOException, UnknownReferenceException {
        if (region == null) throw new NullPointerException();
        share(region);
        regions.addLast(region);
    }

    synchronized void removeRegion(Region region) {
        regions.remove(region);
        queue.remove(region);
    }

    public synchronized void share() throws IOException, UnknownReferenceException {
        Region r = queue.poll();
        if (r == null) {
            queue.addAll(regions);
            return;
        }
        share(r);
    }

    public synchronized void share(Region r) throws IOException, UnknownReferenceException {
        invocations++;

        if (!r.isClosed()) throw new IllegalArgumentException("Region is not closed");
        SystemNode root = r.getRegionRoot(null);
//debug.Debug.out("H:share "+r);

        discMap = new HashMap<ContentValueReference,SystemNode>();
        discriminate(root, r, new Object());
        discMap = null;

        r.persist();

        Object visitToken = new Object();
        for (SystemNode node : root.getChildren()) {
            updateTable(node, r, visitToken);
        }
        r.prepareMend(null);
        boolean isConsistent = r.mend(null);
        assert r.assertInterRegion() : "inter-region inconsistency";
        r.releaseRegionRoot();

        if (!isConsistent) share(r); // global roots changed, re-run
    }

    private synchronized void discriminate(SystemNode node, Region r, Object visitToken)
                                                                     throws IOException {
        if (!r.isContained(node.getLocator())) return;
        if (node.getVisitToken() == visitToken) return;
        node.setVisitToken(visitToken);

        List<SystemNode> children = node.getChildren();
        for (int i=0; i<children.size(); i++) {
            SystemNode child = children.get(i);
            discriminate(child, r, visitToken);

            ContentValueReference vref = ContentValueReference.reference(child, listener);
            SystemNode existing = discMap.get(vref);
            if (existing == null) {
                discMap.put(vref, child);
            } else {
                node.setChild(i, existing);
            }
        }
    }

    private synchronized void updateTable(SystemNode node, Region r, Object visitToken)
                                                                    throws IOException {
        if (!r.isContained(node.getLocator())) return;
        if (node.getVisitToken() == visitToken) return;
        node.setVisitToken(visitToken);

//debug.Debug.out("H:putting node="+Region.toString(node)+", "+node.getReference()+"->"+node.getLocator());
        ContentValueReference vref = (ContentValueReference) node.getReference();
        refTable.put(vref, node.getLocator());

        for (SystemNode child : node.getChildren()) {
            updateTable(child, r, visitToken);
        }
    }

    private class HashListener extends AbstractReferenceListener {
        public ContentValueReference lookup(Node node) {
            SystemNode n = (SystemNode) node;
            if (n.getReference() instanceof ContentValueReference) {
                return (ContentValueReference) n.getReference();
            }
            return null;
        }

        public void referenceComputed(Node node, ContentValueReference vref) {
            if (node == null) throw new NullPointerException();
            if (vref == null) throw new NullPointerException();
            SystemNode n = (SystemNode) node;
            if (n.getLocator() == null) throw new NullPointerException();

            LocalLocator existing = refTable.get(vref);
            if (existing != null) {
//debug.Debug.out("H:getting existing node="+Region.toString(n)+", "+vref+"->"+existing+", nodeRef="+n.getReference());
                n.setLocator(existing);
            }

            //if (n.getReference() == null) {
                n.setReference(vref);
            //}
        }
    }
}
