package org.planx.xmlstore.nodes;

import java.io.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.planx.msd.Discriminator;
import org.planx.msd.Memory;
import org.planx.util.*;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.references.RelativeReference;

/**
 * @author Thomas Ambus
 */
public class NodeConverter {
    private static final int ELEMENT        = 0x00;  // disk type
    private static final int CHARDATA       = 0x01;  // disk type
    private static final int TYPE_MASK      = 0x0F;  // reserve bit 0-3 for type
    private static final int FLAG_IS_SHARED = 0x80;  // bit 7: isShared
    private Streamer<Attribute> attrStreamer;
    private Streamer<Reference> vrefStreamer;
    private Streamer<LocalLocator> locStreamer;
    private XMLStore xmlstore;

    public NodeConverter() {
        this(null);
    }

    public NodeConverter(XMLStore xmlstore) {
        this.xmlstore = xmlstore;
        attrStreamer = DVMAttribute.getStreamer();
        vrefStreamer = Streamers.getPolymorphicStreamer(Reference.class);
        locStreamer = LocalLocator.getStreamer(true);
    }

    /**
     * If a bound is specified, all nodes within it will be saved even
     * if they already have a locator. This is used to migrate nodes from
     * a previous file system state to a new. Use the old file system bound
     * to identify nodes with deprecated locators. That is, saving them
     * anew will give them the new and correct locators.
     */
    public LocalLocator save(SystemNode node, LocalLocator bound,
                             FileSystem fs,
                             LocatorListener listener) throws IOException {
        try {
            synchronized (fs) {
                return saveTree(node, bound, fs, listener);
            }
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
    }

    private LocalLocator saveTree(SystemNode node, LocalLocator bound,
                             FileSystem fs,  LocatorListener listener)
                       throws IOException,  UnknownReferenceException {
        // Check if node is already saved
        LocalLocator loc = node.getLocator();
        if (loc != null) {
            // Do not save nodes outside bound (=always save nodes within)
            if (bound == null || !bound.isContained(loc)) {
                return loc;
            }
        }

        // Save children of node first to gather locators to them
        boolean hasOutgoing = false;
        List<LocalLocator> childLocs = null;
        if (node.getType() == Node.ELEMENT) {
            List<SystemNode> children = node.getChildren();
            childLocs = new ArrayList<LocalLocator>(children.size());
            for (SystemNode n : children) {
                LocalLocator childLoc = saveTree(n, bound, fs, listener);
                childLocs.add(childLoc);
                if (!fs.isContained(childLoc)) hasOutgoing = true;
            }
        }

        // Save the data in the node with children as locators
        loc = fs.allocate();

        int diskType = (node.getType() == Node.ELEMENT) ? ELEMENT : CHARDATA;
        DataOutput out = fs.getOutput(loc);
        out.writeByte(node.isShared() ? (diskType | FLAG_IS_SHARED) : diskType);
        Streamers.writeUTF(out, node.getNodeValue());
        if (diskType == ELEMENT) {
            Streamers.writeList(out, node.getAttributes(), attrStreamer);
            Streamers.writeShortInt(out, childLocs.size());
            for (LocalLocator l : childLocs) {
                locStreamer.toStream(out, l);
            }
        }

        // Optimization: Store LocalLocator in node to prevent redundant save
        node.setLocator(loc);
        assert loc.equals(node.getLocator()) : "node did not accept setLocator";
        if (listener != null) listener.nodeSaved(loc, node, hasOutgoing);
        return loc;
    }

    /**
     * The secondary <code>Reference</code> of the node is used to construct child
     * <code>NodeProxy</code>s with plausible secondary references.
     * TODO: the <code>secondaryRef</code> argument has been deprecated, please remove.
     *
     * @param loc the <code>LocalLocator</code> of the <code>Node</code> to be loaded
     * @param secondaryRef an optional secondary <code>Reference</code> or <code>null</code>
     * @param bound an optional <code>LocalLocator</code> (possibly <code>null</code>) giving
     *              bounds within which all nodes will be loaded. Nodes outside the bound
     *              will be proxied (i.e. lazy loading).
     * @param fs the <code>FileSystem</code> to load from
     */
    public SystemNode load(LocalLocator loc, Reference secondaryRef, LocalLocator bound,
                                                FileSystem fs, LocatorListener listener)
                                            throws IOException, UnknownLocatorException {
        synchronized (fs) {
            return loadTree(loc, secondaryRef, bound, fs, listener);
        }
    }

    private SystemNode loadTree(LocalLocator loc, Reference secondaryRef,
             LocalLocator bound, FileSystem fs, LocatorListener listener)
                             throws IOException, UnknownLocatorException {
        // LocalLocator Streamer might depend on the current FileSystemId
        DataInput in = fs.getInput(loc);

        int diskType = in.readByte();
        boolean isShared = (diskType & FLAG_IS_SHARED) != 0;
        diskType = diskType & TYPE_MASK;

        // Load data
        byte nodeType = (diskType == ELEMENT) ? Node.ELEMENT : Node.CHARDATA;
        String value = Streamers.readUTF(in);
        List<Attribute> attributes = null;
        List<SystemNode> children = null;

        boolean hasOutgoing = false;
        if (nodeType == Node.ELEMENT) {
            attributes = Streamers.readList(in, attrStreamer);

            // Read child Locators
            int size = Streamers.readShortInt(in);
            LocalLocator[] ls = new LocalLocator[size];
            for (int i=0; i<size; i++) {
                LocalLocator eloc = locStreamer.fromStream(in);
                if (listener != null) eloc = listener.locatorLoaded(eloc, loc, i);
                ls[i] = eloc;
            }

            // Process child Locators: load children within bound, proxy others
            SystemNode[] cs = new SystemNode[size];
            for (int i=0; i<size; i++) {
                SystemNode n = null;
                LocalLocator l = ls[i];
                if (bound != null && bound.isContained(l)) {
                    // child is within bound, so load it!
                    if (listener != null) n = listener.lookup(l);
                    if (n == null) {
                        if (l.equals(loc)) throw new IllegalStateException
                            ("Node "+loc+" is pointing to itself "+l);
                        n = loadTree(l, secondaryRef, bound, fs, listener);
                    }
                } else {
                    // child not within bound, so proxy it instead
                    hasOutgoing = true;
                    RelativeReference relRef = (secondaryRef==null) ? null :
                                     new RelativeReference(secondaryRef, i);
                    n = new XMLStoreNodeProxy(l, relRef, xmlstore);
                }
                cs[i] = n;
            }
            children = new UnmodifiableArrayList<SystemNode>(cs);
        }
        // use unchecked constructor
        SystemNode node = new DVMNode(
            nodeType, value, children, attributes, loc, isShared);
        if (listener != null) listener.nodeLoaded(loc, node, hasOutgoing);
        return node;
    }
}
