package org.planx.xmlstore.io;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

/**
 * A polymorphic <code>Streamer</code> that can handle multiple datatypes.
 * A datatype is registered using {@link #addStreamer}. Each datatype must
 * be given a unique id that is used to differentiate between the datatypes
 * when reading data back-in. When writing data the class of the object is
 * examined. If multiple <code>Streamer</code>s are registered which can all
 * handle a specific subclass, it is unspecified which <code>Streamer</code>
 * is used. The most specific <code>Streamer</code> is <i>not</i> chosen.
 * Thus, in general, registering a <code>Streamer</code> with a super class
 * of a class of another registered <code>Streamer</code> should be avoided.
 * Use the most specific class possible when registering <code>Streamer</code>s.
 *
 * @author Thomas Ambus
 */
public class PolymorphicStreamer<T> implements Streamer<T> {
    private Map<Byte,DynamicStreamer<? extends T>> map;

    public PolymorphicStreamer() {
        this.map = new HashMap<Byte,DynamicStreamer<? extends T>>();
    }

    public <E extends T> void addStreamer(byte id, Class<? extends E> cls,
                                        Streamer<E> streamer) {
        if (cls == null) throw new NullPointerException("Class is null");
        if (streamer == null) throw new NullPointerException
            ("Streamer is null");
        map.put(id, new DynamicStreamer<E>(cls, streamer));
    }

    public void toStream(DataOutput out, T obj) throws IOException {
        for (Map.Entry<Byte,DynamicStreamer<? extends T>> entry :
                                                map.entrySet()) {
            DynamicStreamer<? extends T> ds = entry.getValue();
            if (ds.cls.isInstance(obj)) {
                out.writeByte(entry.getKey());
                ds.toStream(out, obj);
                return;
            }
        }
        throw new ClassCastException
            ("No Streamer registered for "+obj.getClass());
    }

    public T fromStream(DataInput in) throws IOException {
        Byte id = in.readByte();
        DynamicStreamer<? extends T> ds = map.get(id);
        if (ds == null) throw new IOException
            ("No Streamer registered with id "+id);
        return ds.fromStream(in);
    }

    private class DynamicStreamer<S extends T> implements Streamer<T> {
        private Streamer<S> streamer;
        Class<? extends S> cls;

        public DynamicStreamer(Class<? extends S> cls, Streamer<S> streamer) {
            this.cls = cls;
            this.streamer = streamer;
        }

        public void toStream(DataOutput out, T obj) throws IOException {
            S o = cls.cast(obj);
            streamer.toStream(out, o);
        }

        public T fromStream(DataInput in) throws IOException {
            return streamer.fromStream(in);
        }
    }
}
