package org.planx.xmlstore.convert;

import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Stack;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.XMLStore;
import org.planx.xmlstore.nodes.*;
import org.planx.xmlstore.references.*;
import org.xml.sax.*;
import org.xml.sax.helpers.*;

/**
 * @author Kasper Bøgebjerg
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class SAXBuilder {
    private static final boolean useReader = true;
    private static XMLReader parser;
    private static SAXParser saxParser;

    static {
        if (useReader) {
            try {
                parser = XMLReaderFactory.createXMLReader("org.apache.xerces.parsers.SAXParser");
            } catch (SAXException e1) {
                try {
                    parser = XMLReaderFactory.createXMLReader();
                } catch (SAXException e2) {
                    throw new NoClassDefFoundError("No SAX parser is available: " + e2);
                }
            }
        } else {
            try {
                SAXParserFactory factory = SAXParserFactory.newInstance();
                factory.setValidating(false);
                saxParser = factory.newSAXParser();
            } catch (SAXException e) {
                throw new RuntimeException(e);
            } catch (ParserConfigurationException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private SAXBuilder() {}

    /**
     * Loads an XML document from a file.
     */
    public static Node build(String fileName) throws XMLException, IOException {
        return build(fileName, new DefaultListener());
    }

    /**
     * Loads an XML document from a file and saves it in XMLStore during loading.
     */
    public static Node build(String fileName, XMLStore xmlstore)
                               throws XMLException, IOException {
        return build(fileName, new XMLStoreNodeListener(xmlstore));
    }

    /**
     * Loads an XML document from a file and saves it in XMLStore during loading.
     * Saves will be done at every depth from the root down to the specified flush
     * depth.
     */
    public static Node build(String fileName, XMLStore xmlstore, int flushDepth)
                                              throws XMLException, IOException {
        return build(fileName, new XMLStoreNodeListener(xmlstore, flushDepth));
    }

    public static Node build(String fileName, NodeListener listener)
                                   throws XMLException, IOException {
        DVMHandler handler = null;
        InputStream in = null;
        try {
            handler = new DVMHandler(listener);
            if (useReader) {
                // XMLReader
                parser.setContentHandler(handler);
                parser.parse(fileName);
            } else {
                // SAXParser
                in = new BufferedInputStream(new FileInputStream(fileName));
                saxParser.parse(in, handler);
                in.close();
                in = null;
            }
            Node n = handler.getRoot();
            handler.clear();
            handler = null;
            return n;
        } catch (SAXException e) {
            throw new XMLException(e);
        } finally {
            if (handler != null) handler.clear();
            if (in != null) in.close();
        }
    }

    /**
     * Loads an XML document from an <code>InputSource</code>.
     */
    public static Node build(InputSource in) throws XMLException, IOException {
        return build(in, new DefaultListener());
    }

    /**
     * Loads an XML document from an <code>InputSource</code>
     * and saves it in XMLStore during loading.
     */
    public static Node build(InputSource in, XMLStore xmlstore)
                              throws XMLException, IOException {
        return build(in, new XMLStoreNodeListener(xmlstore));
    }

    public static Node build(InputSource in, NodeListener listener)
                                  throws XMLException, IOException {
        DVMHandler handler = null;
        try {
            handler = new DVMHandler(listener);
            parser.setContentHandler(handler);
            parser.parse(in);
            Node n = handler.getRoot();
            handler.clear();
            handler = null;
            return n;
        } catch (SAXException e) {
            throw new XMLException(e);
        } finally {
            if (handler != null) handler.clear();
        }
    }

    private static class DefaultListener implements NodeListener {
        public SystemNode nodeCreated(SystemNode node, int depth) {
            return node;
        }
    }

    private static class XMLStoreNodeListener implements NodeListener {
        private XMLStore xmlstore;
        private int flushDepth;

        XMLStoreNodeListener(XMLStore xmlstore) {
            this(xmlstore, 1);
        }

        XMLStoreNodeListener(XMLStore xmlstore, int flushDepth) {
            this.xmlstore = xmlstore;
            this.flushDepth = flushDepth;
        }

        public SystemNode nodeCreated(SystemNode node, int depth) {
            if (depth <= flushDepth) {
                try {
                    Reference ref = xmlstore.save(node);

                    List<SystemNode> children = node.getChildren();
                    for (int i=0,max=children.size(); i<max; i++) {
                        SystemNode child = children.get(i);
                        if (child instanceof NodeProxy) {
                            // FIXME: there's a bug
                            //child.setReference(new RelativeReference(ref, i));
                            ((NodeProxy) child).unload();
                        }
                    }
                    return new XMLStoreNodeProxy(ref, xmlstore);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
            return node;
        }
    }
}
