package org.planx.msd.number;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.planx.msd.*;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset of
 * <code>Integer</code> objects.
 *
 * @author Thomas Ambus
 */
public class IntegerDiscriminator extends NumberDiscriminator<Integer> {
    public IntegerDiscriminator(Memory memory) {super(memory);}

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                  final Extractor<U,? extends Integer,S> e) {
        IntExtractor<U,S> intExt = new IntExtractor<U,S>() {
            public int getLabel(U elm) {
                return e.getLabel(elm).intValue();
            }
            public S getValue(U elm) {
                return e.getValue(elm);
            }
        };
        return discriminate(values, intExt);
    }

    /**
     * Optimized integer discrimination, relying on an array of size 2^16,
     * performing two passes. Furthermore, it is optimistic in the sense
     * that it attempts to use only one pass, i.e. hopes that all ints
     * are 16 bit, forming the result directly from the first pass. If
     * an int larger than 16 bit is encountered
     * it breaks and restarts, performing two passes.
     */
    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                            IntExtractor<U,S> e) {
        Collection<List<U>> result1;
        Collection<List<S>> result2;
        List[] dictionary = memory.dictionary;
        int[] used = memory.used;
        int used_size = 0;
        boolean isFinished = true;

        // Discriminate according to lowest 16 bits
        // Do it optimistic assuming no int is more than 16 bit
        // if there are, break and re-do

        for (U elm : values) {
            int index = e.getLabel(elm);
            if ((index>>CHUNK_BITSIZE) != 0) {
                isFinished = false;
                break;
            }
            List list = dictionary[index];
            if (list == null) {
                used[used_size++] = index;
                list = new ArrayList<S>();
                dictionary[index] = list;
            }
            list.add(e.getValue(elm));
        }

        if (isFinished) {
            // All ints less than 16 bit, form result and return
            result2 = new ArrayList<List<S>>(used_size);
            for (int i=0; i<used_size; i++) {
                int index = used[i];
                result2.add(dictionary[index]);
                dictionary[index] = null;
            }
            return result2;
        }

        // Some ints were more than 16 bit, re-do

        used_size = 0;
        for (U elm : values) {
            int index = e.getLabel(elm) & CHUNK_MASK;
            List list = dictionary[index];
            if (list == null) {
                used[used_size++] = index;
                list = new ArrayList<U>();
                dictionary[index] = list;
            }
            list.add(elm);
        }

        result1 = new ArrayList<List<U>>(used_size);
        for (int i=0; i<used_size; i++) {
            int index = used[i];
            result1.add(dictionary[index]);
            dictionary[index] = null;
        }

        // For each block, discriminate according to highest 16 bits

        result2 = new ArrayList<List<S>>();
        for (List<U> block : result1) {
            used_size = 0;
            for (U elm : block) {
                int index = e.getLabel(elm);
                index = (index>>CHUNK_BITSIZE) & CHUNK_MASK;

                List list = dictionary[index];
                if (list == null) {
                    used[used_size++] = index;
                    list = new ArrayList<S>();
                    dictionary[index] = list;
                }
                list.add(e.getValue(elm));
            }

            for (int i=0; i<used_size; i++) {
                int index = used[i];
                result2.add(dictionary[index]);
                dictionary[index] = null;
            }
        }
        return result2;
    }

    // Unused since IntegerDiscriminator is optimized
    protected int bitsize() {return 32;}

    // Unused since IntegerDiscriminator is optimized
    protected <U> ChunkExtractor<U> chunkExtractor(Extractor<U,
                            ? extends Integer,?> e) {
        return new ChunkExtractor<U>(e) {
            public int getLabel(U elm) {
                Integer value = e.getLabel(elm);
                return (value.intValue() >> offset) & CHUNK_MASK;
            }
        };
    }
}
