package org.planx.msd.array;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.planx.msd.*;
import org.planx.msd.number.IntExtractor;
import org.planx.msd.number.IntegerDiscriminator;
import org.planx.msd.util.*;
import org.planx.util.Array;

/**
 * A <code>Discriminator</code> capable of discriminating a multiset
 * of arrays. A <code>Discriminator</code> for the type contained within
 * the arrays must be provided in the constructor. This implementation
 * performs an initial partitioning on the sizes of the arrays.
 * <p>
 * <b>Note that this implementation is not synchronized.</b> If multiple
 * threads access  an instance of this class concurrently, it must be
 * synchronized externally.
 *
 * @author Thomas Ambus
 */
/*
 * TODO: Merge code with SizeRandomAccessListDiscriminator. Aside from
 * type T[] points where the code differs are marked.
 */
public class ArrayDiscriminator<T> extends AbstractDiscriminator<T[]> {
    private static final int INIT_CAPACITY = 10;
    private Discriminator<T> disc;
    private IntegerDiscriminator intDisc;

    /**
     * Creates a new <code>ArrayDiscriminator</code> able to discriminate
     * a multiset of arrays. The instance will reuse the memory of the
     * specified <code>Memory</code>.
     */
    public ArrayDiscriminator(Discriminator<T> d, Memory memory) {
        disc = d;
        intDisc = new IntegerDiscriminator(memory);
    }

    public <U,S> Collection<List<S>> discriminate(
                            List<? extends U> values,
                            final Extractor<U,? extends T[],S> e) {

        // Check fast way out: only zero or one or two elements

        int vsize = values.size();
        if (vsize == 0) return Collections.emptyList();
        if (vsize == 1) {
            List<S> l1 = Collections.singletonList(
                        e.getValue(values.get(0)));
            return Collections.singletonList(l1);
        }

        // Declarations

        Collection<List<S>> result = new ArrayList<List<S>>();
        Collection<List<U>> partition;
        ElementExtractor<U,T> ext = new ElementExtractor<U,T>(e);

        int[] indexes = new int[INIT_CAPACITY]; // Note: Initialized to 0
        List[] work = new List[INIT_CAPACITY];
        int work_capacity = INIT_CAPACITY;
        int work_head = 0;

        // Perform initial partitioning based on list size

        IntExtractor<U,U> intExt = new IntExtractor<U,U>() {
            public int getLabel(U elm) {
                return e.getLabel(elm).length; // DIFFERS FROM LISTDISC
            }
            public U getValue(U elm) {
                return elm;
            }
        };
        partition = intDisc.discriminate(values, intExt);

        // Add all size-equivalent blocks to unfinished work

        for (List<U> block : partition) {
            // If the block only contains one element it cannot
            // be partitioned any further
            if (block.size() > 1) {
                if (work_head+1 >= work_capacity) {
                    work = Array.ensureCapacity(
                        work, work_head, work_capacity+1);
                    indexes = Array.ensureCapacity(
                        indexes, work_head, work_capacity+1);
                    work_capacity = work.length;
                }
                work[work_head] = block;
                work_head++;
            } else {
                result.add(Discriminators.valueList(block, e));
            }
        }

        // Refine until there are no unfinished blocks

        while (work_head > 0) {
            if (work_head < work_capacity) work[work_head] = null;
            work_head--;

            // Subpartition current unfinished block
            // it is guaranteed that all lists in the block have
            // the same size

            List<U> block = work[work_head];
            int index = indexes[work_head];

            if (e.getLabel(block.get(0)).length > index) {

                // End not reached, partition further

                ext.index = index;
                partition = disc.discriminate(block, ext);

                // Add blocks in partition to result or unfinished

                index++;
                for (List<U> subBlock : partition) {

                    // If the block only contains one element it
                    // cannot be partitioned any further

                    if (subBlock.size() > 1) {
                        if (work_head+1 >= work_capacity) {
                            work = Array.ensureCapacity(
                                work, work_head, work_capacity+1);
                            indexes = Array.ensureCapacity(
                                indexes, work_head, work_capacity+1);
                            work_capacity = work.length;
                        }
                        work[work_head] = subBlock;
                        indexes[work_head] = index;
                        work_head++;
                    } else {
                        result.add(Discriminators.valueList(subBlock, e));
                    }
                }
            } else {
                // End of all lists in block reached, i.e.
                // they are equivalent
                result.add(Discriminators.valueList(block, e));
            }
        }
        return result;
    }

    private static class ElementExtractor<E,F> implements Extractor<E,F,E> {
        private Extractor<E,? extends F[],?> e;
        public int index = 0;

        public ElementExtractor(Extractor<E,? extends F[],?> e) {
            this.e = e;
        }
        public F getLabel(E elm) {
            F[] arr = e.getLabel(elm);
            return arr[index];  // DIFFERS FROM LISTDISC
        }
        public E getValue(E elm) {
            return elm;
        }
    }
}
