package org.nuiton.topia.framework;

/*
 * #%L
 * ToPIA :: Persistence
 * $Id: TopiaHibernateEventListener.java 2847 2013-10-25 12:03:15Z bleny $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-4/topia-persistence/src/main/java/org/nuiton/topia/framework/TopiaHibernateEventListener.java $
 * %%
 * Copyright (C) 2004 - 2013 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.event.spi.PostDeleteEvent;
import org.hibernate.event.spi.PostDeleteEventListener;
import org.hibernate.event.spi.PostInsertEvent;
import org.hibernate.event.spi.PostInsertEventListener;
import org.hibernate.event.spi.PostLoadEvent;
import org.hibernate.event.spi.PostLoadEventListener;
import org.hibernate.event.spi.PostUpdateEvent;
import org.hibernate.event.spi.PostUpdateEventListener;
import org.hibernate.event.spi.PreDeleteEvent;
import org.hibernate.event.spi.PreDeleteEventListener;
import org.hibernate.event.spi.PreInsertEvent;
import org.hibernate.event.spi.PreInsertEventListener;
import org.hibernate.event.spi.PreLoadEvent;
import org.hibernate.event.spi.PreLoadEventListener;
import org.hibernate.event.spi.PreUpdateEvent;
import org.hibernate.event.spi.PreUpdateEventListener;
import org.hibernate.event.spi.SaveOrUpdateEvent;
import org.hibernate.event.spi.SaveOrUpdateEventListener;
import org.nuiton.topia.TopiaDaoSupplier;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.AbstractTopiaDao;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityContextable;

/**
 * @author Arnaud Thimel <thimel@codelutin.com>
 */
public class TopiaHibernateEventListener implements PreInsertEventListener, PostInsertEventListener,
        PreLoadEventListener, PostLoadEventListener, PreUpdateEventListener, PostUpdateEventListener,
        PreDeleteEventListener, PostDeleteEventListener, SaveOrUpdateEventListener {

    private static final Log log = LogFactory.getLog(TopiaHibernateEventListener.class);

    private static final long serialVersionUID = -9206039888626756924L;

    protected TopiaContextImplementor rootContext;

    public TopiaHibernateEventListener(TopiaContextImplementor rootContext) {
        this.rootContext = rootContext;
    }

    /**
     * Recherche le context utilisant la session hibernate passe en
     * parametre
     *
     * @param parent    le context parent
     * @param hibernate la session hibernate que doit utiliser le
     *                  TopiaContext pour etre retourne
     * @return le TopiaContext utilisant cette session hibernate ou null si
     *         aucun TopiaContext n'utilise cette session.
     */
    protected TopiaContextImplementor getContext(
            TopiaContextImplementor parent, Session hibernate) {
        TopiaContextImplementor result = null;

        // FD-20100421 : Ano #546 : no need to copy childContext, the
        // {@link #getChildContext()} provides a thread-safe copy to iterate
        // on it.
//            Set<TopiaContextImplementor> contextChilds = new HashSet<TopiaContextImplementor>(parent.getChildContext());
        if (parent != null) { // TODO AThimel 11/10/13 It should never be null, fix it
            for (TopiaContextImplementor context : parent.getChildContext()) {

                // by sletellier 24/09/09 : Fix concurent acces error
                //            ArrayList<TopiaContextImplementor> children = new ArrayList(parent.getChildContext());
                //            for (TopiaContextImplementor context : children) {
                try {
                    if (context.getHibernate() == hibernate) {
                        result = context;
                    } else {
                        // TODO: poussin 20090706 on pourrait ameliorer en ne faisant pas un parcours recursif, en utilisant la liste children (sans doute a transformer en stack)
                        result = getContext(context, hibernate);
                    }
                    if (result != null) {
                        break;
                    }
                } catch (TopiaException eee) {
                    if (log.isWarnEnabled()) {
                        log.warn("Error durant la recherche d'un context pour"
                                + " lancer un event", eee);
                    }
                }
            }
        }
        return result;
    }

    private void attachContext(Object entity,
                               TopiaDaoSupplier daoSupplier) {
        if (entity instanceof TopiaEntityContextable) {
            TopiaEntityContextable topiaEntityContextable = (TopiaEntityContextable) entity;
            if (topiaEntityContextable.getTopiaDAOSupplier() == null) {
                try {
                    topiaEntityContextable.setTopiaDAOSupplier(daoSupplier);
                } catch (TopiaException eee) {
                    if (log.isWarnEnabled()) {
                        log.warn("Impossible d'initialiser le TopiaContext"
                                + " sur cette entité : " + topiaEntityContextable,
                                eee);
                    }
                }
            }
        }
    }

        /* Création */

    @Override
    public boolean onPreInsert(PreInsertEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event.getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            TopiaEntity entity = (TopiaEntity) event.getEntity();
            context.getFiresSupport().fireOnPreCreate(context, entity, event.getState());

            // when using composition, hibernate will persist entities by him self
            // entity must have an id in this case.
            if (StringUtils.isBlank(entity.getTopiaId())) {
                if (log.isDebugEnabled()) {
                    log.debug("Adding topia id on entity " + entity.getClass().getSimpleName());
                }
            }
        }

        return false;
    }

    @Override
    public void onPostInsert(PostInsertEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event
                .getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            context.getFiresSupport().fireOnPostCreate(context, (TopiaEntity) event.getEntity(), event.getState());
        }
    }

        /* Chargement */

    @Override
    public void onPreLoad(PreLoadEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event
                .getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            //                try {
            context.getFiresSupport().fireOnPreLoad(context, (TopiaEntity) event.getEntity(), event.getState());
            //TODO (thimel 20071213) On commente pour le moment @see(TopiaDAOHibernate#filterElements)
            //                } catch (TopiaVetoException tve) {
            //                    //On ne fait pas de remontee d'exception
            // vers Hibernate pour le preLoad, on va agir au niveau du DAO
            //                }
        }
    }

    @Override
    public void onPostLoad(PostLoadEvent event) {
//            TopiaContextImplementor context = getContext(rootContext, event
//                    .getSession());
        TopiaDaoSupplier daoSupplier = null; // TODO brendan 30/09/13 Implment
        if (daoSupplier != null && event.getEntity() instanceof TopiaEntity) {
            attachContext(event.getEntity(), daoSupplier);
            TopiaEntity entity = (TopiaEntity) event.getEntity();
            AbstractTopiaDao<? extends TopiaEntity> dao = (AbstractTopiaDao) daoSupplier.getDao(entity.getClass());
            dao.getTopiaFiresSupport().fireOnPostLoad(dao, (TopiaEntity) event.getEntity(), new Object[]{});
        }
    }

        /* Modification */

    @Override
    public boolean onPreUpdate(PreUpdateEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event
                .getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            context.getFiresSupport().fireOnPreUpdate(context, (TopiaEntity) event.getEntity(), event.getOldState());
        }
        return false;
    }

    @Override
    public void onPostUpdate(PostUpdateEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event
                .getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            context.getFiresSupport().fireOnPostUpdate(context, (TopiaEntity) event.getEntity(), event.getState());
        }
        // FIXME indexation
        // if (!(entity instanceof NotIndexable) && context.isIndexEnabled()) {
        // context.getIndexEnginImplementor().recordForIndexation(id, event.getState());
        //                }
    }

        /* Suppression */

    @Override
    public boolean onPreDelete(PreDeleteEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event
                .getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            context.getFiresSupport().fireOnPreDelete(context, (TopiaEntity) event.getEntity(), event.getDeletedState());
        }
        return false;
    }

    @Override
    public void onPostDelete(PostDeleteEvent event) {
        TopiaContextImplementor context = getContext(rootContext, event
                .getSession());
        if (context != null && event.getEntity() instanceof TopiaEntity) {
            context.getFiresSupport().fireOnPostDelete(context, (TopiaEntity) event.getEntity(), event.getDeletedState());
        }
//             FIXME indexation
//             if (!(entity instanceof NotIndexable) && context.isIndexEnabled()) {
//             context.getIndexEnginImplementor().recordForIndexation(id, null);
//                            }
    }

    @Override
    public void onSaveOrUpdate(SaveOrUpdateEvent event) throws HibernateException {
        try {
            // this event is called when hibernate try to persist entities
            // using cascade (save)
            TopiaContextImplementor context = getContext(rootContext, event.getSession());
            // warning, event.getEntity() return null here :(
            if (event.getObject() instanceof TopiaEntity) {
                TopiaEntity entity = (TopiaEntity) event.getObject();
                if (StringUtils.isBlank(entity.getTopiaId())) {
                    if (log.isDebugEnabled()) {
                        log.debug("Adding topiaId into entity " + entity.getClass());
                    }
                    // FIXME echatellier 20130713 : hack to find interface for current entity class
                    // #newTopiaId only accept interface
                    Class entityInterface = Class.forName(event.getEntityName().replace("Impl", ""));
                    String topiaId = context.getTopiaIdFactory().newTopiaId(entityInterface, entity);
                    entity.setTopiaId(topiaId);
                }
            }
        } catch (ClassNotFoundException ex) {
            throw new HibernateException("Can't set id", ex);
        }
    }
}
