/*
 * #%L
 * Maven License Plugin
 * 
 * $Id: AbstractLicenseWithDescriptorMojo.java 1797 2010-06-29 10:55:43Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/maven-license-plugin/tags/maven-license-plugin-2.3.1/src/main/java/org/nuiton/license/plugin/AbstractLicenseWithDescriptorMojo.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.license.plugin;

import org.apache.commons.lang.StringUtils;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.nuiton.license.plugin.header.FileHeader;
import org.nuiton.license.plugin.header.transformer.FileHeaderTransformer;
import org.nuiton.license.plugin.model.License;
import org.nuiton.license.plugin.model.LicenseStore;
import org.nuiton.license.plugin.model.descriptor.FileSet;
import org.nuiton.license.plugin.model.descriptor.Header;
import org.nuiton.license.plugin.model.descriptor.LicenseProjectDescriptor;
import org.nuiton.license.plugin.model.descriptor.io.xpp3.LicenseProjectDescriptorXpp3Reader;
import org.nuiton.plugin.PluginWithEncoding;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Abstract mojo which using {@link #descriptor} file and owns a
 * {@link #licenseStore}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 2.1
 */
public abstract class AbstractLicenseWithDescriptorMojo extends AbstractLicenseMojo implements PluginWithEncoding {

    /**
     * Encoding used to read and writes files.
     * <p/>
     * <b>Note:</b> If nothing is filled here, we will use the system
     * property {@code file.encoding}.
     *
     * @parameter expression="${license.encoding}" default-value="${project.build.sourceEncoding}"
     * @required
     * @since 2.1
     */
    private String encoding;

    /**
     * The project license descriptor file.
     *
     * @parameter expression="${license.descriptor}" default-value="src/license/project.xml"
     * @required
     * @since 2.1
     */
    private File descriptor;

    /**
     * To specify an external extra licenses repository resolver (says the base
     * url of the repository where the {@code license.properties} is present).
     *
     * @parameter expression="${license.licenseResolver}"
     * @since 2.1
     */
    private String licenseResolver;

    /**
     * A flag to keep a backup of every modified file.
     *
     * @parameter expression="${license.keepBackup}"  default-value="false"
     * @since 2.1
     */
    private boolean keepBackup;

    /**
     * All available header transformers.
     *
     * @component role="org.nuiton.license.plugin.header.transformer.FileHeaderTransformer"
     * @since 2.1
     */
    private Map<String, FileHeaderTransformer> transformers;

    /** store of licenses */
    private LicenseStore licenseStore;

    /** descriptor of project */
    private LicenseProjectDescriptor licenseProjectDescriptor;

    /**
     * When is sets to {@code true}, will skip execution.
     * <p/>
     * This will take effects in method {@link #checkSkip()}.
     * So the method {@link #doAction()} will never be invoked.
     *
     * @return {@code true} if goal will not be executed
     */
    public abstract boolean isSkip();

    /**
     * Changes internal state {@code skip} to execute (or not) goal.
     *
     * @param skip new state value
     */
    public abstract void setSkip(boolean skip);

    @Override
    protected boolean checkSkip() {
        if (isSkip()) {
            getLog().info("skip flag is on, will skip goal.");
            return false;
        }
        if (!getDescriptor().exists()) {
            getLog().warn("skip - could not find descriptor " + getDescriptor());
            return false;
        }
        return super.checkSkip();
    }

    @Override
    protected void init() throws Exception {

        if (isSkip()) {
            return;
        }

        // init licenses store
        LicenseStore licenseStore = createLicenseStore(getLicenseResolver());
        setLicenseStore(licenseStore);

        // load project descriptor
        File descriptorFile = new File(getDescriptor().getAbsolutePath());
        if (!descriptorFile.exists()) {

            // try to find the file in the direct parent module
            MavenProject mavenProject = getProject().getParent();
            if (mavenProject != null) {

                File basedir = getProject().getBasedir();

                // try with the parent
                String path = descriptorFile.getAbsolutePath().substring(
                        basedir.getAbsolutePath().length() + 1);
                setDescriptor(new File(mavenProject.getBasedir() +
                                       File.separator + path));
                if (isVerbose()) {
                    getLog().info("try in parent module " + getDescriptor());
                }

            }
            descriptorFile = new File(getDescriptor().getAbsolutePath());
        }

        if (!descriptorFile.exists()) {
//            throw new MojoFailureException(
//                    "could not find descriptor " + descriptorFile);
            return;
        }

        getLog().info("Loading descriptor " + descriptorFile);
        LicenseProjectDescriptor licenseProject = null;
        FileReader reader = new FileReader(descriptorFile);
        try {
            licenseProject =
                    new LicenseProjectDescriptorXpp3Reader().read(reader);
            setLicenseProjectDescriptor(licenseProject);
        } finally {
            reader.close();
        }

        // check that license project is sane (known licenses + header types)
        // + populate default license name in LicenseSet (with main License) if
        // none given for licenseSet
        validateLicenseProjectDescriptor(licenseProject,
                                         licenseStore,
                                         getTransformers()
        );
    }

    @Override
    public final String getEncoding() {
        return encoding;
    }

    @Override
    public final void setEncoding(String encoding) {
        this.encoding = encoding;
    }


    public File getDescriptor() {
        return descriptor;
    }

    public boolean isKeepBackup() {
        return keepBackup;
    }

    public Map<String, FileHeaderTransformer> getTransformers() {
        return transformers;
    }

    public String getLicenseResolver() {
        return licenseResolver;
    }

    public LicenseStore getLicenseStore() {
        return licenseStore;
    }

    public LicenseProjectDescriptor getLicenseProjectDescriptor() {
        return licenseProjectDescriptor;
    }

    public void setKeepBackup(boolean keepBackup) {
        this.keepBackup = keepBackup;
    }

    public void setDescriptor(File descriptor) {
        this.descriptor = descriptor;
    }

    public void setLicenseResolver(String licenseResolver) {
        this.licenseResolver = licenseResolver;
    }

    public void setTransformers(
            Map<String, FileHeaderTransformer> transformers) {
        this.transformers = transformers;
    }

    public void setLicenseStore(LicenseStore licenseStore) {
        this.licenseStore = licenseStore;
    }


    public void setLicenseProjectDescriptor(
            LicenseProjectDescriptor licenseProjectDescriptor) {
        this.licenseProjectDescriptor = licenseProjectDescriptor;
    }

    public License getMainLicense()
            throws IllegalArgumentException, IllegalStateException {
        LicenseProjectDescriptor licenseProject = getLicenseProjectDescriptor();
        if (licenseProject == null) {
            throw new IllegalStateException("No license project initialized!");
        }
        String mainLicenseName = licenseProject.getMainLicense();
        if (StringUtils.isEmpty(mainLicenseName)) {
            throw new IllegalArgumentException(
                    "main license name can not be null, nor empty");
        }
        License mainLicense = getLicense(mainLicenseName);
        return mainLicense;
    }

    public License getLicense(String licenseName)
            throws IllegalArgumentException, IllegalStateException {
        if (StringUtils.isEmpty(licenseName)) {
            throw new IllegalArgumentException(
                    "licenseName can not be null, nor empty");
        }
        LicenseStore licenseStore = getLicenseStore();
        if (licenseStore == null) {
            throw new IllegalStateException("No license store initialized!");
        }
        License mainLicense = licenseStore.getLicense(licenseName);
        return mainLicense;
    }

    public FileHeaderTransformer getTransformer(String transformerName)
            throws IllegalArgumentException, IllegalStateException {
        if (StringUtils.isEmpty(transformerName)) {
            throw new IllegalArgumentException(
                    "transformerName can not be null, nor empty!");
        }
        Map<String, FileHeaderTransformer> transformers = getTransformers();
        if (transformers == null) {
            throw new IllegalStateException("No transformers initialized!");
        }
        FileHeaderTransformer transformer =
                transformers.get(transformerName);
        return transformer;
    }

    /**
     * Build a default header given the parameters.
     *
     * @param license         the license type ot use in header
     * @param projectName     project name as header description
     * @param inceptionYear   first year of copyright
     * @param copyrightHolder holder of copyright
     * @param encoding        encoding used to read or write files
     * @param addSvnKeyWords  a flag to add in description section svn keywords
     * @return the new file header
     * @throws IOException if any problem while creating file header
     */
    protected FileHeader buildDefaultFileHeader(
            License license,
            String projectName,
            String inceptionYear,
            String copyrightHolder,
            boolean addSvnKeyWords,
            String encoding) throws IOException {
        FileHeader result = new FileHeader();

        StringBuilder buffer = new StringBuilder();
        buffer.append(projectName);
        if (addSvnKeyWords) {
            // add svn keyworks
            char ls = FileHeaderTransformer.LINE_SEPARATOR;
            buffer.append(ls);

            // breaks the keyword otherwise svn will update them here
            //TC-20100415 : do not generate thoses redundant keywords
//            buffer.append(ls).append("$" + "Author$");
//            buffer.append(ls).append("$" + "LastChangedDate$");
//            buffer.append(ls).append("$" + "LastChangedRevision$");
            buffer.append(ls).append("$" + "Id$");
            buffer.append(ls).append("$" + "HeadURL$");

        }
        result.setDescription(buffer.toString());
        if (getLog().isDebugEnabled()) {
            getLog().debug("header description : " + result.getDescription());
        }

        String licenseContent = license.getHeaderContent(encoding);
        result.setLicense(licenseContent);

        Integer firstYear = Integer.valueOf(inceptionYear);
        result.setCopyrightFirstYear(firstYear);

        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date());
        Integer lastYear = cal.get(Calendar.YEAR);
        if (firstYear < lastYear) {
            result.setCopyrightLastYear(lastYear);
        }
        result.setCopyrightHolder(copyrightHolder);
        return result;
    }

    protected void validateLicenseProjectDescriptor(
            LicenseProjectDescriptor licenseProjectDescriptor,
            LicenseStore licenseStore,
            Map<String, FileHeaderTransformer> transformers)
            throws MojoFailureException {

        List<String> licenseNames = Arrays.asList(licenseStore.getLicenseNames());

        // check licenses is known
        String mainLicense = licenseProjectDescriptor.getMainLicense();
        if (licenseStore.getLicense(mainLicense) == null) {
            throw new MojoFailureException(
                    "main license '" + mainLicense +
                    "' is unknown, use one of " + licenseNames);
        }

        Header singleHeader = licenseProjectDescriptor.getHeader();
        if (singleHeader != null) {

            // add it to set (and remove it from single)
            licenseProjectDescriptor.addHeader(singleHeader);
            licenseProjectDescriptor.setHeader(null);
        }

        for (Header header : licenseProjectDescriptor.getHeaders()) {

            String headerType = header.getCommentStyle();

            if (getTransformer(headerType) == null) {
                throw new MojoFailureException(
                        "headerType '" + headerType +
                        "' is unknown, use one of " + transformers.keySet());
            }
            String licenseName = header.getLicenseName();
            if (StringUtils.isEmpty(licenseName)) {

                // use the main license
                header.setLicenseName(mainLicense);
            } else {

                // check license name
                if (licenseStore.getLicense(licenseName) == null) {
                    throw new MojoFailureException(
                            "license '" + licenseName +
                            "' is unknown, use one of " + licenseNames);
                }
            }

            FileSet singleFileSet = header.getFileSet();
            if (singleFileSet != null) {

                // add it to set (and remove it from single)
                header.addFileSet(singleFileSet);
                header.setFileSet(null);
            }
            for (FileSet fileSet : header.getFileSets()) {

                String singleInclude = fileSet.getInclude();
                if (singleInclude != null) {

                    // add it to set (and remove it from single)
                    fileSet.addInclude(singleInclude);
                    fileSet.setInclude(null);
                }

                String singleExclude = fileSet.getExclude();
                if (singleExclude != null) {

                    // add it to set (and remove it from single)
                    fileSet.addExclude(singleExclude);
                    fileSet.setExclude(null);
                }
            }
        }
    }

}
