/*
 * #%L
 * JRedmine :: Maven plugin
 * 
 * $Id: DisplayDataMojo.java 194 2011-05-20 22:55:40Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/jredmine/tags/jredmine-1.3/jredmine-maven-plugin/src/main/java/org/nuiton/jredmine/plugin/DisplayDataMojo.java $
 * %%
 * Copyright (C) 2009 - 2010 Tony Chemit, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jredmine.plugin;

import org.apache.maven.plugin.MojoFailureException;
import org.nuiton.jredmine.RedmineServiceImplementor;
import org.nuiton.jredmine.model.I18nAble;
import org.nuiton.jredmine.model.IdAble;
import org.nuiton.jredmine.model.ModelHelper;
import org.nuiton.jredmine.model.RedmineModelEnum;
import org.nuiton.jredmine.model.RedmineModelScope;
import org.nuiton.plugin.PluginHelper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Display in console some data from redmine's server.
 * <p/>
 * This goals requires no authentication to the server.
 *
 * @author tchemit
 * @goal display-data
 * @since 1.0.0
 */
public class DisplayDataMojo extends AbstractRedmineMojo {

    /**
     * The data types to display : {@code issueSatus}, {@code issuePriority},
     * {@code issueCategory}, {@code tracker} and {@code user}.
     * <p/>
     * {@code issueStatus} and {@code issuePriority} are common for all
     * projects, so for those datas, no extrat configuration is required.
     * <p/>
     * {@code isseCategoriy}, {@code tracker} and {@code user} are project
     * specific, so you have to fill also the {@code projectId} parameter to
     * obtain data for those types.
     * <p/>
     * Note : you can specifiy several types separated by comma
     *
     * @parameter expression="${types}"
     * @required
     * @since 1.0.0
     */
    protected String types;

    /**
     * A flag to sort of not retrived data by id.
     *
     * @parameter expression="${sortById}" default-value="true"
     * @required
     * @since 1.0.0
     */
    protected boolean sortById;

    /**
     * Flag to know if anonymous connexion to redmine server is required.
     * <p/>
     * For this goal, the default value is {@code true}
     * <p/>
     * <b>Note:</b> If set to {@code false}, you should fill {@link #username}
     * and {@link #password} properties.
     *
     * @parameter expression="${anonymous}" default-value="true"
     * @since 1.1.3
     */
    protected boolean anonymous;

    /** list of actions to perform */
    protected List<RedmineModelEnum> actions;

    public DisplayDataMojo() {
        super(false, false, false);
    }

    @Override
    protected boolean checkRunOnceDone() {
        return false;
    }

    @Override
    protected boolean isGoalSkip() {
        return false;
    }

    @Override
    protected boolean isRunOnce() {
        return false;
    }

    @Override
    protected void init() throws Exception {

        if (versionId != null) {
            // always remove snapshot (TODO-TC20100124 perharps should it be a flag ?)
            versionId = PluginHelper.removeSnapshotSuffix(versionId);
        }

        actions = getDownloadActions(types,
                                     RedmineModelEnum.project,
                                     RedmineModelEnum.version,
                                     RedmineModelEnum.issue,
                                     RedmineModelEnum.news,
                                     RedmineModelEnum.issueStatus,
                                     RedmineModelEnum.issuePriority,
                                     RedmineModelEnum.issueCategory,
                                     RedmineModelEnum.tracker,
                                     RedmineModelEnum.user);
        if (actions == null || actions.isEmpty()) {
            // no need to connect, nothing to do...
            return;
        }
        super.init();
    }

    @Override
    public boolean isAnonymous() {
        return anonymous;
    }

    @Override
    public void setAnonymous(boolean anonymous) {
        this.anonymous = anonymous;
    }

    @Override
    protected boolean checkSkip() {
        if (actions == null || actions.isEmpty()) {
            // no data to treate
            getLog().warn("No data types detected, you must fill the required parameter types, will skip goal");
            return false;
        }
        boolean b = super.checkSkip();
        return b;
    }

    @Override
    protected void doAction() throws Exception {

        StringBuilder buffer = new StringBuilder("\n");
        for (RedmineModelEnum entry : actions) {

            RedmineModelScope scope = entry.getScope();

            String requestName = entry.getRequestAll();
            IdAble[] datas = (IdAble[]) ((RedmineServiceImplementor) service).getDatas(requestName, entry.getModelType(), projectId, versionId);
            List<IdAble> list = Arrays.asList(datas);
            if (sortById) {
                Collections.sort(list, ModelHelper.ID_ABLE_COMPARATOR);
            }
            buffer.append("\nValues of '").append(entry).append('\'');
            if (scope == RedmineModelScope.project || scope == RedmineModelScope.version) {
                buffer.append(" for project '").append(projectId).append("'");
            }
            if (scope == RedmineModelScope.version) {
                buffer.append(" and version '").append(versionId).append("'");
            }
            buffer.append('\n');
            for (IdAble data : list) {
                buffer.append("  - ").append(data.getId()).append(" = ").append(((I18nAble) data).getName()).append('\n');
            }
        }
        System.out.println(buffer.toString());
    }

    protected List<RedmineModelEnum> getDownloadActions(String dataTypes, RedmineModelEnum... universe) throws MojoFailureException {

        if (universe.length == 0) {
            throw new MojoFailureException("List of accepted data types is empty!");
        }

        List<RedmineModelEnum> results = new ArrayList<RedmineModelEnum>();

        List<RedmineModelEnum> universeList = new ArrayList<RedmineModelEnum>(Arrays.asList(universe));

        for (String dataType : dataTypes.split(",")) {

            RedmineModelEnum result;
            try {
                result = RedmineModelEnum.valueOf(dataType.trim());
            } catch (Exception e) {
                throw new MojoFailureException("Unkown data type '" + dataType + "', Accepted types are : " + universeList);
            }
            if (!universeList.contains(result)) {
                // can not accept these value
                throw new MojoFailureException("The data type '" + result + "' can not be used, Accepted types are : " + universeList);
            }

            RedmineModelScope scope = result.getScope();

            if (scope == RedmineModelScope.project || scope == RedmineModelScope.version) {

                // check projectId used
                if (projectId == null || projectId.isEmpty()) {
                    throw new MojoFailureException("The data type '" + result + "', depends on a project, but the 'projectId' was not filled.");
                }
            }
            if (scope == RedmineModelScope.version) {
                // check versionId use
                if (versionId == null || versionId.isEmpty()) {
                    throw new MojoFailureException("The data type '" + result + "', depends on a version, but the 'versionId' was not filled.");
                }
            }
            results.add(result);
        }
        return results;
    }
}
