package jaxx.runtime.swing;

import java.applet.Applet;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;
import java.util.Vector;
import javax.help.CSH;
import javax.help.HelpBroker;
import javax.help.HelpSet;
import javax.swing.AbstractButton;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import jaxx.runtime.JAXXContext;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * La classe pour encapsuler l'aide de l'application.
 *
 * @param <B> le type de broker
 * @author tony
 * @since 1.4
 */
public abstract class JaxxHelpBroker<B extends JaxxHelpBroker<?>> {

    public static final String JAXX_CONTEXT_ENTRY = "jaxxcontext";
    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(JaxxHelpBroker.class);
    protected final String helpsetName;
    protected final String defaultID;
    protected final String helpKey;
    // Main HelpSet & Broker
    protected final HelpSet helpset;
    protected final HelpBroker helpBroker;
    protected Hashtable<Component, Cursor> cursors;
    protected Cursor onItemCursor;
    protected final Map<Component, String> cache;

    protected JaxxHelpBroker(String helpsetName, String helpKey, String defaultID) {
        if (helpsetName == null) {
            throw new NullPointerException("parameter helpsetName can not be null!");
        }
        this.helpsetName = helpsetName;
        this.helpKey = helpKey;
        this.defaultID = defaultID;
        cache = new HashMap<Component, String>();
        try {
            ClassLoader cl = getClass().getClassLoader();
            URL url = HelpSet.findHelpSet(cl, helpsetName);
            helpset = new HelpSet(cl, url);
            helpBroker = helpset.createHelpBroker();
        } catch (Exception ee) {
            throw new IllegalStateException("could not find help set " + helpsetName + " for reason " + ee.getMessage(), ee);
        }
    }

    public void prepareUI(JAXXObject c) {
        if (c == null) {
            throw new NullPointerException("parameter c can not be null!");
        }

        // l'ui doit avoir un boutton showHelp
        AbstractButton help = getShowHelpButton(c);

        if (help == null) {
            if (log.isDebugEnabled()) {
                log.debug("no showButton detected for " + c.getClass());
            }
        } else {

            // attach context to button
            help.putClientProperty(JAXX_CONTEXT_ENTRY, c.getDelegateContext());

            // add tracking action
            ActionListener listener = getShowHelpAction();
            if (log.isDebugEnabled()) {
                log.debug("adding tracking action " + listener);
            }
            help.addActionListener(listener);

        }
        if (log.isDebugEnabled()) {
            log.debug("done for " + c);
        }
    }

    public HelpBroker getHelpBroker() {
        return helpBroker;
    }

    public String getHelpKey() {
        return helpKey;
    }

    public HelpSet getHelpset() {
        return helpset;
    }

    public String getHelpsetName() {
        return helpsetName;
    }

    public String getDefaultID() {
        return defaultID;
    }

    public void showHelpSet() {
        if (log.isDebugEnabled()) {
            log.debug(this);
        }
        new CSH.DisplayHelpFromSource(helpBroker);
    }

    public void showHelp(JAXXContext context, String helpId) {
    }

    public void installUI(Component comp, String helpId) {
        CSH.setHelpIDString(comp, helpId);
        if (log.isDebugEnabled()) {
            log.debug(helpId + " : " + comp.getName());
        }
        cache.put(comp, helpId);
    }

    public class ShowHelpForTrackedComponentAction implements ActionListener {

        @Override
        public void actionPerformed(ActionEvent e) {
            AbstractButton source = (AbstractButton) e.getSource();

            JAXXContext context = (JAXXContext) source.getClientProperty(JAXX_CONTEXT_ENTRY);

            // prepare cursor
            onItemCursor = (Cursor) UIManager.get("HelpOnItemCursor");
            Vector<?> topComponents = null;
            cursors = null;

            if (onItemCursor != null) {
                cursors = new Hashtable<Component, Cursor>();
                topComponents = getTopContainers(source);
                Enumeration<?> enums = topComponents.elements();
                while (enums.hasMoreElements()) {
                    setAndStoreCursors((Container) enums.nextElement(), onItemCursor);
                }
            }

            // get the tracked component
            Component comp = null;
            try {
                MouseEvent event = getMouseEvent();
                if (event == null) {
                    // tracking canceled
                    return;
                }
                comp = (Component) event.getSource();
                if (log.isDebugEnabled()) {
                    log.debug("component traking " + comp.getName() + " : " + comp.getClass().getName());
                }
                comp = SwingUtil.getDeepestObjectAt(comp, event.getX(), event.getY());
                if (log.isDebugEnabled()) {
                    log.debug("deepest component " + comp.getName() + " : " + comp.getClass().getName());
                }
            } finally {
                // restore the old cursors
                if (topComponents != null) {
                    Enumeration<?> containers = topComponents.elements();
                    while (containers.hasMoreElements()) {
                        resetAndRestoreCursors((Container) containers.nextElement());
                    }
                }
                cursors = null;
            }

            String helpID = findHelpId(comp);
            showHelp(context, helpID);
        }

        public String findHelpId(Component comp) {
            String helpID = CSH.getHelpIDString(comp);
            if (defaultID.equals(helpID)) {
                String id = cache.get(comp);
                // on verifie qu'on est bien sur sur le bon id
                if (helpID.equals(id)) {
                    // ok
                    return helpID;
                }
                if (log.isDebugEnabled()) {
                    log.debug("will try to find better id for comp : " + comp.getName());
                }
                // on est pas sur le bon id
                // on recherche parmis les parents
                helpID = findExtactHelpId(comp);
            }
            if (log.isInfoEnabled()) {
                log.info("helpID " + helpID + " for comp " + comp.getName() + " : " + comp.getClass().getName());
            }
            return helpID;
        }

        protected String findExtactHelpId(Component comp) {
            Container parent = comp.getParent();
            while (parent != null) {
                String id = cache.get(parent);
                if (id == null) {
                    // ce container n'a pas d'id
                    // on va directement sur le parent
                    parent = parent.getParent();
                    continue;
                }
                // le parent possède un id
                // on utilise cet id
                return id;
            }
            // on a pas trouve d'id
            // on retourne l'id par defaut
            return defaultID;
        }
    }

    protected AbstractButton getShowHelpButton(JAXXObject c) {
        return (AbstractButton) c.getObjectById("showHelp");
    }

    protected ActionListener getShowHelpAction() {
        return new ShowHelpForTrackedComponentAction();
    }

    //-------------------------------------------------------------------------
    //--- Copy CSH code but with accessible modifiers and little improvments
    //-------------------------------------------------------------------------
    /*
     * Get all top level containers to change it's cursors
     */
    protected Vector<?> getTopContainers(Object source) {
        // This method is used to obtain all top level components of application
        // for which the changing of cursor to question mark is wanted.
        // Method Frame.getFrames() is used to get list of Frames and
        // Frame.getOwnedWindows() method on elements of the list
        // returns all Windows, Dialogs etc. It works correctly in application.
        // Problem is in applets. There is no way how to get reference to applets
        // from elsewhere than applet itself. So, if request for CSH (this means
        // pressing help button or select help menu item) does't come from component
        // in a Applet, cursor for applets is not changed to question mark. Only for
        // Frames, Windows and Dialogs is cursor changed properly.

        Vector<Component> containers = new Vector<Component>();
        Component topComponent = null;
        topComponent = getRoot(source);
        if (topComponent instanceof Applet) {
            try {
                Enumeration<Applet> applets = ((Applet) topComponent).getAppletContext().getApplets();
                while (applets.hasMoreElements()) {
                    containers.add(applets.nextElement());
                }
            } catch (NullPointerException npe) {
                containers.add(topComponent);
            }
        }
        Frame frames[] = Frame.getFrames();
        for (int i = 0; i < frames.length; i++) {
            Window[] windows = frames[i].getOwnedWindows();
            for (int j = 0; j < windows.length; j++) {
                containers.add(windows[j]);
            }
            if (!containers.contains(frames[i])) {
                containers.add(frames[i]);
            }
        }
        return containers;
    }

    protected Component getRoot(Object comp) {
        Object parent = comp;
        while (parent != null) {
            comp = parent;
            if (comp instanceof MenuComponent) {
                parent = ((MenuComponent) comp).getParent();
            } else if (comp instanceof Component) {
                if (comp instanceof Window) {
                    break;
                }
                if (comp instanceof Applet) {
                    break;
                }
                parent = ((Component) comp).getParent();
            } else {
                break;
            }
        }
        if (comp instanceof Component) {
            return ((Component) comp);
        }
        return null;
    }

    /*
     * Set the cursor for a component and its children.
     * Store the old cursors for future resetting
     */
    protected void setAndStoreCursors(Component comp, Cursor cursor) {
        if (comp == null) {
            return;
        }
        Cursor compCursor = comp.getCursor();
        if (compCursor != cursor) {
            cursors.put(comp, compCursor);
            log.debug("set cursor on " + comp);
            comp.setCursor(cursor);
        }
        if (comp instanceof Container) {
            Component component[] = ((Container) comp).getComponents();
            for (int i = 0; i < component.length; i++) {
                setAndStoreCursors(component[i], cursor);
            }
        }
    }

    /*
     * Actually restore the cursor for a component and its children
     */
    protected void resetAndRestoreCursors(Component comp) {
        if (comp == null) {
            return;
        }
        Cursor oldCursor = cursors.get(comp);
        if (oldCursor != null) {
            log.debug("restored cursor " + oldCursor + " on " + comp);
            comp.setCursor(oldCursor);
        }
        if (comp instanceof Container) {
            Component component[] = ((Container) comp).getComponents();
            for (int i = 0; i < component.length; i++) {
                resetAndRestoreCursors(component[i]);
            }
        }
    }

    /**
     * Context Sensitive Event Tracking
     *
     * Creates a new EventDispatchThread from which to dispatch events. This
     * method returns when stopModal is invoked.
     *
     * @return MouseEvent The mouse event occurred. Null if
     * cancelled on an undetermined object.
     */
    public static MouseEvent getMouseEvent() {
        // Should the cursor change to a quesiton mark here or
        // require the user to change the cursor externally to this method?
        // The problem is that each component can have it's own cursor.
        // For that reason it might be better to have the user change the
        // cusor rather than us.

        // To track context-sensitive events get the event queue and process
        // the events the same way EventDispatchThread does. Filter out
        // ContextSensitiveEvents SelectObject & Cancel (MouseDown & ???).
        // Note: This code only handles mouse events. Accessiblity might
        // require additional functionality or event trapping

        // If the eventQueue can't be retrieved, the thread gets interrupted,
        // or the thread isn't a instanceof EventDispatchThread then return
        // a null as we won't be able to trap events.
        try {
            if (EventQueue.isDispatchThread()) {
                EventQueue eq = null;

                // Find the eventQueue. If we can't get to it then just return
                // null since we won't be able to trap any events.

                try {
                    eq = Toolkit.getDefaultToolkit().getSystemEventQueue();
                } catch (Exception ee) {
                    log.debug(ee);
                }

                // Safe guard
                if (eq == null) {
                    return null;
                }

                int eventNumber = -1;

                // Process the events until an object has been selected or
                // the context-sensitive search has been canceled.
                while (true) {
                    // This is essentially the body of EventDispatchThread
                    // modified to trap context-senstive events and act
                    // appropriately
                    eventNumber++;
                    AWTEvent event = eq.getNextEvent();
                    Object src = event.getSource();
                    // can't call eq.dispatchEvent
                    // so I pasted it's body here

                    if (log.isDebugEnabled()) {
                        log.debug(event);
                    }

                    // Not sure if I should suppress ActiveEvents or not
                    // Modal dialogs do. For now we will not suppress the
                    // ActiveEvent events

                    if (event instanceof ActiveEvent) {
                        ((ActiveEvent) event).dispatch();
                        continue;
                    }

                    if (src instanceof Component) {
                        // Trap the context-sensitive events here
                        if (event instanceof KeyEvent) {
                            KeyEvent e = (KeyEvent) event;
                            // if this is the cancel key then exit
                            // otherwise pass all other keys up
                            if (e.getKeyCode() == KeyEvent.VK_CANCEL ||
                                    e.getKeyCode() == KeyEvent.VK_ESCAPE) {
                                e.consume();
                                return null;
                            } else {
                                e.consume();
                                // dispatchEvent(event);
                            }
                        } else if (event instanceof MouseEvent) {
                            MouseEvent e = (MouseEvent) event;
                            int eID = e.getID();

                            if ((eID == MouseEvent.MOUSE_CLICKED ||
                                    eID == MouseEvent.MOUSE_PRESSED ||
                                    eID == MouseEvent.MOUSE_RELEASED) &&
                                    SwingUtilities.isRightMouseButton(e)) {
                                // cancel tracking
                                e.consume();
                                if (log.isDebugEnabled()) {
                                    log.debug("tracking canceled!!!");
                                }
                                return null;
                            }

                            if ((eID == MouseEvent.MOUSE_CLICKED ||
                                    eID == MouseEvent.MOUSE_PRESSED ||
                                    eID == MouseEvent.MOUSE_RELEASED) &&
                                    SwingUtilities.isLeftMouseButton(e)) {
                                if (eID == MouseEvent.MOUSE_CLICKED) {
                                    if (eventNumber == 0) {
                                        dispatchEvent(event);
                                        continue;
                                    }
                                }
                                e.consume();
                                return e;
                            } else {
                                e.consume();
                            }
                        } else {
                            dispatchEvent(event);
                        }
                    } else if (src instanceof MenuComponent) {
                        if (event instanceof InputEvent) {
                            ((InputEvent) event).consume();
                        }
                    } else {
                        log.error("unable to dispatch event: " + event);
                    }
                }
            }
        } catch (InterruptedException e) {
            if (log.isDebugEnabled()) {
                log.debug(e);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Fall Through code");
        }
        return null;
    }

    private static void dispatchEvent(AWTEvent event) {
        Object src = event.getSource();
        if (event instanceof ActiveEvent) {
            // This could become the sole method of dispatching in time.
            ((ActiveEvent) event).dispatch();
        } else if (src instanceof Component) {
            ((Component) src).dispatchEvent(event);
        } else if (src instanceof MenuComponent) {
            ((MenuComponent) src).dispatchEvent(event);
        } else {
            log.error("unable to dispatch event: " + event);
        }
    }
}
