package org.nuiton.jaxx.application.swing.util;

/*
 * #%L
 * JAXX :: Application Swing
 * %%
 * Copyright (C) 2008 - 2014 Code Lutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import jaxx.runtime.swing.JAXXRuntimeException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.jaxx.application.ApplicationTechnicalException;

/**
 * Global exception handler.
 *
 * Catch all application uncaught and display it in a custom JoptionPane
 * or JXErrorPane.
 *
 * See http://stackoverflow.com/a/4448569/1165234 for details.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 2.8
 */
public class ApplicationExceptionHandler implements Thread.UncaughtExceptionHandler {

    private static final Log log =
            LogFactory.getLog(ApplicationExceptionHandler.class);

    final ApplicationErrorHelper errorHelper;

    public ApplicationExceptionHandler(ApplicationErrorHelper errorHelper) {
        this.errorHelper = errorHelper;
    }

    @Override
    public void uncaughtException(Thread t, Throwable ex) {
        handleException(t.getName(), ex);
    }

    public void handle(Throwable thrown) {
        // for EDT exceptions
        handleException(Thread.currentThread().getName(), thrown);
    }

    protected void handleException(String tname, Throwable ex) {
        if (log.isErrorEnabled()) {
            log.error("Global application exception [" + tname + "]", ex);
        }

        Throwable cause = getCause(ex);

        showErrorDialog(cause.getMessage(), cause);
    }

    protected Throwable getCause(Throwable ex) {

        Throwable cause = ex;

        if (cause instanceof ApplicationTechnicalException) {
            cause = cause.getCause();
        }

        if (cause instanceof JAXXRuntimeException) {
            cause = cause.getCause();
        }
        return cause;
    }

    public void showErrorDialog(String message, Throwable cause) {
        errorHelper.showErrorDialog(message, cause);
    }
}
