package org.nuiton.eugene.plugin.writer;

import org.nuiton.eugene.models.Model;
import org.nuiton.plugin.PluginHelper;
import org.nuiton.plugin.PluginIOContext;
import org.nuiton.util.FasterCachedResourceResolver;
import org.nuiton.util.FileUtil;
import org.nuiton.util.Resource;
import org.nuiton.util.ResourceResolver;

import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.URIResolver;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.util.List;

/**
 * Implentation pour les writer to type xmi (qui transforme du xmi via xsl).
 *
 * @author tchemit
 * @since 2.0.0
 */
public abstract class BaseXmiToModelFileWriter extends BaseModelFileWriter {
    public static final String PROP_RESOLVER = "resolver";
    public static final String PROP_EXTRACTED_PACKAGES = "extractedPackages";
    public static final String PROP_FULL_PACKAGE_PATH = "fullPackagePath";

    /**
     * Get style sheet.
     *
     * @param model the model file used to determine the stylesheet to use
     * @return the stylesheet name
     */
    protected abstract String getStyleSheet(File model);

    /**
     * Get extension.
     *
     * @return the extension
     */
    protected abstract String getExtension();

    @Override
    public String getInputProtocol() {
        return "xmi";
    }

    @Override
    public <M extends Model> String getOutputProtocol(Class<M> modelType) {
        // no next writer : write model files
        return null;
    }

    @Override
    public boolean acceptInclude(String include) {
        return include.startsWith("xmi:") || include.endsWith(".xmi") || include.endsWith(".uml");
    }

    @Override
    public String getDefaultIncludes() {
        return "**/*.xmi";
    }

    @Override
    public String getDefaultInputDirectory() {
        return "src/main/xmi";
    }

    @Override
    public String getDefaultOutputDirectory() {
        return "models";
    }

    protected TransformerFactory transformerFactory;

    protected TransformerFactory getTransformerFactory() {
        if (transformerFactory == null) {
            transformerFactory = TransformerFactory.newInstance();
        }
        return transformerFactory;
    }

    public String getFullPackagePath() {
        return getProperty(PROP_FULL_PACKAGE_PATH, String.class);
    }

    public String getExtractedPackages() {
        return getProperty(PROP_EXTRACTED_PACKAGES, String.class);
    }

    public String getResolver() {
        return getProperty(PROP_RESOLVER, String.class);
    }


    @Override
    public void generate(File outputDir, File inputDirectory, String includePattern, boolean overwrite) throws IOException {
        long t0 = System.nanoTime();
        try {
            getLog().info("Processing XSL tranformation on " + inputDirectory + " for " + includePattern);

            getLog().info(" with fullPackagePath   : " + getFullPackagePath());
            getLog().info(" with extractedPackages : " + getExtractedPackages());
//            getLog().info(" with acceptedXmiTypes  : " + getFullPackagePath());
            getLog().info(" with resolver          : " + getResolver());

            // recuperation des fichiers a traiter
            List<File> files = PluginHelper.getIncludedFiles(inputDirectory, new String[]{includePattern}, null);

            // lancement des traitements xsl sur les fichiers trouvés
            // dans le repertoire
            actionXsl(outputDir, inputDirectory, files, getClassLoader(), overwrite);
        } finally {
            getLog().info("xsl done in " + PluginHelper.convertTime(System.nanoTime() - t0));
        }

        getLog().info("Copy resources files");

        PluginIOContext ioContext = new PluginIOContext();
        ioContext.setInput(inputDirectory);
        ioContext.setOutput(outputDir);
//            String[] excludes = getSuffixPattern("**/*");
        PluginHelper.copyFiles(ioContext, null, new String[]{includePattern}, overwrite);
    }


    protected void actionXsl(File outputDir, File dir, List<File> files, ClassLoader fixedClassLoader, boolean overwrite) throws IOException {

        for (File file : files) {
            try {
                if (getLog().isDebugEnabled()) {
                    getLog().debug("treate file : " + file);
                }
                // Prepare resolver, stylesheet
                URIResolver fileResolver = getUriResolver(file, fixedClassLoader);
                String styleSheet = getStyleSheet(file);
                URL xsl = Resource.getURL(styleSheet);

                //TC-20090820 : using recursive for xmi
//                File result = new File(destDirModel, FileUtil.basename(file,
//                        acceptedSuffixes).concat(".").concat(getExtension()));

                String extension = "." + FileUtil.extension(file);
                String filename = FileUtil.basename(file, extension).concat(".").concat(getExtension());
                String relatifPath = file.getParentFile().getAbsolutePath().substring(dir.getAbsolutePath().length());
                File dstDir = outputDir;
                if (!relatifPath.isEmpty()) {
                    dstDir = new File(dstDir, relatifPath);
                    dstDir.mkdirs();
                }
                File result = new File(dstDir, filename);
                if (!overwrite && file.lastModified() < result.lastModified()) {
                    getLog().info("file up-to-date : " + result);
                    continue;
                }
                if (getLog().isDebugEnabled()) {
                    getLog().debug("generate " + result);
                }

                // Create the xsl transformer and set parameters
                Transformer transformer = getTransformerFactory().newTransformer(new StreamSource(xsl.openStream()));

                transformer.setParameter(PROP_FULL_PACKAGE_PATH, getFullPackagePath());
                //transformer.setParameter("extraPackages", getExtractedPackages());

                transformer.setURIResolver(fileResolver);
                transformer.transform(new StreamSource(file), new StreamResult(
                        new FileOutputStream(result)));

            } catch (Exception e) {
                throw new IOException(e.getMessage(), e);
            }
        }
    }


    protected URIResolver getUriResolver(File model, ClassLoader cl) {
        URIResolver result = null;

        try {
            Class<?> clazz = Class.forName(getResolver(), true, cl);

            // Try to set the base using the constructor
            try {
                // Look for a constructor with a String parameter (base)
                Constructor<?> withBaseConstructor = clazz.getConstructor(String.class);
                // Set the xmi folder as the base
                String base = model.getParentFile().getAbsolutePath();
                // Instantiate
                result = (URIResolver) withBaseConstructor.newInstance(base);
            } catch (Exception eee) {
                getLog().warn("Unable to instantiate resolver with String parameter",
                        eee);
            }

            // If resolver is still not created, create it using the default
            // constructor
            if (result == null) {
                result = (URIResolver) clazz.newInstance();
            }

            if (result instanceof ResourceResolver) {
                ((ResourceResolver) result).setVerbose(getConfiguration().isVerbose());
                ((ResourceResolver) result).setCl(cl);
                if (result instanceof FasterCachedResourceResolver) {
                    boolean offline = getConfiguration().isOffline();
                    getLog().debug("using offline mode  ? : " + offline);
                    ((FasterCachedResourceResolver) result).setOffline(offline);
                }
            }

        } catch (Exception eee) {
            getLog().warn("Unable to instantiate resolver using the default constructor", eee);
        }

        return result;
    }


}