/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors. 
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.classpool.scoped;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.WeakHashMap;

import javassist.ClassPool;

/**
 * An implementation of <code>ScopedClassPoolRepository</code>.
 * It is an singleton.
 *
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @author <a href="flavia.rainone@jboss.com">Flavia Rainone</a>
 * @version $Revision: 102187 $
 */
public abstract class ScopedClassPoolRepositoryImpl implements ScopedClassPoolRepository {

   /** Whether to prune */
   private boolean prune = true;

   /** Whether to prune when added to the classpool's cache */
   boolean pruneWhenCached;

   /** The registered classloaders */
   protected Map<ClassLoader, ScopedClassPool> registeredCLs = Collections
   .synchronizedMap(new WeakHashMap<ClassLoader, ScopedClassPool>());

   /** The default class pool */
   protected ClassPool classpool;

   /** The factory for creating class pools */
   protected ScopedClassPoolFactory factory = null;

   /**
    * Constructor
    * @param systemClassPool the systemClassPool
    */
   protected ScopedClassPoolRepositoryImpl(ClassPool systemClassPool) {
      classpool = systemClassPool;
   }

   /**
    * Returns the value of the prune attribute.
    * 
    * @return the prune.
    */
   public boolean isPrune() {
      return prune;
   }

   /**
    * Set the prune attribute.
    * 
    * @param prune     a new value.
    */
   public void setPrune(boolean prune) {
      this.prune = prune;
   }

   /**
    * Creates a ClassPool corresponding to {@code classLoader}.
    * 
    * @param cl  the classLoader corresponding to the created ClassPool
    * @param src the source ClassPool
    * @return    the created ClassPool
    */
   public ScopedClassPool createScopedClassPool(ClassLoader cl, ClassPool src) {
      return factory.create(cl, src, this);
   }

   /**
    * Finds the ClassPool corresponding to {@code cl}
    * 
    * @param cl the classLoader
    * @return   the ClassPoool that corresponds to {@code cl}
    */
   public ClassPool findClassPool(ClassLoader cl) {
      if (cl == null)
         return registerClassLoader(ClassLoader.getSystemClassLoader());

      return registerClassLoader(cl);
   }

   /**
    * Register a classloader.
    * 
    * @param ucl       the classloader.
    * @return          the classpool
    */
   public ClassPool registerClassLoader(ClassLoader ucl) {
      synchronized (registeredCLs) {
         // FIXME: Probably want to take this method out later
         // so that AOP framework can be independent of JMX
         // This is in here so that we can remove a UCL from the ClassPool as
         // a
         // ClassPool.classpath
         if (registeredCLs.containsKey(ucl)) {
            return registeredCLs.get(ucl);
         }
         ScopedClassPool pool = createScopedClassPool(ucl, classpool);
         registeredCLs.put(ucl, pool);
         return pool;
      }
   }

   /**
    * Get the registered classloaders
    * 
    * @return the registered classloaders
    */
   public Map<ClassLoader, ScopedClassPool> getRegisteredCLs() {
      clearUnregisteredClassLoaders();
      return registeredCLs;
   }

   /**
    * This method will check to see if a register classloader has been
    * undeployed (as in JBoss)
    */
   public void clearUnregisteredClassLoaders() {
      ArrayList<ClassLoader> toUnregister = null;
      synchronized (registeredCLs) {
         Iterator<ScopedClassPool> it = registeredCLs.values().iterator();
         while (it.hasNext()) {
            ScopedClassPool pool = it.next();
            if (pool.isUnloadedClassLoader()) {
               it.remove();
               ClassLoader cl = pool.getClassLoader();
               if (cl != null) {
                  if (toUnregister == null) {
                     toUnregister = new ArrayList<ClassLoader>();
                  }
                  toUnregister.add(cl);
               }
            }
         }
         if (toUnregister != null) {
            for (int i = 0; i < toUnregister.size(); i++) {
               unregisterClassLoader(toUnregister.get(i));
            }
         }
      }
   }

   public void unregisterClassLoader(ClassLoader cl) {
      synchronized (registeredCLs) {
         ScopedClassPool pool = registeredCLs.remove(cl);
         if (pool != null)
            pool.close();
      }
   }

   public void insertDelegate(ScopedClassPoolRepository delegate) {
      // Noop - this is the end
   }

   /**
    * Sets the classpool factory that should be used. This method should always
    * be called before the repository is used.
    */
   public void setClassPoolFactory(ScopedClassPoolFactory factory) {
      this.factory = factory;
   }

   /**
   * Returns the classpool factory.
   */
   public ScopedClassPoolFactory getClassPoolFactory() {
      return factory;
   }
}
