/*
 * #%L
 * Pollen :: VoteCounting strategy :: Instant Runoff
 * $Id: InstantRunoffVoteCountingStrategy.java 3747 2012-11-20 23:58:38Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.2/pollen-votecounting-instant-runoff/src/main/java/org/chorem/pollen/votecounting/InstantRunoffVoteCountingStrategy.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.chorem.pollen.votecounting.model.Voter;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * InstantRunoff.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public class InstantRunoffVoteCountingStrategy extends AbstractVoteCountingStrategy {

    @Override
    public VoteCountingResult votecount(Set<Voter> voters) {

        // get empty result by choice
        Map<String, ChoiceScore> scores = newEmptyChoiceScoreMap(voters);

        // calcul du score minimum pour atteindre la majorité absolue
        double totalWeight = 0.;
        for (Voter voter : voters) {
            totalWeight += voter.getWeight();
        }
        totalWeight /= 2;

        // calcul pour chaque votant de ses choix préférés
        Map<Voter, List<Set<String>>> topRankChoices =
                buildVoterSortedChoices(voters);

        Set<String> choiceIdsToExclude = Sets.newHashSet();
        Set<String> choiceIdsToKeep = Sets.newHashSet(scores.keySet());

        //FIXME Must use round of elimination to order scores
        round(topRankChoices,
              choiceIdsToExclude,
              choiceIdsToKeep,
              scores,
              totalWeight);

        // order scores (using their value) and return result
        VoteCountingResult result = orderByValues(scores.values());
        return result;
    }

    protected void round(Map<Voter, List<Set<String>>> topRankChoices,
                         Set<String> idsToExclude,
                         Set<String> idsToInclude,
                         Map<String, ChoiceScore> resultByChoice,
                         double totalWeight) {

        List<ChoiceScore> results = applyScores(topRankChoices,
                                                idsToExclude,
                                                idsToInclude,
                                                resultByChoice);

        if (!results.isEmpty()) {

            // get best score
            BigDecimal scoreValue = results.get(results.size() - 1).getScoreValue();
            double max = scoreValue == null ? 0 : scoreValue.doubleValue();

            if (max < totalWeight) {

                // pas de majorité absolue, il faut éliminer le(s) choix les plus mauvais

                guessChoiceIdsToRemove(idsToExclude, results);


                // on ne veux plus utiliser ces choix dans le tour suivant
                idsToInclude.removeAll(idsToExclude);

                // nouveau tour
                round(topRankChoices,
                      idsToExclude,
                      idsToInclude,
                      resultByChoice,
                      totalWeight);

            } else {

                // majorité absolue trouvée plus rien à faire en fait :)
            }
        }
    }

    protected List<ChoiceScore> applyScores(Map<Voter, List<Set<String>>> topRankChoices,
                                            Set<String> idsToExclude,
                                            Set<String> idsToInclude,
                                            Map<String, ChoiceScore> resultByChoice) {

        if (CollectionUtils.isNotEmpty(idsToExclude)) {

            // on supprime des classements les ids données

            for (List<Set<String>> choicesByLevel : topRankChoices.values()) {
                Iterator<Set<String>> itr = choicesByLevel.iterator();
                while (itr.hasNext()) {
                    Set<String> choiceIds = itr.next();
                    choiceIds.removeAll(idsToExclude);
                    if (choiceIds.isEmpty()) {

                        // remove this level
                        itr.remove();
                    }
                }
            }
        }

        // on remet à zero les scores pour les ids a conserver
        for (String id : idsToInclude) {
            resultByChoice.get(id).setScoreValue(null);
        }

        // on calcule les scores à partir des classements

        for (Map.Entry<Voter, List<Set<String>>> entry : topRankChoices.entrySet()) {
            List<Set<String>> idsByLevel = entry.getValue();
            if (!idsByLevel.isEmpty()) {

                Set<String> winnerIds = idsByLevel.get(0);
                Voter voter = entry.getKey();
                double voterWeight = voter.getWeight();
                for (String id : winnerIds) {
                    if (idsToInclude.contains(id)) {
                        ChoiceScore choiceScore = resultByChoice.get(id);
                        choiceScore.addScoreValue(voterWeight);
                    }
                }
            }
        }

        // recopy choices to run rounds on it and eliminates choices until
        // there is a choice > 50
        List<ChoiceScore> results = Lists.newArrayList();

        for (String id : idsToInclude) {
            results.add(resultByChoice.get(id));
        }

        Collections.sort(results);

        // on supprime tout score à 0
        Iterator<ChoiceScore> itr = results.iterator();
        while (itr.hasNext()) {
            ChoiceScore choiceScore = itr.next();
            if (choiceScore.getScoreValue() == null ||
                ZERO_D.equals(choiceScore.getScoreValue())) {
                itr.remove();
            } else {
                // score > 0 on peut s'arreter
                break;
            }
        }
        return results;
    }

    protected void guessChoiceIdsToRemove(Set<String> idsToExclude,
                                          List<ChoiceScore> results) {

        double min = results.get(0).getScoreValue().doubleValue();

        idsToExclude.clear();

        for (ChoiceScore choiceScore : results) {

            BigDecimal scoreValue = choiceScore.getScoreValue();
            if (min == (scoreValue==null?0:scoreValue.doubleValue())) {

                idsToExclude.add(choiceScore.getChoiceId());
            } else {
                // value > min, on peut s'arreter là
                break;
            }
        }
    }

}
