/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: PollenApplicationListener.java 3708 2012-09-29 12:36:25Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.3/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/PollenApplicationListener.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui;

import com.google.common.collect.Maps;
import com.opensymphony.xwork2.ActionContext;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.PollenApplicationContext;
import org.chorem.pollen.PollenConfiguration;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.PollenTopiaRootContextFactory;
import org.chorem.pollen.entities.PollenDAOHelper;
import org.chorem.pollen.services.DefaultPollenServiceContext;
import org.chorem.pollen.services.PollenNotifierWorker;
import org.chorem.pollen.services.PollenServiceContext;
import org.chorem.pollen.services.PollenServiceFactory;
import org.chorem.pollen.services.impl.UserService;
import org.chorem.pollen.votecounting.VoteCountingFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.framework.TopiaUtil;
import org.nuiton.util.FileUtil;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.Locale;
import java.util.Map;

/**
 * To listen start or end of the application.
 * <p/>
 * On start we will load the configuration and check connection to internal
 * database, creates schema and create an admin user in none found in database.
 * <p/>
 * On stop, just release the application configuration.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenApplicationListener implements ServletContextListener {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(PollenApplicationListener.class);

    @Override
    public void contextInitialized(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application starting at " + new Date() + "...");
        }

        // keep the servlet context to init shiro filters
        //FIXME tchemit 2012-04-10 change this when shiro will be properly init :(
        PollenUIUtils.setServletContext(sce.getServletContext());

        // init I18n
        DefaultI18nInitializer i18nInitializer =
                new DefaultI18nInitializer("pollen-i18n");
        i18nInitializer.setMissingKeyReturnNull(true);
        I18n.init(i18nInitializer, Locale.getDefault());

        PollenApplicationContext applicationContext =
                new PollenApplicationContext();
        PollenApplicationContext.set(sce.getServletContext(),
                                     applicationContext);

        // initialize configuration
        PollenConfiguration configuration = prepareConfiguration();

        if (log.isInfoEnabled()) {
            log.info("Base url " + configuration.getApplicationUrl());
        }
        applicationContext.setConfiguration(configuration);
        applicationContext.setVoteCountingFactory(new VoteCountingFactory());

        File temporaryDirectory = configuration.getTemporaryDirectory();
        try {
            FileUtils.deleteDirectory(temporaryDirectory);
            FileUtil.createDirectoryIfNecessary(temporaryDirectory);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Could not delete tmp dir " + temporaryDirectory, e);
            }
        }

        if (log.isInfoEnabled()) {
            log.info("Initializing RootContext...");
        }

        Map<String, Object> map = Maps.newHashMap();
        map.put(ActionContext.APPLICATION, Maps.newHashMap());
        ActionContext actionContext = new ActionContext(map);
        ActionContext.setContext(actionContext);
        PollenApplicationContext.set(actionContext, applicationContext);

        // init db
        try {

            PollenTopiaRootContextFactory f =
                    new PollenTopiaRootContextFactory();
            TopiaContext rootContext = f.newDatabaseFromConfig(configuration);
            applicationContext.setRootContext(rootContext);

            initDB(applicationContext);

        } catch (TopiaException e) {
            throw new PollenTechnicalException("Could not init db", e);
        } finally {
            ActionContext.setContext(null);
            PollenApplicationContext.set(actionContext, null);
        }

        // int pollen notifier worker
        initPollenNotifierworker(applicationContext);
    }

    @Override
    public void contextDestroyed(ServletContextEvent sce) {

        if (log.isInfoEnabled()) {
            log.info("Application is ending at " + new Date() + "...");
        }

        ServletContext servletContext = sce.getServletContext();
        PollenApplicationContext applicationContext =
                PollenApplicationContext.get(servletContext);

        if (applicationContext != null) {

            // close pollen notifier worker
            closePollenNotifierWorker(applicationContext.getPollenNotifierWorker());

            // close topia root context to pollen db
            closeRootcontext(applicationContext.getRootContext());
        }
    }

    /**
     * Prepare the configuration to be used.
     * <p/>
     * <strong>Note:</strong> This method is here mainly to be able to
     * override configuration for tests environnements.
     *
     * @return the instanciated configuration
     * @since 1.4
     */
    protected PollenConfiguration prepareConfiguration() {
        String configurationFile = System.getProperty("pollenConfigurationFile");
        PollenConfiguration configuration;
        if (StringUtils.isNotBlank(configurationFile)) {
            if (log.isInfoEnabled()) {
                log.info("Use specific configuration file : " + configurationFile);
            }
            configuration = new PollenConfiguration(configurationFile, null);
        } else {
            configuration = new PollenConfiguration();
        }
        if (log.isInfoEnabled()) {
            StringBuilder builder = new StringBuilder();
            builder.append("\n-----------------------------------------------------------------------------------------------------");
            builder.append("\nPollen configuration:");
            builder.append(configuration.printConfig());
            builder.append("\n-----------------------------------------------------------------------------------------------------");
            log.info(builder.toString());
        }
        return configuration;
    }

    private void initPollenNotifierworker(PollenApplicationContext applicationContext) {
        PollenServiceContext serviceContext = DefaultPollenServiceContext.newContext(
                Locale.getDefault(),
                null,
                applicationContext.getConfiguration(),
                new PollenServiceFactory(),
                applicationContext.getVoteCountingFactory()
        );
        PollenNotifierWorker pollenNotifierWorker = new PollenNotifierWorker(
                serviceContext,
                applicationContext.getRootContext()
        );
        applicationContext.setPollenNotifierWorker(pollenNotifierWorker);

        pollenNotifierWorker.run();
    }

    private void closePollenNotifierWorker(PollenNotifierWorker pollenNotifierWorker) {

        if (pollenNotifierWorker != null) {
            if (log.isInfoEnabled()) {
                log.info("Shuting down pollenNotifierWorker... " +
                         pollenNotifierWorker);
            }
            try {
                pollenNotifierWorker.close();
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Could not close pollenNotifierWorker", e);
                }
            }
        }
    }

    private void closeRootcontext(TopiaContext rootContext) {

        if (rootContext != null) {
            if (log.isInfoEnabled()) {
                log.info("Shuting down RootContext... " + rootContext);
            }
            if (!rootContext.isClosed()) {
                try {
                    rootContext.closeContext();
                } catch (TopiaException te) {
                    if (log.isErrorEnabled()) {
                        log.error("Could not close rootContext", te);
                    }
                }
            }
        }
    }

    protected void initDB(PollenApplicationContext applicationContext) throws TopiaException {

        ActionContext context = ActionContext.getContext();
        PollenApplicationContext.set(context, applicationContext);
        TopiaContext rootContext = applicationContext.getRootContext();

        boolean schemaFound = TopiaUtil.isSchemaExist(
                rootContext,
                PollenDAOHelper.PollenEntityEnum.UserAccount.getImplementation().getName()
        );

        if (!schemaFound) {
            rootContext.updateSchema();
        }

        PollenConfiguration configuration =
                applicationContext.getConfiguration();

        PollenServiceFactory serviceFactory =
                new PollenServiceFactory();

        TopiaContext transaction = rootContext.beginTransaction();

        try {
            PollenServiceContext serviceContext = DefaultPollenServiceContext.newContext(
                    Locale.getDefault(),
                    transaction,
                    configuration,
                    serviceFactory,
                    applicationContext.getVoteCountingFactory()
            );

            UserService service =
                    serviceFactory.newService(UserService.class, serviceContext);

            int nbUsers = service.getNbUsers();
            if (nbUsers == 0) {

                service.createDefaultUsers();

                transaction.commitTransaction();
            }

        } finally {
            closeTransaction(transaction);
        }
    }

    /**
     * Try to close the given transaction.
     *
     * @param tx the transaction to close
     * @throws TopiaException if could not close the transaction
     */
    protected void closeTransaction(TopiaContext tx) throws TopiaException {
        if (tx != null && !tx.isClosed()) {
            tx.closeContext();
        }
    }

}
