/*
 * #%L
 * Pollen :: Services
 * $Id: SecurityService.java 3528 2012-06-19 13:22:18Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4/pollen-services/src/main/java/org/chorem/pollen/services/impl/SecurityService.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services.impl;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.bean.PollUrl;
import org.chorem.pollen.business.persistence.Comment;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.Vote;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.services.PollenServiceSupport;
import org.nuiton.topia.TopiaException;

import java.util.Date;
import java.util.Set;

import static org.nuiton.i18n.I18n.n_;

/**
 * Service to manager security.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4
 */
public class SecurityService extends PollenServiceSupport {

    public boolean isPollCreator(Poll poll, String accountId,
                                 UserAccount pollenUserAccount) {

        PollAccount creator = poll.getCreator();

        boolean result = creator.getAccountId().equals(accountId);
        if (!result) {

            if (pollenUserAccount != null) {

                // try to match userAccount
                result = pollenUserAccount.equals(creator.getUserAccount());
            }
        }
        return result;
    }

    public boolean isCanClosePoll(Poll poll, AccountIdRole accountIdRole) {

        boolean result = !poll.isClosed();

        if (result) {

            // poll can be closed, check user can do action
            result = accountIdRole == AccountIdRole.CREATOR;
        }

        return result;
    }

    public void removeAccountIdWhenConnected(PollUrl url,
                                             UserAccount userAccount) {
        if (userAccount != null) {

            // remove accountId from url
            url.getPollUri().setAccountId(null);
        }
    }

    /**
     * To define meaning of a accountId.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 1.4
     */
    public enum AccountIdRole {

        /**
         * Creator of the poll.
         * <p/>
         * This role can access to everything, but can not vote.
         */
        CREATOR,
        /** A user that has voted on a poll. */
        VOTER,
        /**
         * A user that was invited to a restricted poll.
         * <p/>
         * It might have alreay voted or not.
         */
        RESTRICTED_VOTER,
        /** When accountId does not exists for a poll. */
        UNDEFINED
    }

    public static final Set<AccountIdRole> NONE_FREE_ACCOUNT_ID_ROLES = Sets.newHashSet(
            AccountIdRole.RESTRICTED_VOTER,
            AccountIdRole.CREATOR
    );

    public AccountIdRole getAccountIdRole(Poll poll, String accountId) {

        Preconditions.checkNotNull(poll);

        AccountIdRole result = AccountIdRole.UNDEFINED;

        if (StringUtils.isNotBlank(accountId)) {

            if (accountId.equals(poll.getCreator().getAccountId())) {

                result = AccountIdRole.CREATOR;
            } else {

                PollAccountDAO dao = getDAO(PollAccount.class);

                PollUri pollUri = PollUri.newPollUri(poll.getPollId(), accountId);

                if (poll.getPollType() == PollType.FREE) {

                    //

                    boolean found = isVoterAccountId(dao, pollUri);

                    if (found) {
                        result = AccountIdRole.VOTER;
                    }
                } else {

                    boolean found = isRestrictPollAccountId(dao, pollUri);

                    if (found) {
                        result = AccountIdRole.RESTRICTED_VOTER;
                    }
                }
            }
        }
        return result;
    }

    public boolean isCanAccessResult(Poll poll,
                                     String accountId,
                                     SecurityService.AccountIdRole accountIdRole,
                                     UserAccount userAccount) {

        if (isPollCreator(poll, accountId, userAccount)) {
            accountIdRole = AccountIdRole.CREATOR;
        }

        String errorMessage = isCanAccessResult(poll, accountIdRole);
        return errorMessage == null;
    }

    public String isCanAccessResult(Poll poll,
                                    SecurityService.AccountIdRole accountIdRole) {

        // check now poll results can be displayed

        boolean publicResults = poll.isPublicResults();
        boolean continuousResults = poll.isContinuousResults();

        if (!continuousResults && !poll.isClosed()) {

            // results are not continuous and poll is not closed
            return n_("pollen.security.error.poll.not.closed.and.results.not.continuous");
        }

        if (!publicResults &&
            accountIdRole != SecurityService.AccountIdRole.CREATOR) {

            // poll results are private, only poll admin can see results
            return n_("pollen.security.error.poll.result.private.and.access.not.granted");
        }

        boolean pollIsFree = PollType.FREE == poll.getPollType();

        if (publicResults &&
            !pollIsFree &&
            !SecurityService.NONE_FREE_ACCOUNT_ID_ROLES.contains(accountIdRole)) {

            // on none free poll, only creator or restricted user can have it
            return n_("pollen.security.error.poll.not.free.and.access.not.granted");
        }
        return null;
    }

    public String isCanAccessVote(Poll poll,
                                  String accountId,
                                  AccountIdRole accountIdRole) {

        if (AccountIdRole.CREATOR == accountIdRole) {

            // poll admin can alwyas access vote page
            return null;
        }

        boolean pollIsFree = PollType.FREE == poll.getPollType();

        if (pollIsFree && poll.getCreator().getAccountId().equals(accountId)) {

            // on free poll, only the creator (using his creatorId as accountId) can not vote
            return n_("pollen.security.error.poll.free.creatorId.can.not.vote");
        }
        if (!pollIsFree && AccountIdRole.RESTRICTED_VOTER != accountIdRole) {

            // on none free poll, only restricted user can vote
            return n_("pollen.security.error.poll.not.free.and.access.not.granted");
        }
        return null;
    }

    public boolean isCanVote(Poll poll,
                             String accountId,
                             AccountIdRole accountIdRole) {

        Date now = serviceContext.getCurrentTime();

        if (!poll.isRunning(now)) {

            // poll is not running, can not vote
            return false;
        }

        boolean pollIsFree = PollType.FREE == poll.getPollType();

        if (pollIsFree && poll.getCreator().getAccountId().equals(accountId)) {

            // on free poll, only the creator (using his creatorId as accountId) can not vote
            return false;
        }
        if (!pollIsFree && AccountIdRole.RESTRICTED_VOTER != accountIdRole) {

            // on none free poll, only restricted user can vote
            return false;
        }

        // ok can vote
        return true;
    }

    public boolean isCanModifyVote(Poll poll, String voteId,
                                   String accountId,
                                   UserAccount userConnected) {

        Date now = serviceContext.getCurrentTime();

        if (!poll.isRunning(now)) {

            // poll is not running can not modify anything
            return false;
        }

        if (poll.isAnonymous()) {

            // poll is anonymous, no vote can be modify
            return false;
        }

        Vote vote = poll.getVoteByTopiaId(voteId);

        if (vote == null) {

            // vote not found, can not modify it
            return false;
        }

        PollAccount votePollAccount = vote.getPollAccount();

        if (votePollAccount.getAccountId().equals(accountId)) {

            // accountId is voteAccountId, can modifiy the vote
            return true;
        }

        if (userConnected != null) {

            if (userConnected.equals(votePollAccount.getUserAccount())) {

                // user conntected is the voter
                return true;
            }
        }

        // can not modify vote in other cases
        return false;
    }

    public boolean isCanDeleteVote(Poll poll,
                                   String voteId,
                                   String accountId,
                                   AccountIdRole accountIdRole,
                                   UserAccount userConnected) {

        Date now = serviceContext.getCurrentTime();

        if (!poll.isRunning(now)) {

            // poll is not running can not remove anything
            return false;
        }

        if (poll.isAnonymous()) {

            // poll is anonymous, no vote can be delete (?) FIXME Check this
            return false;
        }


        Vote vote = poll.getVoteByTopiaId(voteId);

        if (vote == null) {

            // vote not found, can not delete it
            return false;
        }

        if (accountIdRole == AccountIdRole.CREATOR) {

            // poll admin can delete any vote
            return true;
        }

        PollAccount votePollAccount = vote.getPollAccount();

        if (votePollAccount.getAccountId().equals(accountId)) {

            // owner of vote (linked by accountId) can delete his own vote
            return true;
        }

        if (userConnected != null) {

            if (userConnected.equals(votePollAccount.getUserAccount())) {

                // owner of vote (linked by userAccount) can delete his own vote
                return true;
            }
        }

        // can not modify vote in other cases
        return false;
    }

    public boolean isCanDeleteComment(Comment comment,
                                      String accountId,
                                      AccountIdRole accountIdRole,
                                      UserAccount userAccount) {

        if (accountIdRole == AccountIdRole.CREATOR) {

            // poll admin can always delete comments
            return true;
        }

        PollAccount commentAccount = comment.getPollAccount();

        if (commentAccount.getAccountId().equals(accountId)) {

            // owner of comment (linked by accountId) can delete his comment
            return true;
        }

        if (userAccount != null) {

            if (userAccount.equals(commentAccount.getUserAccount())) {
                // owner of comment (linked by userAccount) can delete his comment
                return true;
            }
        }
        return false;
    }

    private boolean isVoterAccountId(PollAccountDAO dao, PollUri uri) {
        try {

            return dao.existsByPollVoteAccountId(uri.getPollId(), uri.getAccountId());

        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not check pollAccount existence from poll '" +
                    uri.getPollId() + "' and account '" + uri.getAccountId() + "'", e);
        }
    }

    private boolean isRestrictPollAccountId(PollAccountDAO dao, PollUri uri) {
        try {

            PollAccount result =
                    dao.getRestrictedPollAccount(uri.getPollId(), uri.getAccountId());

            return result != null;

        } catch (TopiaException e) {
            throw new PollenTechnicalException(
                    "Could not check pollAccount existence from poll '" +
                    uri.getPollId() + "' and account '" + uri.getAccountId() + "'", e);
        }
    }

//    /**
//     * Vote is allowed if {@code poll} is running and {@code pollAccount} is
//     * defined in the {@code poll} restricted list if it's not a {@link PollType#FREE}
//     * poll. The account must be defined previously using
//     * {@link PollService#getPollAccountEditable(String, UserAccount, Poll)} to
//     * have a proper link between userAccount and pollAccount even if not already
//     * created in dabase. The poll creator can't vote. The token is just use
//     * for moderate purpose.
//     *
//     * @param poll            Poll
//     * @param accountEditable Account to check
//     * @return true if vote is allowed, false otherwise
//     */
//    public boolean isVoteAllowed(Poll poll, PollAccount accountEditable) {
//
//        Preconditions.checkNotNull(poll);
//        Preconditions.checkNotNull(accountEditable);
//
//        Date now = serviceContext.getCurrentTime();
//
//        boolean result;
//        if (poll.getCreator().equals(accountEditable)) {
//
//            // The creator user can't vote
//            result = false;
//        } else {
//
//            // The poll must be running and account allowed for restricted poll
//            result = poll.isRunning(now);
//            if (poll.getPollType() != PollType.FREE) {
//
//                PersonToListDAO dao = getDAO(PersonToList.class);
//                PersonToList personToList =
//                        dao.findByPollAndAccount(poll, accountEditable);
//
//                result &= personToList != null;
//            }
//        }
//        return result;
//    }

//    public boolean isUpdateAllowed(Poll poll, String voteId,
//                                   String accountId, UserAccount userConnected) {
//
//        Date now = serviceContext.getCurrentTime();
//
//        boolean result = false;
//
//        Vote vote = poll.getVoteByTopiaId(voteId);
//
//        // can only modify a vote if poll is running.
//        if (vote != null && poll.isRunning(now)) {
//            PollAccount votePollAccount = vote.getPollAccount();
//
//            // si le votant du vote correspond au votant actuel (pollAccountId)
//            if (accountId != null
//                && accountId.equals(votePollAccount.getAccountId())) {
//                result = true;
//            }
//
//            // si l'utilisateur du vote correspond à l'utilisateur actuel (user)
//            if (userConnected != null) {
//                UserAccount voteUserAccount = votePollAccount.getUserAccount();
//                result = userConnected.equals(voteUserAccount);
//            }
//        }
//        return result;
//    }

//    public boolean isCanDeleteComment(Comment comment,
//                                      PollAccount pollAccount,
//                                      UserAccount userAccount,
//                                      boolean isPollCreator) {
//
//        boolean result = false;
//
//        PollAccount commentAccount = comment.getPollAccount();
//
//        if (isPollCreator) {
//
//            // poll creator has admin rights on his poll
//            result = true;
//        } else if (userAccount != null) {
//
//            // loggued
//            boolean isAdmin = userAccount.isAdministrator();
////            boolean isCommentAccount = userAccount.equals(commentAccount.getUserAccount());
//
//            // pollen admin or comment owner (linked by user account)
//            result = isAdmin ||
//                     userAccount.equals(commentAccount.getUserAccount());
//
//        } else if (pollAccount != null) {
//
//            // comment owner (linked by poll account)
//            result = pollAccount.equals(commentAccount);
//        }
//        return result;
//    }

}
