/*
 * #%L
 * Pollen :: Persistence
 * $Id: PollImpl.java 3595 2012-08-12 11:45:39Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.4.5/pollen-persistence/src/main/java/org/chorem/pollen/business/persistence/PollImpl.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.business.persistence;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.chorem.pollen.common.PollType;

import java.util.Date;
import java.util.List;

public class PollImpl extends PollAbstract {

    private static final long serialVersionUID = 1L;

    @Override
    public boolean isPollFree() {
        return PollType.FREE == getPollType();
    }

    @Override
    public boolean isAnonymous() {
        return PollVoteVisibility.NOBODY == getPollVoteVisibility();
    }

    @Override
    public List<Vote> getVote() {
        if (vote == null) {
            vote = Lists.newArrayList();
        }
        return vote;
    }

    @Override
    public List<Choice> getChoice() {
        if (choice == null) {
            choice = Lists.newArrayList();
        }
        return choice;
    }

    @Override
    public List<Result> getResult() {
        if (result == null) {
            result = Lists.newArrayList();
        }
        return result;
    }

    @Override
    public List<Comment> getComment() {
        if (comment == null) {
            comment = Lists.newArrayList();
        }
        return comment;
    }

    @Override
    public List<PreventRule> getPreventRule() {
        if (preventRule == null) {
            preventRule = Lists.newArrayList();
        }
        return preventRule;
    }

    @Override
    public List<VotingList> getVotingList() {
        if (votingList == null) {
            votingList = Lists.newArrayList();
        }
        return votingList;
    }

    @Override
    public Vote getVoteByPollAccount(PollAccount account) {
        Preconditions.checkNotNull(account);
        Vote resultVote = null;
        if (!isVoteEmpty()) {
            for (Vote vote : this.getVote()) {
                if (account.equals(vote.getPollAccount())) {
                    resultVote = vote;
                    break;
                }
            }
        }
        return resultVote;
    }

    @Override
    public String getAdminId() {
        return getVoteId(getCreator());
    }

    @Override
    public String getVoteId(PollAccount account) {
        String voteId = getPollId();
        if (account != null) {
            voteId += ":" + account.getAccountId();
        }
        return voteId;
    }

    @Override
    public boolean isStarted(Date currentDate) {
        return beginDate == null || beginDate.before(currentDate);
    }

    @Override
    public boolean isRunning(Date currentDate) {
        return !isClosed() &&
               isStarted(currentDate) &&
               !isFinished(currentDate);
    }

    @Override
    public boolean isFinished(Date currentDate) {
        return endDate != null && currentDate.after(endDate);
    }

    @Override
    public boolean isAddChoiceStarted(Date currentDate) {
        return isChoiceAddAllowed() &&
               (beginChoiceDate == null || beginChoiceDate.before(currentDate));
    }

    @Override
    public boolean isAddChoiceRunning(Date currentDate) {
        return isChoiceAddAllowed() &&
               !isClosed() &&
               !isFinished(currentDate) &&
               isAddChoiceStarted(currentDate) &&
               !isAddChoiceFinished(currentDate);
    }

    @Override
    public boolean isAddChoiceFinished(Date currentDate) {
        return !isChoiceAddAllowed() ||
               (endChoiceDate != null && endChoiceDate.before(currentDate));
    }

    @Override
    public PreventRule getPreventRuleByScope(String scope) {
        Preconditions.checkNotNull(scope);

        PreventRule rule = null;
        for (PreventRule value : getPreventRule()) {

            if (scope.equals(value.getScope())) {
                rule = value;
                break;
            }
        }
        return rule;
    }

    @Override
    public PersonToList getPersonToListByVote(Vote vote) {
        Preconditions.checkNotNull(vote);

        PollAccount pollAccount = vote.getPollAccount();

        PersonToList personToList = null;
        for (VotingList votingList : getVotingList()) {

            personToList = votingList.getPollAccountPersonToList(pollAccount);

            if (personToList != null) {
                break;
            }
        }
        return personToList;
    }

    @Override
    public List<PollAccount> getPollAccounts(boolean withNoVote) {

        List<PollAccount> pollAccounts = Lists.newLinkedList();

        for (VotingList votingList : getVotingList()) {

            for (PersonToList personToList :
                    votingList.getPollAccountPersonToList()) {

                // ignore those with existing vote
                if (withNoVote && personToList.isHasVoted()) {
                    continue;
                }

                pollAccounts.add(personToList.getPollAccount());
//
//                if (!withNoVote || !personToList.isHasVoted()) {
//
//                    result.add(personToList.getPollAccount());
//                }
            }
        }
        return pollAccounts;
    }

} //PollImpl
