package fr.ifremer.tutti.ui.swing.launcher;

/*
 * #%L
 * Tutti :: Launcher
 * %%
 * Copyright (C) 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import javax.swing.JOptionPane;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Launcher application that do post-update logic, launch application and listen to exit code.
 *
 * @author Tony Chemit (chemit@codelutin.com)
 * @since 1.0
 */
public class Launcher {

    public static final String APPLICATION_LAUNCHER_TITLE = "Allegro Campagne";

    public static final String APPLICATION_DIR = "tutti";

    public static final String DEBUG_APPLICATION_ARG = "--debug";

    public static final String DEBUG_UPDATER_ARG = "--debug-updater";

    /**
     * Directory where launcher is executed.
     */
    private final Path basedir;

    /**
     * Extra arguments to pass to the application (debug options are removed from this list).
     */
    private final List<String> arguments;

    /**
     * Path to the jre directory (basedir/jre).
     */
    private final Path jrePath;

    /**
     * Starts application in debug mode ?
     */
    private final boolean debugApplication;

    /**
     * Starts updater in debug mode ?
     */
    private final boolean debugUpdater;

    public static void main(String... args) {

        Launcher launcher = new Launcher(args);
        launcher.execute();

    }

    public Launcher(String... args) {

        this.basedir = Paths.get(System.getProperty("user.dir"));

        this.jrePath = basedir.resolve("jre");

        List<String> fullArguments = new ArrayList<>(Arrays.asList(args));

        this.debugApplication = fullArguments.remove(DEBUG_APPLICATION_ARG);
        this.debugUpdater = fullArguments.remove(DEBUG_UPDATER_ARG);

        this.arguments = Collections.unmodifiableList(fullArguments);

    }

    public void execute() {

        try {

            ExitCode exitCode;

            do {

                // launch updater
                exitCode = executeUpdater();

                if (ExitCodes.isRuntimeUpdateExitCode(exitCode)) {

                    // if updater has detected runtime update, stop now
                    System.exit(exitCode.getExitCode());

                }

                if (!ExitCodes.isNormalExitCode(exitCode)) {

                    // update fails
                    throw new RuntimeException("Allegro Campaign Updater has encountered a problem, program stopped.\nPlease check updater log file.");

                }

                // launch application
                exitCode = executeApplication();

            } while (exitCode.isRelaunch());

            if (!ExitCodes.isNormalExitCode(exitCode)) {

                // show exit code 
                throw new RuntimeException("exit code = " + exitCode.getExitCode());

            }

        } catch (Exception ex) {
            JOptionPane.showMessageDialog(null, ex.toString(), APPLICATION_LAUNCHER_TITLE, JOptionPane.ERROR_MESSAGE);
        }

    }

    protected ExitCode executeUpdater() throws Exception {

        // load launcher properties
        LauncherProperties launcherProperties = new LauncherProperties(basedir);
        launcherProperties.load();

        Path logFile = launcherProperties.getUpdaterLogFile();
        if (Files.notExists(logFile.getParent())) {
            Files.createDirectory(logFile.getParent());
        }

        JavaCommand program = JavaCommand
                .create(jrePath)
                .setDebug(debugUpdater)
                .setJarPath(launcherProperties.getUpdaterJarFile())
                .setLogFile(logFile);

        ExitCode exitCode = program.execute();
        return exitCode;

    }

    protected ExitCode executeApplication() throws Exception {

        // load launcher properties
        LauncherProperties launcherProperties = new LauncherProperties(basedir);
        launcherProperties.load();

        Path logFile = launcherProperties.getApplicationLogFile();
        if (Files.notExists(logFile.getParent())) {
            Files.createDirectory(logFile.getParent());
        }

        JavaCommand program = JavaCommand
                .create(jrePath)
                .setDebug(debugApplication)
                .setInheritIO(true)
                .addArguments(launcherProperties.getApplicationJvmOptions())
                .addSystemProperty("tutti.log.file", logFile.toString())
                .setJarPath(launcherProperties.getApplicationJarFile())
                .addArguments("--launch-mode", "full")
                .addArguments("--basedir", basedir.toFile().getAbsolutePath())
                .addArguments(launcherProperties.getApplicationOtherOptions())
                .addArgument(arguments);

        ExitCode exitCode = program.execute();
        return exitCode;

    }

}
