package fr.ifremer.coselmar.services.v1;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.StringWriter;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import com.auth0.jwt.Algorithm;
import com.auth0.jwt.JWTSigner;
import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.github.mustachejava.MustacheException;
import com.github.mustachejava.MustacheFactory;
import com.google.common.base.Preconditions;
import fr.ifremer.coselmar.beans.UserAccountCreatedMail;
import fr.ifremer.coselmar.beans.UserBean;
import fr.ifremer.coselmar.beans.UserSearchBean;
import fr.ifremer.coselmar.beans.UserWebToken;
import fr.ifremer.coselmar.converter.BeanEntityConverter;
import fr.ifremer.coselmar.persistence.SearchRequestBean;
import fr.ifremer.coselmar.persistence.entity.CoselmarUser;
import fr.ifremer.coselmar.persistence.entity.CoselmarUserRole;
import fr.ifremer.coselmar.services.CoselmarTechnicalException;
import fr.ifremer.coselmar.services.CoselmarWebServiceSupport;
import fr.ifremer.coselmar.services.config.CoselmarServicesConfig;
import fr.ifremer.coselmar.services.errors.InvalidCredentialException;
import fr.ifremer.coselmar.services.errors.UnauthorizedException;
import org.apache.commons.io.Charsets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.mail.Email;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.SimpleEmail;
import org.debux.webmotion.server.render.Render;
import org.nuiton.topia.persistence.TopiaNoResultException;

import static org.apache.commons.logging.LogFactory.getLog;

/**
 * @author ymartel <martel@codelutin.com>
 */
public class UsersWebService extends CoselmarWebServiceSupport {

    private static final Log log = getLog(UsersWebService.class);

    public UserBean getUser(String userId) throws InvalidCredentialException, UnauthorizedException, TopiaNoResultException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        // Who is allowed here ? Admin and user himself
        if (!StringUtils.equals(userWebToken.getRole(), CoselmarUserRole.ADMIN.name())
            && !StringUtils.equals(userWebToken.getUserId(), userId)) {
            if (log.isDebugEnabled()) {
                String message = String.format("A non admin user try to see account details with shortId '%s'", userId);
                log.debug(message);
            }
            throw new UnauthorizedException("Not allowed to see user details");
        }

        // reconstitute full id
        String fullId = CoselmarUser.class.getCanonicalName() + getPersistenceContext().getTopiaIdFactory().getSeparator() + userId;

        CoselmarUser user = getCoselmarUserDao().forTopiaIdEquals(fullId).findUnique();
        UserBean userBean = BeanEntityConverter.toBean(userId, user);
        return userBean;
    }

    public List<UserBean> getUsers(String searchKeyword, UserSearchBean search, boolean onlyActive) {

        List<CoselmarUser> userList;
        if (StringUtils.isNotBlank(searchKeyword)) {
            userList = getCoselmarUserDao().findAllLikeKeywords(Arrays.asList(searchKeyword), onlyActive);

        } else if (search != null) {
            // Search default parameter if not given
            SearchRequestBean requestBean = new SearchRequestBean();
            requestBean.setLimit(search.getLimit());
            requestBean.setPage(search.getPage());
            requestBean.setFullTextSearch(search.getFullTextSearch());

            CoselmarUser example = BeanEntityConverter.fromBean(search);

            userList = getCoselmarUserDao().findAllByExample(example, search.isOnlyActive(), requestBean);

        } else {
            if (onlyActive) {
                userList = getCoselmarUserDao().forActiveEquals(true).findAll();

            } else {
                userList = getCoselmarUserDao().findAll();
            }

        }

        List<UserBean> result = new ArrayList<>(userList.size());

        for (CoselmarUser user : userList) {
            String userLightId = getPersistenceContext().getTopiaIdFactory().getRandomPart(user.getTopiaId());
            UserBean userBean = BeanEntityConverter.toBean(userLightId, user);
            result.add(userBean);
        }

        return result;
    }

    public void addUser(UserBean user) throws InvalidParameterException {
        Preconditions.checkNotNull(user);

        CoselmarUser userEntity = getCoselmarUserDao().create();

        userEntity.setFirstname(user.getFirstName());
        userEntity.setName(user.getName());

        String mail = getCleanMail(user.getMail());
        if (StringUtils.isNotBlank(mail)) {
            checkMailUniqueness(mail, null);
            userEntity.setMail(mail);
        }

        userEntity.setRole(CoselmarUserRole.valueOf(user.getRole().toUpperCase()));
        userEntity.setQualification(user.getQualification());
        userEntity.setOrganization(user.getOrganization());
        userEntity.setActive(true);

        String password = user.getPassword();
        if (StringUtils.isBlank(password)) {
            password = getServicesContext().generatePassword();
        }

        //generate a password & a salt
        String salt = getServicesContext().generateSalt();
        String encodedPassword = getServicesContext().encodePassword(salt, password);
        userEntity.setPassword(encodedPassword);
        userEntity.setSalt(salt);

        commit();

        // send mail to user with password
        if (StringUtils.isNotBlank(mail)) {
            UserAccountCreatedMail userAccountCreatedMail = new UserAccountCreatedMail(getServicesContext().getLocale());
            userAccountCreatedMail.setUser(user);
            userAccountCreatedMail.setPassword(password);
            userAccountCreatedMail.setTo(user.getMail());

            sendMail(userAccountCreatedMail);
        }
    }

    public void modifyUser(UserBean user) throws InvalidCredentialException, UnauthorizedException, InvalidParameterException, TopiaNoResultException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        boolean isAdmin = StringUtils.equals(userWebToken.getRole(), CoselmarUserRole.ADMIN.name());

        String userId = user.getId();
        if (StringUtils.isBlank(userId)) {
            throw new InvalidParameterException("User.id is mandatory");
        }

        // Admin does not need to give password, he should not know it anyway !
        if (StringUtils.isBlank(user.getPassword()) && !isAdmin) {
            throw new InvalidParameterException("User.password is mandatory");
        }

        // Who is allowed here ? Admin and user himself only
        if (!isAdmin && !StringUtils.equals(userWebToken.getUserId(), userId)) {
            if (log.isDebugEnabled()) {
                String message = String.format("A non admin user try to modify account details with shortId '%s'", userId);
                log.debug(message);
            }
            throw new UnauthorizedException("Not allowed to modify user details");
        }

        // Ok, now, retrieve this user
        String fullId = CoselmarUser.class.getCanonicalName() +
            getPersistenceContext().getTopiaIdFactory().getSeparator() + userId;
        CoselmarUser coselmarUser = getCoselmarUserDao().forTopiaIdEquals(fullId).findAny();

        // Last check : the password
        if (!isAdmin) {
            checkPassword(coselmarUser.getPassword(), coselmarUser.getSalt(), user.getPassword());
        }

        // Ok, now let's start the user update !

        // start with mail : should be unique
        String mail = user.getMail();
        if (StringUtils.isNotBlank(mail)) {
            checkMailUniqueness(mail, fullId);
            coselmarUser.setMail(mail);
        } else {
            coselmarUser.setMail(null);
        }

        String firstName = user.getFirstName();
        if (StringUtils.isNotBlank(firstName)) {
            coselmarUser.setFirstname(firstName);
        } else {
            user.setFirstName(coselmarUser.getFirstname());
        }

        String userName = user.getName();
        if (StringUtils.isNotBlank(userName)) {
            coselmarUser.setName(userName);
        } else {
            // only in case of mail need
            user.setName(coselmarUser.getName());
        }

        String userRole = user.getRole();
        if (StringUtils.isNotBlank(userRole) && isAdmin) { // only admin can update UserRole
            coselmarUser.setRole(CoselmarUserRole.valueOf(userRole.toUpperCase()));
        }

        String organization = user.getOrganization();
        if (StringUtils.isNotBlank(organization)) {
            coselmarUser.setOrganization(organization);
        }

        String qualification = user.getQualification();
        if (StringUtils.isNotBlank(qualification)) {
            coselmarUser.setQualification(qualification);
        }

        String phoneNumber = user.getPhoneNumber();
        if (StringUtils.isNotBlank(phoneNumber)) {
            coselmarUser.setPhoneNumber(phoneNumber);
        }

        String newPassword = user.getNewPassword();
        if (StringUtils.isNotBlank(newPassword)) {
            String salt = getServicesContext().generateSalt();
            String encodedPassword = getServicesContext().encodePassword(salt, newPassword);
            coselmarUser.setSalt(salt);
            coselmarUser.setPassword(encodedPassword);

            //if it is a modification by Admin, send new mail to user
            if (isAdmin) {
                UserAccountCreatedMail userAccountCreatedMail = new UserAccountCreatedMail(getServicesContext().getLocale());
                userAccountCreatedMail.setUser(user);
                userAccountCreatedMail.setPassword(newPassword);
                userAccountCreatedMail.setTo(coselmarUser.getMail());
                sendMail(userAccountCreatedMail);
            }
        }

        boolean active = user.isActive();
        coselmarUser.setActive(active);

        commit();
    }

    public Render login(String mail, String password) throws InvalidCredentialException {
        Preconditions.checkNotNull(mail);
        Preconditions.checkNotNull(password);

        CoselmarUser user = getCoselmarUserDao().forMailEquals(getCleanMail(mail)).addEquals(CoselmarUser.PROPERTY_ACTIVE, true).findAnyOrNull();

        if (user == null) {
            throw new InvalidCredentialException("Invalid mail");
        }

        String salt = user.getSalt();
        checkPassword(user.getPassword(), salt, password);

        // return a Json Web Token for authentication
        JWTSigner jwtSigner = new JWTSigner(getCoselmarServicesConfig().getWebSecurityKey());
        JWTSigner.Options signerOption = new JWTSigner.Options();
        signerOption.setAlgorithm(Algorithm.HS512);

        String userTopiaId = user.getTopiaId();
        String shortId = getPersistenceContext().getTopiaIdFactory().getRandomPart(userTopiaId);
        Map<String, Object> claims = UserWebToken.toJwtClaims(shortId, user.getFirstname(), user.getName(), user.getRole().name());
        String webToken = jwtSigner.sign(claims, signerOption);

        return renderJSON("jwt", webToken);

    }

    public void deleteUser(String userId) throws InvalidCredentialException, UnauthorizedException {

        // Check authentication
        String authorization = getContext().getHeader("Authorization");
        UserWebToken userWebToken = checkAuthentication(authorization);

        boolean isAdmin = StringUtils.equals(userWebToken.getRole(), CoselmarUserRole.ADMIN.name());

        // Only admin is authorized to do this
        if (!isAdmin) {
            if (log.isDebugEnabled()) {
                String message = String.format("A non admin user try to delete account with shortId '%s'", userId);
                log.debug(message);
            }
            throw new UnauthorizedException("Not allowed to delete user");
        }

        // reconstitute full id
        String fullId = CoselmarUser.class.getCanonicalName() + "_" + userId;

        CoselmarUser user = getCoselmarUserDao().forTopiaIdEquals(fullId).findUnique();

        getCoselmarUserDao().delete(user);

        commit();
    }

    /////////////////////////////////////////////
    /////////////    Internal Part      /////////
    /////////////////////////////////////////////

    /**
     * Check if the mail is already used by an other user
     * @param mail      : simply mail to check
     * @param userId    : the current user#id : this parameter is needed to exclude from the search this user, cause it could already have this mail
     * @throws InvalidParameterException if the mail is already used.
     */
    protected void checkMailUniqueness(String mail, String userId) throws InvalidParameterException {
        boolean mailAlreadyUsed;

        if (StringUtils.isNotBlank(userId)) {
            mailAlreadyUsed = getCoselmarUserDao().forMailEquals(mail).addNotEquals(CoselmarUser.PROPERTY_TOPIA_ID, userId).exists();

        } else {
            mailAlreadyUsed = getCoselmarUserDao().forMailEquals(mail).exists();
        }

        if (mailAlreadyUsed) {
            String msg = String.format("mail '%s' is already used", mail);
            throw new InvalidParameterException(msg);
        }
    }

    /**
     * Check that the password is the same : encode the one given with the salt,
     * and compare with the user password from database
     *
     * @param currentPassword : password from database
     * @param salt            : salt use to encode database password
     * @param password        : given password we want to check
     * @throws InvalidCredentialException if the given password is not the same as one from database
     */
    protected void checkPassword(String currentPassword, String salt, String password) throws InvalidCredentialException {
        String encodedPassword = getServicesContext().encodePassword(salt, password);

        if (!encodedPassword.equals(currentPassword)){
            throw new InvalidCredentialException("Invalid password given");
        }
    }

    /////////////////////////////////////////////
    ///////////////   MAIL PART   ///////////////
    /////////////////////////////////////////////

    protected void sendMail(UserAccountCreatedMail mail) {

        if (getCoselmarServicesConfig().isDevMode()) {

            if (log.isInfoEnabled()) {
                log.info("an email should have been sent if not in devMode: to = " +
                    mail.getTo() + ". subject = '" + mail.getSubject() +
                    "'. body = \n" + getBody(mail));
            }

            if (!mail.isRecipientProvided() && log.isWarnEnabled()) {
                log.warn("email has no recipient, would not have been sent " + mail);
            }

        } else {

            CoselmarServicesConfig applicationConfig = getCoselmarServicesConfig();

            String body = getBody(mail);

            if (mail.isRecipientProvided()) {

                Email newEmail = new SimpleEmail();
                newEmail.setHostName(applicationConfig.getSmtpHost());
                newEmail.setSmtpPort(applicationConfig.getSmtpPort());
                newEmail.setCharset(Charsets.UTF_8.name());
                newEmail.setSubject(mail.getSubject());

                try {
                    newEmail.setFrom(applicationConfig.getSmtpFrom());
                    String to = mail.getTo();
                    newEmail.addTo(to);
                    newEmail.setMsg(body);

                    newEmail.send();

                } catch (EmailException e) {
                    throw new CoselmarTechnicalException(e);
                }

            } else {
                if (log.isErrorEnabled()) {
                    log.error("email has no recipient, won't be sent " + mail);
                }
            }
        }
    }

    protected String getBody(UserAccountCreatedMail mail) {

        Mustache mustache = getMustache(mail);
        StringWriter stringWriter = new StringWriter();
        mustache.execute(stringWriter, mail);

        return stringWriter.toString();
    }

    protected Mustache getMustache(UserAccountCreatedMail mail) {

        MustacheFactory mustacheFactory = new DefaultMustacheFactory("mail/");
        Locale locale = mail.getLocale();
        String templateName = mail.getClass().getSimpleName() + "_" + locale.getLanguage() + ".mustache";

        Mustache mustache;
        try {
            mustache = mustacheFactory.compile(templateName);

        } catch (MustacheException e) {
            // fallback with no locale
            templateName = mail.getClass().getSimpleName() + ".mustache";
            mustache = mustacheFactory.compile(templateName);
        }

        return mustache;
    }
}
