package fr.ifremer.coselmar.services.v1;

/*
 * #%L
 * Coselmar :: Rest Services
 * $Id:$
 * $HeadURL:$
 * %%
 * Copyright (C) 2014 Ifremer, Code Lutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.google.common.base.Preconditions;
import fr.ifremer.coselmar.beans.DocumentBean;
import fr.ifremer.coselmar.converter.BeanEntityConverter;
import fr.ifremer.coselmar.persistence.entity.Document;
import fr.ifremer.coselmar.persistence.entity.DocumentPrivacy;
import fr.ifremer.coselmar.services.CoselmarTechnicalException;
import fr.ifremer.coselmar.services.CoselmarWebServiceSupport;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.debux.webmotion.server.call.UploadFile;
import org.debux.webmotion.server.render.Render;
import org.nuiton.util.DateUtil;

import static org.apache.commons.logging.LogFactory.getLog;

/**
 * @author ymartel <martel@codelutin.com>
 */
public class DocumentsWebService extends CoselmarWebServiceSupport {

    private static final Log log = getLog(DocumentsWebService.class);

    public DocumentBean getDocument(String documentId) {

        // reconstitute full id
        String fullId = Document.class.getCanonicalName() + "_" + documentId;

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();
        //TODO ymartel 20141103 : manage file ?
        DocumentBean documentBean = BeanEntityConverter.toBean(document);
        return documentBean;
    }

    public List<DocumentBean> getDocuments(List<String> searchKeywords) {

        List<Document> documentList;
        if (searchKeywords != null && !searchKeywords.isEmpty()) {
            documentList = getDocumentDao().findAllContainingAllKeywords(searchKeywords);
        } else {
            documentList = getDocumentDao().findAll();
        }
        List<DocumentBean> result = new ArrayList<>(documentList.size());

        for (Document document : documentList) {
            //TODO ymartel 20141103 : manage file ?
            DocumentBean documentBean = BeanEntityConverter.toBean(document);
            result.add(documentBean);
        }

        return result;
    }

    public void addDocument(DocumentBean document, UploadFile uploadFile) {
        Preconditions.checkNotNull(document);
        Preconditions.checkNotNull(uploadFile);

        // Document File
        String fileName = uploadFile.getName();
        File uploadedFile = uploadFile.getFile();
        String contentType = uploadFile.getContentType();

        if (log.isInfoEnabled()) {
            String message = String.format("File name : %s, content-type : %s", fileName, contentType);
            log.info(message);
        }

        // put the document in the good directory
        // TODO ymartel 20141105 : with user management, put the document in a user specifique folder
        File dataDirectory = getCoselmarServicesConfig().getDataDirectory();
        String absolutePath = dataDirectory.getAbsolutePath();
        Date now = getNow();
        String formatedDay = DateUtil.formatDate(now, "yyyymmdd");
        String prefix = formatedDay + "-";
        File destFile = new File(absolutePath + "/" + prefix + fileName);
        try {
            FileUtils.moveFile(uploadedFile, destFile);
        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("error during File transfer", e);
            }
            throw new CoselmarTechnicalException("Internal error during file transfer");
        }

        // Document Metadata
        Document documentEntity = getDocumentDao().create();

        documentEntity.setName(fileName);
        documentEntity.setPrivacy(DocumentPrivacy.valueOf(document.getPrivacy().toUpperCase()));
        documentEntity.addAllKeywords(document.getKeywords());
        documentEntity.setMimeType(contentType);

        Date depositDate = document.getDepositDate();
        if (depositDate != null) {
            documentEntity.setDepositDate(new Date(depositDate.getTime()));
        } else {
            documentEntity.setDepositDate(new Date());
        }

        commit();
    }

    public Render getDocumentFile(String documentId) {

        // reconstitute full id
        String fullId = Document.class.getCanonicalName() + "_" + documentId;

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();
        //TODO ymartel 20141103 : manage file owner ?

        // Get file attached to document
        String fileName = document.getName();
        Date depositeDate = document.getDepositDate();

        File dataDirectory = getCoselmarServicesConfig().getDataDirectory();
        String absolutePath = dataDirectory.getAbsolutePath();

        String formatedDay = DateUtil.formatDate(depositeDate, "yyyymmdd");
        String prefix = formatedDay + "-";
        File documentFile = new File(absolutePath + "/" + prefix + fileName);

        String fileMimeType = document.getMimeType();
        try {
            InputStream fileStream = new FileInputStream(documentFile);
            return renderDownload(fileStream, fileName, fileMimeType);

        } catch (FileNotFoundException e) {
            if (log.isErrorEnabled()) {
                String message = String.format("Unable to retrieve file %s", fileName);
                log.error(message);
            }
            //TODO ymartel 20141106 : manage 404 error
            throw new CoselmarTechnicalException("File does not exist");
        }

    }

    public void saveDocument(DocumentBean documentBean) {
        throw new CoselmarTechnicalException("not yet implemented");
    }

    public void deleteDocument(String documentId) {

        // reconstitute full id
        String fullId = Document.class.getCanonicalName() + "_" + documentId;

        Document document = getDocumentDao().forTopiaIdEquals(fullId).findUnique();
        getDocumentDao().delete(document);

        commit();
    }

}
