/*
Copyright 2012 Opera Software ASA

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package com.opera.core.systems.mobile;

import com.opera.core.systems.EmulationProfile;

import org.json.JSONException;
import org.json.JSONObject;
import org.openqa.selenium.Dimension;

/**
 * Provides an Opera Mobile emulation profile.
 *
 * Example usage:
 *
 * <pre><code>
 *   OperaMobileEmulation.Builder profile = OperaMobileEmulation.builder();
 *   profile.setResolution(new Dimension(480, 800));
 *   profile.setPPI(123);
 *   profile.setUserAgent(UserAgent.ANDROID);
 *   profile.setIME(IME.TABLET);
 *
 *   OperaSettings settings = new OperaSettings();
 *   settings.emulate(profile.build());
 *
 *   WebDriver driver = new OperaDriver(settings);
 * </code></pre>
 */
public class OperaMobileEmulation implements EmulationProfile {

  public enum UserAgent {
    ANDROID("Android"),
    DESKTOP("Desktop"),
    MEEGO("MeeGo"),
    S60("S60");

    private final String uaIdentifier;

    UserAgent(String userAgentIdentifier) {
      uaIdentifier = userAgentIdentifier;
    }

    public String toString() {
      return uaIdentifier;
    }

    public static UserAgent find(String uaString) {
      for (UserAgent agent : values()) {
        if (agent.toString().equalsIgnoreCase(uaString)) {
          return agent;
        }
      }

      return null;
    }
  }

  // Most commonly used emulation settings for mobile devices
  private String profileName = null;
  private Dimension resolution = new Dimension(480, 800);
  private int ppi = 233;
  private IME input = IME.TOUCH;
  private UserAgent userAgent = UserAgent.ANDROID;

  public String getProfileName() {
    return profileName;
  }

  public Dimension getResolution() {
    return resolution;
  }

  public int getPPI() {
    return ppi;
  }

  public IME getIME() {
    return input;
  }

  public String getUserAgent() {
    return userAgent.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (!(o instanceof EmulationProfile)) {
      return false;
    }

    EmulationProfile other = (EmulationProfile) o;
    return this.getProfileName().equals(other.getProfileName()) &&
           this.getResolution().getHeight() == other.getResolution().getHeight() &&
           this.getResolution().getWidth() == other.getResolution().getWidth() &&
           this.getPPI() == other.getPPI() &&
           this.getIME() == other.getIME() &&
           this.getUserAgent().equals(other.getUserAgent());

  }

  @Override
  public String toString() {
    return String
        .format(
            "EmulationProfile [profileName=%s, width=%d, height=%d, ppi=%d, ime=%s, userAgent=%s]",
            getProfileName(), getResolution().getWidth(), getPPI(), getResolution().getHeight(),
            getIME().toString().toLowerCase(), getUserAgent());
  }

  public JSONObject toJson() throws JSONException {
    JSONObject json = new JSONObject();
    json.put("profileName", getProfileName());
    json.put("width", getResolution().getWidth());
    json.put("height", getResolution().getHeight());
    json.put("ppi", getPPI());
    json.put("ime", getIME().toString().toLowerCase());
    json.put("userAgent", getUserAgent());
    return json;
  }

  public static EmulationProfile fromJson(JSONObject json) throws JSONException {
    return builder()
        .setProfileName(String.valueOf(json.get("profileName")))
        .setWidth(Integer.parseInt(String.valueOf(json.get("width"))))
        .setHeight(Integer.parseInt(String.valueOf(json.get("height"))))
        .setPPI(Integer.parseInt(String.valueOf(json.get("ppi"))))
        .setIME(IME.valueOf(String.valueOf(json.get("ime")).toUpperCase()))
        .setUserAgent(UserAgent.find(String.valueOf(json.get("userAgent"))))
        .build();
  }

  /**
   * Returns a new builder.  The generated builder is equivalent to the builder created by the
   * {@link OperaMobileEmulation.Builder} constructor.
   */
  public static Builder builder() {
    return new Builder();
  }

  /**
   * A builder for creating {@link OperaMobileEmulation} emulation profiles.
   */
  public static class Builder {

    private final OperaMobileEmulation config;

    /**
     * Creates a new builder.  The returned builder is equivalent to the builder generated by {@link
     * OperaMobileEmulation#builder()}.
     */
    public Builder() {
      config = new OperaMobileEmulation();
    }

    /**
     * Sets this emulation profile's unique description.
     *
     * @param profileName a unique description of this profile
     * @return self-reference
     */
    public Builder setProfileName(String profileName) {
      config.profileName = profileName;
      return this;
    }

    /**
     * Sets the resolution of the emulation profile to the given dimensions.
     *
     * @param resolution the dimensions of the screen
     * @return self-reference
     */
    public Builder setResolution(Dimension resolution) {
      config.resolution = resolution;
      return this;
    }

    /**
     * Sets the resolution of the emulation profile to the given dimensions by <code>width</code>
     * and <code>height</code>.
     *
     * @param width  the width of the screen
     * @param height the height of the screen
     * @return self-reference
     */
    public Builder setResolution(int width, int height) {
      setResolution(new Dimension(width, height));
      return this;
    }

    /**
     * Sets the width of the screen emulation resolution.
     *
     * @param width the width of the screen
     * @return self-reference
     */
    public Builder setWidth(int width) {
      setResolution(new Dimension(width, config.getResolution().getHeight()));
      return this;
    }

    /**
     * Sets the height of the screen emulation resolution.
     *
     * @param height the height of the screen
     * @return self-reference
     */
    public Builder setHeight(int height) {
      setResolution(new Dimension(config.getResolution().getWidth(), height));
      return this;
    }

    /**
     * Sets the pixel density of the device's screen.
     *
     * @param ppi the pixel density as an integer
     * @return self-reference
     */
    public Builder setPPI(int ppi) {
      config.ppi = ppi;
      return this;
    }

    /**
     * Sets the input mechanism to use for this emulation profile.
     *
     * @param ime the IME to use
     * @return self-reference
     */
    public Builder setIME(IME ime) {
      config.input = ime;
      return this;
    }

    /**
     * Sets the user agent to use for this emulation profile.  While {@link
     * com.opera.core.systems.EmulationProfile#getUserAgent()} returns a string, {@link
     * OperaMobileEmulation.UserAgent} provides a convenience enum of the supported user agent
     * strings in Opera Mobile.
     *
     * @param userAgent the user agent to use
     * @return self-reference
     */
    public Builder setUserAgent(UserAgent userAgent) {
      config.userAgent = userAgent;
      return this;
    }

    /**
     * Returns a newly-created {@link EmulationProfile} based on the contents of the builder.
     */
    public EmulationProfile build() {
      return config;
    }

  }

}